# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pypj', 'pypj.resources.package', 'pypj.task']

package_data = \
{'': ['*'], 'pypj': ['resources/*']}

install_requires = \
['single-source>=0.2.0,<0.3.0']

entry_points = \
{'console_scripts': ['pypj = pypj.main:main']}

setup_kwargs = {
    'name': 'pypj',
    'version': '1.0.2',
    'description': 'Python project initializer',
    'long_description': '![pypj Logo](https://raw.githubusercontent.com/edge-minato/pypj/main/doc/img/logo.png)\n\n[![pypi version](https://img.shields.io/pypi/v/pypj.svg?style=flat)](https://pypi.org/pypi/pypj/)\n[![python versions](https://img.shields.io/pypi/pyversions/pypj.svg?style=flat)](https://pypi.org/pypi/pypj/)\n[![license](https://img.shields.io/pypi/l/pypj.svg?style=flat)](https://github.com/edge-minato/pypj/blob/master/LICENSE)\n[![Unittest](https://github.com/edge-minato/pypj/actions/workflows/unittest.yml/badge.svg)](https://github.com/edge-minato/pypj/actions/workflows/unittest.yml)\n[![codecov](https://codecov.io/gh/edge-minato/pypj/branch/main/graph/badge.svg?token=YDZAMKUNS0)](https://codecov.io/gh/edge-minato/pypj)\n[![Code style](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black")\n[![Downloads](https://pepy.tech/badge/pypj)](https://pepy.tech/project/pypj)\n[![Downloads](https://pepy.tech/badge/pypj/week)](https://pepy.tech/project/pypj)\n\n`pypj` provides you a modern python project template. All the basic dev package installations, their configurations, and test workflows will be done, so you can focus on coding.\n\n## What will be provided\n\nWhat the _"Modern"_ means is expressed as following directory structure. Some developers prefer another tools, and you can remove or customize the tools to be prepared. Most of all configurations regarding the code style tools, like formatter and linter, are aggregated in `pyproject.toml`.\n\n```\nmy-package/\n├── .github\n│   ├── dependabot.yml       # Dependency updater\n│\xa0\xa0 └── workflows\n│\xa0\xa0     ├── publish.yml      # Tagging on GitHub triggers publishing to Pypi\n│\xa0\xa0     └── unittest.yml     # On each push and PR, execute the unittest\n├── .pre-commit-config.yaml  # Checks format and styles of each file\n├── .venv                    # venv can be outside of project directory\n├── .vscode\n│\xa0\xa0 └── settings.json        # Format, Lint, Type check and Import sort on save\n├── Makefile                 # Useful command alias\n├── README.md                # How to start with pypj\n├── my-package               # Your package, can be "src"\n├── poetry.lock\n├── pyproject.toml           # Configured settings\n└── tests\n```\n\n## Developing tools pypj provides\n\n- Package manager: [`poetry`](https://github.com/python-poetry/poetry)\n- Formatter: [`black`](https://github.com/psf/black)\n- Linter: [`pflake8`](https://github.com/csachs/pyproject-flake8) (\\*1)\n  - Plugin: [`flake8-bugbear`](https://github.com/PyCQA/flake8-bugbear)\n- Type checker: [`mypy`](https://github.com/python/mypy)\n- Import sorter: [`isort`](https://github.com/PyCQA/isort)\n- Test framework:\n  - [`pytest`](https://github.com/pytest-dev/pytest)\n    - [`pytest-cov`](https://github.com/pytest-dev/pytest-cov)\n    - [`pytest-mock`](https://github.com/pytest-dev/pytest-mock)\n  - [`tox`](https://github.com/tox-dev/tox)\n    - [`tox-gh-actions`](https://github.com/ymyzk/tox-gh-actions)\n- Git hooks manager: [`pre-commit`](https://github.com/pre-commit/pre-commit)\n\n(\\*1) `pflake8` wraps `flake8` to aggregate settings to `pyproject.toml`\n\n## Coding format pypj provides\n\n- Max line length: `119` as default\n- Type hinting: `required`\n- And some detailed configurations\n\n## Customize\n\nHere is an actual interaction to customize.\n\n```\nPackage name: my-package\nDo you want to customize settings? (y/N): y\nMax line length (119):\nUse src directory (y/N):\nKeep venv in project (Y/n):\nUse github workflows (Y/n):\nUse vscode settings (Y/n):\nUse pre-commit (Y/n):\nUse command alias as Makefile (Y/n):\nAre you sure? (Y/n): y\n```\n\n## Other features\n\n- Single filed configurations on `pyproject.toml`\n- Single sourced versioning: [`single-source`](https://github.com/rabbit72/single-source)\n- Command alias: [`make`](https://www.gnu.org/software/make/)\n- CI/CD\n  - unittest workflow\n  - publish package workflow\n  - dependency updater configuration\n\n## Requirements\n\n- `python3`\n- `poetry` [[Installation guide](https://python-poetry.org/docs/#installation)]\n\n## Installation\n\n```sh\npip install pypj\n```\n\n## Usage\n\nSee also [README.md](pypj/resources/README.md) which will be generated with `pypj` command, it shows more actual usage.\n\n```\n$ pypj\n\n┌─┐┬ ┬┌─┐┬\n├─┘└┬┘├─┘│    python : 3.8.5\n┴   ┴ ┴ └┘    poetry : 1.1.8\n\nPackage name: my-package\nDo you want to customize settings? (y/N): N\nDo you want to proceed? (y/N): y\nTask: Initialize package: my-package\n      Command: poetry new my-package ✨\n      Poetry new done 🚀\n      Command: poetry config virtualenvs.in-project true ✨\n      Command: poetry add -D black ✨\n      Command: poetry add -D pyproject-flake8 ✨\n      Command: poetry add -D mypy ✨\n      Command: poetry add -D isort ✨\n      Command: poetry add -D pytest ✨\n      Command: poetry add -D tox ✨\n      Command: poetry add -D pytest-cov ✨\n      Command: poetry add -D pytest-mock ✨\n      Command: poetry add -D tox-gh-actions ✨\n      Configure: __init__.py  ✨\n      Create : my-package ✨\nTask: Create README.md\n      Create : README.md ✨\nTask: Configure pyproject.toml settings\n      Write  : pyproject.toml ✨\nTask: Create github actions\n      Create : unittest.yml ✨\n      Create : publish.yml ✨\n      Create : dependabot.yml ✨\nTask: Configure vscode settings\n      Create : .vscode/settings.json ✨\nTask: Create makefile\n      Create : Makefile ✨\nTask: Configure pre-commit\n      Create : .pre-commit-config.yaml ✨\n\nComplete! 🚀\nLet\'s make the world better! ✨😋🐍🌎\n```\n\n## Example configurations on `pyproject.toml`\n\nWith default setting, this kind of `pyproject.toml` file will be generated.\n\n```toml\n[tool.poetry]\nname = "my-package"\nversion = "0.1.0"\ndescription = ""\nauthors = ["you <you@example.com>"]\n\n[tool.poetry.dependencies]\npython = "^3.8"\n\n[tool.poetry.dev-dependencies]\npytest = "^5.2"\nblack = "^21.8b0"\npyproject-flake8 = "^0.0.1-alpha.2"\nmypy = "^0.910"\nisort = "^5.9.3"\ntox = "^3.24.3"\npytest-cov = "^2.12.1"\npytest-mock = "^3.6.1"\ntox-gh-actions = "^2.7.0"\n\n[build-system]\nrequires = ["poetry-core>=1.0.0"]\nbuild-backend = "poetry.core.masonry.api"\n\n[tool.black]\nline-length = 119\nexclude = \'\'\'\n(\n    migrations\n    | .mypy_cache\n    | .pytest_cache\n    | .tox\n    | .venv\n    | dist\n)\n\'\'\'\n\n[tool.flake8]\nmax-line-length = 119\nmax-complexity = 10\nselect = "C,E,F,W,B"\nignore = "E203"\n\n[tool.mypy]\n# common\npython_version = 3.8\nshow_column_numbers = true\nshow_error_context = true\nignore_missing_imports = true\ncheck_untyped_defs = true\ndisallow_untyped_defs = true\n# warning\nwarn_return_any = true\nwarn_unused_configs = true\nwarn_redundant_casts = true\n\n[tool.isort]\nprofile = "black"\nline_length = 119\n\n[tool.tox]\nlegacy_tox_ini = """\n[tox]\nenvlist = py38, flake8, black, mypy, isort\nskipsdist = true\nisolated_build = true\nskip_missing_interpreters = true\n[testenv]\nwhitelist_externals = poetry\nrequire_locked_deps = true\ninstall_dev_deps = true\ncommands =\n    poetry install -vv --no-root\n    pytest ./tests -v --cov=pypj --cov-branch --durations=0\n[testenv:flake8]\ncommands = poetry run pflake8 ./my-package\n[testenv:black]\ncommands = poetry run black ./my-package\n[testenv:mypy]\ncommands = poetry run mypy ./my-package\n[testenv:isort]\ncommands = poetry run isort ./my-package\n"""\n```\n\n## Alias as Makefile\n\n```Makefile\n.PHONY: install update clean build run debug test style\nPACKAGE := $(shell grep name pyproject.toml -m1 | awk -F" " \'{print $$3}\')\nVERSION := $(shell grep version pyproject.toml -m1 | awk -F" " \'{print $$3}\')\n\ninstall:\n        poetry install\n        poetry run pre-commit install\n\nupdate:\n        poetry update\n        poetry run pre-commit autoupdate\n\nclean:\n        rm -rf dist\n\nbuild: clean\n        poetry build\n\nrun:\n        poetry run ${PACKAGE} # Just in case the package provides a command\n\ndebug:\n        poetry run pytest ./tests -s -v --cov=pypj --cov-branch --durations=0\n\ntest:\n        poetry run tox\n\nstyle:\n        poetry run tox -e black,flake8,mypy,isort\n```\n\n## Supported python versions\n\n- Supported: `3.7`, `3.8`, `3.9`\n- Is going to be supported: `3.10`\n- Not supported: `3.6` or less\n\n**NOTE**: According to [Status of Python branches](https://devguide.python.org/#status-of-python-branches), the EoL of Python 3.6 is `2021-12-23`.\n\n\n## FAQ\n\n* Is there any restrictions regarding the package naming?\n  * -> A name of python package is defined at [PEP-008 #Package and Module Names](https://www.python.org/dev/peps/pep-0008/#package-and-module-names) and it can be expressed as regex: `/^[a-zA-Z][0-9a-zA-Z\\-_]*/`. `pypj` follows this rule.\n',
    'author': 'edge-minato',
    'author_email': 'edge.minato@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/edge-minato/pypj',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
