
## Generating Matlab bindings for iDynTree Classes

As per the documentation of [iDynTree](https://github.com/robotology/idyntree) for [modifications of Matlab/Octave bindings](https://github.com/robotology/idyntree#matlaboctave-bindings-modifications), the prerequisite is an experimental version of [Swig](http://www.swig.org/svn.html) with Matlab support.
This can be found in the repository [https://github.com/robotology-dependencies/swig/](https://github.com/robotology-dependencies/swig/) (branch `matlab`).

## Installing experimental version of Swig
- Clone the repository from [https://github.com/robotology-dependencies/swig/](https://github.com/robotology-dependencies/swig/)
- Install the required dependencies for SWIG listed in the **Required Tools** section of the [Swig page](http://www.swig.org/svn.html).
- Follow the installation instructions in [http://www.swig.org/Doc3.0/SWIGDocumentation.html#Preface_unix_installation](http://www.swig.org/Doc3.0/SWIGDocumentation.html#Preface_unix_installation) to install Swig locally. By default SWIG installs itself in `/usr/local`, in Unix environment. However, it is preferable to install Swig in a different location so as to avoid conflicts with other installed versions of SWIG.
- Proper installation of SWIG will generate a `swig` executable in `<swig_install_dir>/bin`.

This executable will be used to generate the iDynTree Matlab/Octave bindings for custom classes.

## Including the custom classes for bindings generations
In this section, as a toy example, let us consider a custom class `DiscreteKalmanFilterHelper` for which the bindings need to be generated. Let's assume the class `DiscreteKalmanFilterHelper` exists in the header file `iDynTree/Estimation/KalmanFilter.h`.

- Take a glance at `bindings/iDynTree.i`. It follows a particular structure, adding the necessary header files twice for the desired classes, one using the prefix `#` and another using the prefix `%`.
    - It is also important to notice the order in which these headers are added, which ensures that the dependent header file is compiled after the dependee header file.
    - There are also sections for adding library specific data-structure; sensors-related, estimation-related, etc.
    - Templatized classes are explicitly specialized, so that the bindings are generated for the template specializations.

- Coming back to our toy example, we will add the header file in the section related to the `Estimation related classes`, in two different places
```
#include "iDynTree/Estimation/KalmanFilter.h"
```
and
```
%include "iDynTree/Estimation/KalmanFilter.h"
```

Now, all is left is to compile and generate the bindings.

## Generating the bindings
- In the build folder of `iDynTree`, run `ccmake` to open the CMake Gui.
    - Set the CMake option `SWIG_EXECUTABLE` to `<swig_install_dir>/bin/swig`.
    - Set the CMake option `IDYNTREE_GENERATE_MATLAB` to `ON`.
    - Set the CMake option `IDYNTREE_USES_MATLAB` or `IDYNTREE_USES_OCTAVE` to `ON`.
- Having configured the CMake, run `make` and `make install` to install the bindings in the `mex` directory of the iDynTree install location.

__NOTE__: It is possible that you might face some compiler warnings or errors which needs to be handled properly for successful generation.

- Now, it is possible to use the `iDynTree.DiscreteKalmanFilterHelper` object in Matlab environment, if Matlab is configured to use the Matlab bindings as described in [iDynTree/Bindings](https://github.com/robotology/idyntree#bindings) section.

- Additionally, you will notice in the source folder that the bindings for other classes (not only related to `DiscreteKalmanFilterHelper`) have been modified in `iDynTree/bindings/matlab/autogenerated/+iDynTree/`. This is because the generation of the bindings, modifies a certain function numbering in all of these files, and the class related methods in Matlab are called depending on this specific numbering.

## Adding the generated bindings to the iDynTree Library
Usually, the generation of bindings is only allowed by the developers. So in order to create a pull request to add the custom generated bindings to iDynTree, we will follow some contribution instructions.
- Along with the modified `bindings/iDynTree.i` file, commit all the autogenerated binding files, not only the changes related specifically to your class. The latter might break the complete usage of the bindings due to the dependence on certain function numbering followed by SWIG.
- Keep only the bindings related changes in a single commit with the commit message `[bindings] update matlab bindings`, if you're generating matlab bindings. Similar message applies for python or octave bindings.
- Open a Pull request and wait for review!

## Ending note - IMPORTANT
In this tutorial, we addressed a basic feature of generating bindings for a simple class.
It must be noted that there are certain additional rules that need to be followed to generate bindings for templatized classes.
Additionally, there are certain features that the SWIG does not support, for example, bindings cannot be generated for nested classes. In general, it is useful to check out the [SWIG documentation for C++](http://www.swig.org/Doc1.3/SWIGPlus.html).

It might also be helpful to take a look into the files `iDynTree/bindings/ignore.i`, `iDynTree/bindings/sensors.i`, etc. to handle specific use cases. In particular, these are used to ignore certain methods for which the bindings need not be generated or to extend the functionality of certain classes using methods that needs to be explicitly generated as bindings only.
