# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['backplane']

package_data = \
{'': ['*']}

install_requires = \
['GitPython>=3.1.11,<4.0.0',
 'PyYAML>=5.3.1,<6.0.0',
 'anyconfig>=0.9.11,<0.10.0',
 'docker-compose>=1.27.4,<2.0.0',
 'docker>=4.3.1,<5.0.0',
 'packaging>=20.4,<21.0',
 'python-dotenv>=0.15.0,<0.16.0',
 'read-version>=0.3.1,<0.4.0',
 'typer[all]>=0.3.2,<0.4.0',
 'validators>=0.18.1,<0.19.0']

entry_points = \
{'console_scripts': ['backplane = backplane.main:app']}

setup_kwargs = {
    'name': 'backplane',
    'version': '0.7.4',
    'description': 'a dead-simple backplane for Dockerized applications',
    'long_description': '<div>\n  <img align="left" src="https://raw.githubusercontent.com/wearep3r/backplane/master/logo.png" width="175" alt="logo" />\n  <h1 align="left">backplane</h1>\n</div>\n\n**[Website](https://backplane.sh)** — **[Documentation](https://github.com/wearep3r/backplane)** — **[Source Code](https://github.com/wearep3r/backplane)**\n\nA batteries-included launchpad for your Docker Compose services - with free SSL and Git-based continuous delivery. Run any Docker app [manually](examples/) or from [backplane\'s app-store](https://github.com/backplane-apps) in no time.\n\n[!["Version"](https://img.shields.io/github/v/tag/wearep3r/backplane?label=version)](https://github.com/wearep3r/backplane)\n[!["p3r. Slack"](https://img.shields.io/badge/slack-@wearep3r/general-purple.svg?logo=slack&label=Slack)](https://join.slack.com/t/wearep3r/shared_invite/zt-d9ao21f9-pb70o46~82P~gxDTNy_JWw)\n\n---\n\n## Get started\n\n> 🚀 Check out our [Examples](examples) section for quick-start templates for [Wordpress](examples/wordpress), [Sonarqube](examples/sonarqube) and more\n\n```bash\npip install backplane\nbackplane init\nbackplane up\n```\n\nYou can now visit the dashboards of [Traefik](https://doc.traefik.io/traefik/) and [Portainer](https://www.portainer.io/) in your browser:\n\n- [traefik.127-0-0-1.ns0.co](http://traefik.127-0-0-1.ns0.co)\n- [portainer.127-0-0-1.ns0.co](http://portainer.127-0-0-1.ns0.co)\n\n## Configure your Docker Compose services\n\nExposing one of your services through **backplane** is easy:\n\n- add it to the `backplane` Docker network \n- add a label `backplane.enabled` with value `true`\n\n**backplane** will automatically pick up the service\'s name (e.g. `whoami`) and exposes it as a subdomain of your **backplane domain** (defaults to `127-0-0-1.ns0.co`).\n\n> **NOTE**: this assumes that your service is accessible on port 80 inside the container. If that is NOT the case, see [Advanced configuration](#-advanced-configuration)\n\n```yaml\nversion: "3.3"\n\nservices:\n  whoami:\n    image: "traefik/whoami"\n    container_name: "whoami"\n    networks:\n      - backplane\n    labels:\n      - "backplane.enabled=true"\n\nnetworks:\n  backplane:\n    external: true\n```\n\nYour service will be exposed as [http://whoami.127-0-0-1.ns0.co](http://whoami.127-0-0-1.ns0.co).\n\n## Use backplane with HTTPS\n\nUse `--https` and add a mail address for LetsEncrypt on installation to enable additional security for your applications. An optional `--domain` can be set on installation (defaults to `$SERVER_IP.ns0.co`, e.g. `193-43-54-23.ns0.co` if `--https` is set).\n\n```bash\nbackplane init --https --mail letsencrypt@mydomain.com [--domain mydomain.com]\nbackplane up\n```\n\nThis enables the following additional features:\n\n- access your Docker Compose services as subdomains of `mydomain.com`\n- automatic SSL for your Docker Compose services through LetsEncrypt (HTTP-Validation, so this doesn\'t work on your developer machine unless you deal with the necessary port-forwardings)\n- automatic HTTP to HTTPS redirect\n- sane security defaults\n\nThe Docker Compose stack definition doesn\'t change from the one without `--https`. **backplane** deals with the necessary configuration.\n\n```yaml\nversion: "3.3"\n\nservices:\n  whoami:\n    image: "traefik/whoami"\n    container_name: "whoami"\n    networks:\n      - backplane\n    labels:\n      - "backplane.enabled=true"\n\nnetworks:\n  backplane:\n    external: true\n```\n\nYour container will be exposed as [https://whoami.mydomain.com](https://whoami.mydomain.com).\n\n## Authentication\n\nThe default username for authentication with backplane services is `admin`, the default password is `backplane`.\n\nAssuming you don\'t want to roll with the defaults when running **backplane** on a public server, you can add `--user` and `--password` to the `init` command to specify your own values.\n\n```bash\nbackplane init --https --user testuser --password testpassword\n```\n\n### Authentication for your services\n\nTraefik comes with a [BasicAuth Middleware](https://doc.traefik.io/traefik/middlewares/basicauth/) that you can use to protect your services with the username and password configured above. All you need to do is to activate the Traefik middleware for your service:\n\n```yaml\nversion: "3.3"\n\nservices:\n  whoami:\n    image: "traefik/whoami"\n    container_name: "whoami"\n    networks:\n      - backplane\n    labels:\n      - "backplane.enabled=true"\n      - "traefik.http.routers.whoami.middlewares=auth@docker"\n\nnetworks:\n  backplane:\n    external: true\n```\n\nWhen initalized with `--https`, authentication will be activated for Traefik and Portainer automatically.\n\n## Deploy to backplane (Experimental)\n\n> **NOTE**: this is still WIP and subject to change. We try to provide an unopinonated wrapper around docker-compose with a few "augmentations" that is fully compatible with **standard** Docker Compose stacks. We also plan to integrate with Portainer\'s templating system to make installing applications even easier.\n\n**backplane** offers multiple ways to launch your applications. They all expect your application to live inside a repository (i.e. a directory). **backplane** can deploy from a plain directory or local and remote git repositories.\n\n**backplane** implements a simple workflow around `docker-compose` to "install" your applications to a Docker engine it has access to. Basically **backplane** does this:\n\n- load variables from `.env`\n- augment with global configuration (i.e. `BACKPLANE_DOMAIN=127-0-0-1.ns0.co`)\n- use `--build` if necessary (i.e. if there\'s a `build:` section in `docker-compose.yml`)\n- run `docker-compose up -d`\n\n**backplane** (as of now) does not take care of the lifecycle of the application. To interface with it, use the bundled Portainer to manage your application from a UI or fall back to standard docker/docker-compose tooling.\n\nInstalled applications will be saved to your local **backplane** config (default: `~/.backplane/contexts/default/backplane.yml`).\n\nAn application that can be installed with **backplane** should contain:\n\n- a `docker-compose.yml` file\n- an optional `.env` file configuring your stack\n- the application code\n- an optional `Dockerfile` to build the application (**backplane** expects the `build:` section of the `docker-compose.yml` file to be correctly configured)\n\nHere are a few examples:\n\n- [Grafana Loki](https://github.com/backplane-apps/loki)\n- [Docker Registry](https://github.com/backplane-apps/registry)\n- [backplane itself](https://github.com/wearep3r/backplane)\n\n### With the CLI\n\n**backplane** can deploy an application directly from its repository directory. Assuming your application provides the necessary files, just run the following command from within your application directory:\n\n```bash\nbackplane install\n```\n\nOptional arguments:\n\n- `--name`: the name of your application (translates to the `docker-compose` project, i.e. `-p NAME`); defaults to the name of the application directory (i.e. `$PWD`)\n- `--to` (or `-t`): the destination path of your application; defaults to the current directory (i.e. `$PWD`)\n- `--from` (or `-f`): a git repository, directory or URL where **backplane** can find the application; defaults to the current directory\n- `--compose-file` (or `-c`): the compose file to be used for installation (defaults to `docker-compose.yml`)\n- `app name`: if specified, **backplane** ignores the `path` argument and tries to install the application by cloning the repository from the given source to `~/.backplane/contexts/default/apps/$NAME`, where `$NAME` equals to the `NAME` argument (if given) or defaults to the name of the git repository\n\n#### Examples\n\n**local directory, custom name**:\n\n```bash\nbackplane install --name my-awesome-app-beta --from $HOME/development/my-awesome-app\n```\n\n- sets the application name to `my-awesome-app-beta`\n- installs the application from `$HOME/development/my-awesome-app`\n\n**remote git repository, default name**:\n\n```bash\nbackplane install --from https://github.com/backplane-apps/registry\n```\n\n- clones `https://github.com/backplane-apps/registry` to `~/.backplane/contexts/default/apps/registry`\n- installs the application from `~/.backplane/contexts/default/apps/registry`\n\n**local git repository, default name**:\n\n```bash\nbackplane install --from $HOME/development/my-awesome-app\n```\n\n- clones `$HOME/development/my-awesome-app` to `~/.backplane/contexts/default/apps/my-awesome-app`\n- installs the application from `~/.backplane/contexts/default/apps/my-awesome-app`\n\nThis mechanism is used by the `backplane` service running alongside Traefik and Portainer. This service enables you to `git push` to your **backplane**. Read more about this in the next paragraph.\n\n**backplane app registry, default name**:\n\nWe\'re building a central registry for backplane-comtatible applications on [GitHub](https://github.com/backplane-apps). Installing one of those is as easy as running:\n\n```bash\nbackplane install loki\n```\n\n- clones `https://github.com/backplane-apps/loki` to `~/.backplane/contexts/default/apps/loki`\n- installs the application from `~/.backplane/contexts/default/apps/loki`\n\nOur plan is to keep these apps compatible to Portainer\'s templating system to make them available as one-click installations from within the Portainer UI. One issue we currently face with this is that Portainer templates are only compatible with docker-compose configuration version "2". \n\n### With git\n\n**backplane** contains a small Git Repository service with a dead-simple CI/CD workflow for your applications. The following section explains how to push your code to a remote **backplane** where it then will be automatically built and deployed according to the workflows described in the previous sections.\n\nThis might also make sense on local development machines, but is primarily meant as a method to deploy your applications to remote **backplane** hosts in a safe way. For the following parts we assume that you have a server somewhere in the internet that you have access to via SSH (public-key authentication) and you want to use **backplane** to deploy and run your Docker Compose services on that server.\n\nLet\'s assume our remote **backplane** has the following attributes:\n\n- ip: 1.2.3.4\n- backplane-domain: 1-2-3-4.ns0.co\n- https: enabled\n\n> **ATTENTION**: the Git Repository service uses `~/.ssh` of the user that initialized **backplane** (i.e. `backplane init`) to determine the `authorized_keys` that are eligible to push to **backplane** via git. Make sure to add all relevant public keys to `~/.ssh/authorized_keys` on your **backplane** host\n\n> **TIP**: `cat ~/.ssh/id_rsa.pub | pbcopy` copies your SSH public key to your clipboard\n\n#### Update your ssh config\n\nAdd the following to your local `~/.ssh/config` file. This allows you to reach your remote **backplane** under `backplane` without further configuration.\n\n```bash\nHost backplane\n    HostName 1.2.3.4\n    User backplane\n    Port 2222\n```\n\n**Wrapup**:\n\n- our remote **backplane** (on 1.2.3.4) is now available as `backplane` when connecting with ssh\n- **backplane** runs on port 2222; ports ins git remote-urls can cause problems which is why we *mask* port and ip behind the `backplane` hostname\n\n> **NOTE**: replace the value of "HostName" with your server\'s IP or hostname. For convenience, we\'re using [ns0](https://ns0.co) here to provide wildcard DNS on IP basis\n\n#### Update your git remote\n\nAssuming your application repository is called `whoami`, this is how you add your remote **backplane** to your git remotes:\n\n```bash\ngit remote add origin "git@backplane:whoami"\n```\n\n**Wrapup**:\n\n- our previously configured remote **backplane** becomes our new git remote-url\n- we\'re connecting as user `git`\n- our repository on the remote it called `whoami`\n\n#### Deploy to your server\n\n> **HINT**: as you see, we\'re using the [Conventional Commit](https://www.conventionalcommits.org/en/v1.0.0/) format here. This will likely be a part of backplane\'s future roadmap in the form of automated versioning based on commits. Just FYI.\n\n```bash\ngit commit -am "feat: figured out who I am"\ngit push backplane master\n```\n\nThat\'s it! **backplane** will build and deploy your application and expose it automatically as `https://whoami.1-2-3-4.ns0.co`.\n\n## What is backplane\n\n**backplane** consists of 3 main services running as Docker containers on your host:\n\n- [Traefik](https://doc.traefik.io/traefik/), a very popular, cloud-native reverse-proxy\n- [Portainer](https://www.portainer.io/), a very popular management interface for Docker\n- [backplane](https://github.com/wearep3r/backplane), this software\n\nIt aims to provide simple access to core prerequisites of modern app development:\n\n- Endpoint exposure\n- Container management\n- Deployment workflows\n\nTo develop and run modern web-based applications you need a few core ingredients, like a reverse-proxy handling request routing, a way to manage containers and a way to deploy your code. **backplane** offers this for local development as well as on production nodes in a seemless way.\n\n**backplane** makes it easy to bypass long CI pipelines and deploy your application to a remote backplane host with ease.\n\n**backplane** is mainly aimed at small to medium sized development teams or solo-developers that don\'t require complex infrastructure. Use it for rapid prototyping or simple deployment scenarios where the full weight of modern CI/CD and PaaS offerings just isn\'t bearable.\n\nYou can migrate from local development to production with a simple `git push` when using **backplane** on both ends. Think of it as a micro-PaaS that you can use locally.\n\n## What backplane is NOT\n\n- a PaaS solution; backplane only provides a well-configured reverse-proxy and a management interface for containers\n- meant for production use. You can, though, but at your own risk\n\n## Advanced configuration\n\n**backplane** is only a thin wrapper around Traefik and Portainer. If you require more complex routing scenarios or have more complex service setups (e.g. multiple domains per container), simply use Traefik\'s label-based configuration.\n\n[Read more](https://doc.traefik.io/traefik/) in the docs.\n\n### Expose containers with non-standard ports\n\n**backplane** expects your services to listen to port 80 inside their containers. If that is not the case, you need to tell the backplane about it. Add the following additional labels to tell backplane your service is accessible on port 9000:\n\n```yaml\nlabels:\n  - backplane.enabled=true\n  - "traefik.http.services.custom-http.loadbalancer.server.port=9000"\n```\n\n## Examples\n\nIn the [examples](examples) directory you\'ll find examples showing how to integrate backplane with your existing services\n\nChange to any of the example folders and run `backplane install`. The example\'s `README` will hold additional information on how to use it.\n\n## Development\n\n### Dependencies\n\n```bash\npip install poetry\npoetry shell\npoetry install\n```\n\n### Build\n\n```bash\npoetry build\n```\n\n#### Build Docker\n\n```bash\ndocker build --build-arg BUILD_DATE=$(date -u +\'%Y-%m-%dT%H:%M:%SZ\') --build-arg BUILD_VERSION=$(backplane --version) --build-arg VCS_REF=$(git rev-parse --short HEAD) -t wearep3r/backplane .\ndocker tag wearep3r/backplane wearep3r/backplane:$(backplane --version)\ndocker push wearep3r/backplane:$(backplane --version)\ndocker tag wearep3r/backplane wearep3r/backplane:latest\ndocker push wearep3r/backplane:latest\n```\n\n### Generate release\n\n```bash\nsemantic-release version\n```\n\n### Publish release\n\n```bash\nsemantic-release publish\n```\n\n## Author\n\nFabian Peter, [p3r.](https://www.p3r.one/)',
    'author': 'Fabian Peter',
    'author_email': 'fabian@p3r.link',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
