"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RefactorSpaces = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const shared_1 = require("../shared");
/**
 * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class RefactorSpaces extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'refactor-spaces';
        this.accessLevelList = {
            Write: [
                'CreateApplication',
                'CreateEnvironment',
                'CreateRoute',
                'CreateService',
                'DeleteApplication',
                'DeleteEnvironment',
                'DeleteResourcePolicy',
                'DeleteRoute',
                'DeleteService',
                'PutResourcePolicy'
            ],
            Read: [
                'GetApplication',
                'GetEnvironment',
                'GetResourcePolicy',
                'GetRoute',
                'GetService',
                'ListApplications',
                'ListEnvironmentVpcs',
                'ListEnvironments',
                'ListRoutes',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
    /**
     * Grants permission to create an application within an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permission to create a route within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        return this.to('CreateRoute');
    }
    /**
     * Grants permission to create a service within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to delete an application from an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        return this.to('DeleteRoute');
    }
    /**
     * Grants permission to delete a service from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to get more information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get more information for an environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment() {
        return this.to('GetEnvironment');
    }
    /**
     * Grants permission to get the details about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get more information about a route
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetRoute.html
     */
    toGetRoute() {
        return this.to('GetRoute');
    }
    /**
     * Grants permission to get more information about a service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetService.html
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to list all the applications in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list all the VPCs for the environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironmentVpcs.html
     */
    toListEnvironmentVpcs() {
        return this.to('ListEnvironmentVpcs');
    }
    /**
     * Grants permission to list all environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironment.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list all the routes in an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        return this.to('ListRoutes');
    }
    /**
     * Grants permission to list all the services in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to list all the tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     */
    onApplication(environmentId, applicationId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}`);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifServiceCreatedByAccount()
     */
    onService(environmentId, applicationId, serviceId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}/service/${serviceId}`);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param routeId - Identifier for the routeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifRouteCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifSourcePath()
     */
    onRoute(environmentId, applicationId, routeId, account, region, partition) {
        return this.on(`arn:${partition || RefactorSpaces.defaultPartition}:refactor-spaces:${region || '*'}:${account || '*'}:environment/${environmentId}/application/${applicationId}/route/${routeId}`);
    }
    /**
     * Filters access by restricting the action to only those accounts that created the application within an environment
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationCreatedByAccount(value, operator) {
        return this.if(`ApplicationCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the accounts that created the resource
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedByAccountIds(value, operator) {
        return this.if(`CreatedByAccountIds`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the route within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRouteCreatedByAccount(value, operator) {
        return this.if(`RouteCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the service within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceCreatedByAccount(value, operator) {
        return this.if(`ServiceCreatedByAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the path of the route
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourcePath(value, operator) {
        return this.if(`SourcePath`, value, operator || 'StringLike');
    }
}
exports.RefactorSpaces = RefactorSpaces;
_a = JSII_RTTI_SYMBOL_1;
RefactorSpaces[_a] = { fqn: "iam-floyd.RefactorSpaces", version: "0.374.0" };
//# sourceMappingURL=data:application/json;base64,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