# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fawltydeps']

package_data = \
{'': ['*']}

install_requires = \
['isort>=5.10,<6.0', 'pydantic>=1.10.4,<2.0.0']

extras_require = \
{':python_version < "3.11"': ['importlib_metadata>=5.0.0,<6.0.0',
                              'tomli>=2.0.1,<3.0.0'],
 ':python_version < "3.8"': ['typing-extensions>=4.4.0,<5.0.0']}

entry_points = \
{'console_scripts': ['fawltydeps = fawltydeps.main:main']}

setup_kwargs = {
    'name': 'fawltydeps',
    'version': '0.3.0',
    'description': 'Find undeclared 3rd-party dependencies in your Python project.',
    'long_description': "# FawltyDeps\n\nA dependency checker for Python.\n\nFind _undeclared_ and/or _unused_ 3rd-party dependencies in your Python project.\n\n## Key Concepts\n\n- **_undeclared_ dependency**: a package that's used (in particular, `import`ed) by a project and which lacks a corresponding declaration to ensure that it's available.\n  For example, you `import numpy`, but you've forgotten to include `numpy` in your `requirements.txt`.\n  Pragmatically, this means the project is prone to runtime errors.\n- **_unused_ dependency**: a package that's declared as necessary for a project but which is never used by project code.\n  For example, you have `numpy` listed in your `requirements.txt`, but you never actually `import numpy`.\n  Pragmatically, this means that project installation may consume more space than needed and will be more likely to break with future software releases; in short, these are costs paid for no benefit.\n\n## Installation\n\nThe library is distributed with PyPI, so simply:\n\n```\npip install fawltydeps\n```\n\nor any other way to install Python packages from PyPI should be enough to make it available in your environment.\n\nConsider adding `fawltydeps` to your development dependencies, to help you catch undeclared and unused dependencies in your projects.\n\n## Usage\n\nTo check the project in the current directory run:\n\n```\nfawltydeps\n```\n\nThis will find imports in all the Python code under the current directory,\nextract dependencies declared by your project, and then report\n[_undeclared_ and _unused_ dependencies](#key-concepts).\n\n### Available Actions\n\nFawltyDeps provides the following options for controlling what actions to perform. Only\none of these can be used at a time:\n\n- `--check`: Report both undeclared and unused dependencies\n- `--check-undeclared`: Report only undeclared dependencies\n- `--check-unused`: Report only unused dependencies\n- `--list-imports`: List third-party imports extracted from the project\n- `--list-deps`: List declared dependencies extracted from the project\n\nWhen none of these are specified, the default action is `--check`.\n\n### Where to find Python code\n\nThe `--code` option tells FawltyDeps where to find the Python code to parse for\n`import` statements. You can pass either of these:\n\n- a single file: Either a Python file (`*.py`) or a Jupyter Notebook (`*.ipynb`)\n- a directory: FawltyDeps will find all Python files and Jupyter notebooks under this directory.\n- `-`: Passing a single dash (`--code=-`) tells FawltyDeps to read Python code\n  from stdin.\n\nIf no `--code` option is passed, FawltyDeps will find all Python code under the\ncurrent directory, i.e. same as `--code=.`\n\n### Where to find declared dependencies\n\nThe `--deps` option tells FawltyDeps where to look for your project's declared\ndependencies. A number of file formats are supported:\n\n- `*requirements*.txt` and `*requirements*.in`\n- `pyproject.toml` (following PEP 621 or Poetry conventions)\n- `setup.py` (only limited support for simple files with a single `setup()`\n  call and no computation involved for setting the `install_requires` and\n  `extras_require` arguments)\n- `setup.cfg`\n\nThe `--deps` option accepts either a directory, in which case FawltyDeps will go\nlooking for the above files under that directory. or a file, in case you want to\nbe explicit about where to find the declared dependencies.\n\nIf no `--deps` option is passed, FawltyDeps will look for the above files under\nthe current directory, i.e. same as `--deps=.`\n\n### More help\n\nRun `fawltydeps --help` to get the full list of available options.\n\n## Documentation\n\nThis project began with an exploration and design phase, yielding this [design document](./docs/DesignDoc.md), which lays out the main objective for this project and compares various strategies considered\n\nIn the [code design](./docs/CodeDesign.md) section of documentation we lay out rules which we adopt to guide code architecture decisions and maintain code quality as the project evolves.\n\n## Development\n\n### Poetry\n\nThe project uses [Poetry](https://python-poetry.org/). Install Poetry, and then\nrun:\n\n```sh\npoetry install --with=dev\n```\n\nto create a virtualenv with all (development) dependencies installed.\n\nFrom there you can run:\n\n```sh\npoetry shell\n```\n\nto jump into a development shell with this virtualenv activated. Here you will\nhave all the dependencies declared in our [`pyproject.toml`](./pyproject.toml)\ninstalled. (Without this shell activated you will have to prefix the more\nspecific commands below with `poetry run ...`).\n\n### Nox\n\nWe use [Nox](https://nox.thea.codes/en/stable/) for test/workflow automation:\n\n```sh\nnox --list        # List sessions\nnox               # Run all available sessions\nnox -R            # Run all available sessions, while reusing virtualenvs (i.e. faster)\nnox -s tests      # Run unit tests on supported Python versions (that are available)\nnox -s tests-3.7  # Run unit tests on Python v3.7 (assuming it is available locally)\nnox -s integration_tests-3.11  # Run integration tests on Python 3.11\nnox -s lint       # Run linters (mypy + pylint) on all supported Python versions\nnox -s format     # Check formatting (isort + black)\nnox -s reformat   # Fix formatting (isort + black)\n```\n\nIf you want to run a command individually, the corresponding session is defined inside\n[`noxfile.py`](./noxfile.py). For example, these\ncommands will work:\n\n```sh\npytest                   # Run unit tests\npytest -m integration    # Run integration tests\nmypy                     # Run static type checking\npylint fawltydeps tests  # Run Pylint\nisort fawltydeps tests   # Fix sorting of import statements\nblack .                  # Fix code formatting\n```\n\n### Shortcut: Nix\n\nWe have a [`shell.nix`](./shell.nix) which provides Poetry in addition to all of\nour supported Python versions. If you have [Nix](https://nixos.org) available\non your machine, then running:\n\n```sh\nnix-shell\n```\n\nwill put you inside a shell where the Poetry virtualenv (with all development\ndependencies) is activated, and all supported Python versions are available.\nThis also provides isolation from whatever Python version(s) and packages are\ninstalled on your system.\n\nFrom there, a simple `nox` will run all tests + linters against all supported\nPython versions, as well as checking/formatting the code.\n\n## Integration tests\n\nIn addition to comprehensive unit tests under `tests/`, we also verify\nFawltyDeps' behavior with integration tests which (among other things) include\ntesting with real-world projects. To that end, we have a framework in\n[`tests/test_real_projects.py`](./tests/test_real_projects.py) for downloading\nand unpacking tarballs of 3rd-party projects, and then running fawltydeps on them,\nwhile verifying their output. These projects, along with the expected FawltyDeps\noutputs, are defined in TOML files under\n[`tests/real_projects`](./tests/real_projects).\n\n## Contributing\nFor bug reports, when a user reports that `fawltydeps` does not work on their project, we adopt the following process:\n\n- The project is added to `real_projects`.\n- We isolate the problems/issues/features and define/express them succinctly as a sample project under `sample_projects`.\n- We examine the issue more closely and update core logic, adding/altering unit tests along the way.\n\nThe resulting updates are introduced to `fawltydeps` and reflected in our expectations, first in the TOML for the sample project(s) and then finally in the `real_projects` TOML.\n\nIf you find a project where FawltyDeps is not doing a good job, we would appreciate\nif you add that project under [`tests/real_projects`](./tests/real_projects).\nTo see how these tests work, look at the existing files in that directory.\n\n\n",
    'author': 'Maria Knorps',
    'author_email': 'maria.knorps@tweag.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/tweag/FawltyDeps',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
