"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_a = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _a === void 0 ? void 0 : _a.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibHVzdHJlLWZpbGUtc3lzdGVtLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibHVzdHJlLWZpbGUtc3lzdGVtLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsOENBQTJGO0FBQzNGLHdDQUFvRDtBQUNwRCwrQ0FBaUc7QUFDakcsbURBQThDO0FBRzlDOztHQUVHO0FBQ0gsSUFBWSxvQkFjWDtBQWRELFdBQVksb0JBQW9CO0lBQzlCOztPQUVHO0lBQ0gsK0NBQXVCLENBQUE7SUFDdkI7OztPQUdHO0lBQ0gsK0NBQXVCLENBQUE7SUFDdkI7O09BRUc7SUFDSCxxREFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBZFcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUFjL0I7QUEyRUQ7Ozs7OztHQU1HO0FBQ0gsTUFBYSxnQkFBaUIsU0FBUSw0QkFBYztJQWtFbEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE0Qjs7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFCLE1BQU0sbUJBQW1CLEdBQUc7WUFDMUIscUJBQXFCLEVBQUUsS0FBSyxDQUFDLG1CQUFtQixDQUFDLHdCQUF3QjtZQUN6RSwwQkFBMEIsUUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsMEJBQTBCLDBDQUFFLFdBQVcsRUFBRTtTQUNoRyxDQUFDO1FBQ0YsTUFBTSxtQkFBbUIsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsbUJBQW1CLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUU5RixNQUFNLGFBQWEsR0FBRyxDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSx1QkFBYSxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUM5RixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUMsQ0FBQztRQUNKLGFBQWEsQ0FBQyxjQUFjLENBQzFCLGFBQWEsRUFDYixjQUFJLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQzdHLElBQUksQ0FBQyxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFeEUsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxjQUFjLEVBQUUsZ0JBQWdCLENBQUMsd0JBQXdCO1lBQ3pELFNBQVMsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDO1lBQ3JDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUN4QixRQUFRLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3pELG1CQUFtQjtZQUNuQixnQkFBZ0IsRUFBRSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUM7WUFDakQsZUFBZSxFQUFFLEtBQUssQ0FBQyxrQkFBa0I7U0FDMUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQztRQUN4QyxJQUFJLENBQUMsT0FBTyxHQUFHLEdBQUcsSUFBSSxDQUFDLFlBQVksUUFBUSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxVQUFHLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDakYsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUFDO0lBQ3ZELENBQUM7SUFoR0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsOEJBQThCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7UUFDcEcsTUFBTSxNQUFPLFNBQVEsNEJBQWM7WUFBbkM7O2dCQUNrQixZQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztnQkFDeEIsaUJBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO2dCQUNsQyxnQkFBVyxHQUFHLGdCQUFnQixDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUMzRixDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBWUQ7O09BRUc7SUFDSyxNQUFNLENBQUMsb0JBQW9CLENBQUMsYUFBNkI7UUFDL0QsTUFBTSxXQUFXLEdBQUcsSUFBSSxxQkFBVyxDQUFDO1lBQ2xDLGNBQWMsRUFBRSxDQUFDLGFBQWEsQ0FBQztZQUMvQixXQUFXLEVBQUUsY0FBSSxDQUFDLFFBQVEsQ0FDeEIsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUM3QyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQztJQStERDs7T0FFRztJQUNLLGFBQWEsQ0FBQyxLQUE0QjtRQUNoRCxNQUFNLG1CQUFtQixHQUFHLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQztRQUN0RCxNQUFNLGNBQWMsR0FBRyxtQkFBbUIsQ0FBQyxjQUFjLENBQUM7UUFFMUQsdUVBQXVFO1FBQ3ZFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLENBQUMsVUFBVSxFQUFFLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXhGLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxtQkFBbUIsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2pGLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxjQUFjLEVBQUUsbUJBQW1CLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNwRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsY0FBYyxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7T0FFRztJQUNLLGtCQUFrQixDQUFDLFVBQW1CLEVBQUUsVUFBbUI7UUFDakUsSUFBSSxVQUFVLEtBQUssU0FBUyxFQUFFO1lBQUUsT0FBTztTQUFFO1FBQ3pDLElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRSxDQUFDLENBQUM7U0FDdEY7UUFFRCxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUFFLE9BQU87U0FBRTtRQUVqRixJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEtBQUssWUFBSyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUNyRSxNQUFNLElBQUksS0FBSyxDQUFDLHVGQUF1RixDQUFDLENBQUM7U0FDMUc7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLG9CQUFvQixVQUFVLHdFQUF3RSxDQUFDLENBQUM7U0FDekg7UUFDRCxJQUFJLFVBQVUsQ0FBQyxNQUFNLEdBQUcsR0FBRyxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLFVBQVUsZ0RBQWdELENBQUMsQ0FBQztTQUNqRztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNLLDZCQUE2QixDQUFDLHFCQUE4QjtRQUNsRSxJQUFJLHFCQUFxQixLQUFLLFNBQVMsRUFBRTtZQUFFLE9BQU87U0FBRTtRQUVwRCxJQUFJLHFCQUFxQixHQUFHLENBQUMsSUFBSSxxQkFBcUIsR0FBRyxNQUFNLEVBQUU7WUFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxtQ0FBbUMscUJBQXFCLGdEQUFnRCxDQUFDLENBQUM7U0FDM0g7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxrQkFBa0IsQ0FBQyxVQUFtQjtRQUM1QyxJQUFJLFVBQVUsS0FBSyxTQUFTLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUFFLE9BQU87U0FBRTtRQUUzRSxNQUFNLE1BQU0sR0FBRyxVQUFVLENBQUM7UUFFMUIsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLFVBQVUsd0VBQXdFLENBQUMsQ0FBQztTQUN6SDtRQUNELElBQUksVUFBVSxDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsVUFBVSxnREFBZ0QsQ0FBQyxDQUFDO1NBQ2pHO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssZ0NBQWdDLENBQUMsY0FBb0MsRUFBRSx3QkFBaUM7UUFDOUcsSUFBSSx3QkFBd0IsS0FBSyxTQUFTLEVBQUU7WUFBRSxPQUFPO1NBQUU7UUFFdkQsSUFBSSxjQUFjLEtBQUssb0JBQW9CLENBQUMsWUFBWSxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsK0VBQStFLENBQUMsQ0FBQztTQUNsRztRQUVELElBQUksQ0FBQyxDQUFDLEVBQUUsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDLEVBQUU7WUFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssdUJBQXVCLENBQUMsY0FBb0MsRUFBRSxlQUF1QjtRQUMzRixJQUFJLGNBQWMsS0FBSyxvQkFBb0IsQ0FBQyxTQUFTLEVBQUU7WUFDckQsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDLElBQUksZUFBZSxHQUFHLElBQUksS0FBSyxDQUFDLEVBQUU7Z0JBQ2pGLE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQzthQUN4RjtTQUNGO2FBQU07WUFDTCxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxJQUFJLGVBQWUsR0FBRyxJQUFJLEtBQUssQ0FBQyxFQUFFO2dCQUMzRSxNQUFNLElBQUksS0FBSyxDQUFDLDhEQUE4RCxDQUFDLENBQUM7YUFDakY7U0FDRjtJQUNILENBQUM7O0FBak1ILDRDQWtNQztBQW5MQzs7R0FFRztBQUNxQix5Q0FBd0IsR0FBVyxRQUFRLENBQUM7QUFFcEU7O0dBRUc7QUFDcUIsbUNBQWtCLEdBQUcsRUFBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7Q29ubmVjdGlvbnMsIElTZWN1cml0eUdyb3VwLCBJU3VibmV0LCBQb3J0LCBTZWN1cml0eUdyb3VwfSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7QXdzLCBDb25zdHJ1Y3QsIFRva2VufSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7RmlsZVN5c3RlbUF0dHJpYnV0ZXMsIEZpbGVTeXN0ZW1CYXNlLCBGaWxlU3lzdGVtUHJvcHMsIElGaWxlU3lzdGVtfSBmcm9tICcuL2ZpbGUtc3lzdGVtJztcbmltcG9ydCB7Q2ZuRmlsZVN5c3RlbX0gZnJvbSAnLi9mc3guZ2VuZXJhdGVkJztcbmltcG9ydCB7THVzdHJlTWFpbnRlbmFuY2VUaW1lfSBmcm9tICcuL21haW50ZW5hbmNlLXRpbWUnO1xuXG4vKipcbiAqIFRoZSBkaWZmZXJlbnQga2luZHMgb2YgZmlsZSBzeXN0ZW0gZGVwbG95bWVudHMgdXNlZCBieSBMdXN0cmUuXG4gKi9cbmV4cG9ydCBlbnVtIEx1c3RyZURlcGxveW1lbnRUeXBlIHtcbiAgLyoqXG4gICAqIE9yaWdpbmFsIHR5cGUgZm9yIHNob3J0ZXIgdGVybSBkYXRhIHByb2Nlc3NpbmcuIERhdGEgaXMgbm90IHJlcGxpY2F0ZWQgYW5kIGRvZXMgbm90IHBlcnNpc3Qgb24gc2VydmVyIGZhaWwuXG4gICAqL1xuICBTQ1JBVENIXzEgPSAnU0NSQVRDSF8xJyxcbiAgLyoqXG4gICAqIE5ld2VyIHR5cGUgZm9yIHNob3J0ZXIgdGVybSBkYXRhIHByb2Nlc3NpbmcuIERhdGEgaXMgbm90IHJlcGxpY2F0ZWQgYW5kIGRvZXMgbm90IHBlcnNpc3Qgb24gc2VydmVyIGZhaWwuXG4gICAqIFByb3ZpZGVzIGJldHRlciBzdXBwb3J0IGZvciBzcGlreSB3b3JrbG9hZHMuXG4gICAqL1xuICBTQ1JBVENIXzIgPSAnU0NSQVRDSF8yJyxcbiAgLyoqXG4gICAqIExvbmcgdGVybSBzdG9yYWdlLiBEYXRhIGlzIHJlcGxpY2F0ZWQgYW5kIGZpbGUgc2VydmVycyBhcmUgcmVwbGFjZWQgaWYgdGhleSBmYWlsLlxuICAgKi9cbiAgUEVSU0lTVEVOVF8xID0gJ1BFUlNJU1RFTlRfMSdcbn1cblxuLyoqXG4gKiBUaGUgY29uZmlndXJhdGlvbiBmb3IgdGhlIEFtYXpvbiBGU3ggZm9yIEx1c3RyZSBmaWxlIHN5c3RlbS5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWZzeC1maWxlc3lzdGVtLWx1c3RyZWNvbmZpZ3VyYXRpb24uaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEx1c3RyZUNvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogVGhlIHR5cGUgb2YgYmFja2luZyBmaWxlIHN5c3RlbSBkZXBsb3ltZW50IHVzZWQgYnkgRlN4LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudFR5cGU6IEx1c3RyZURlcGxveW1lbnRUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCBpbiBBbWF6b24gUzMgd2hlcmUgdGhlIHJvb3Qgb2YgeW91ciBBbWF6b24gRlN4IGZpbGUgc3lzdGVtIGlzIGV4cG9ydGVkLiBUaGUgcGF0aCBtdXN0IHVzZSB0aGUgc2FtZVxuICAgKiBBbWF6b24gUzMgYnVja2V0IGFzIHNwZWNpZmllZCBpbiBJbXBvcnRQYXRoLiBJZiB5b3Ugb25seSBzcGVjaWZ5IGEgYnVja2V0IG5hbWUsIHN1Y2ggYXMgczM6Ly9pbXBvcnQtYnVja2V0LCB5b3VcbiAgICogZ2V0IGEgMToxIG1hcHBpbmcgb2YgZmlsZSBzeXN0ZW0gb2JqZWN0cyB0byBTMyBidWNrZXQgb2JqZWN0cy4gVGhpcyBtYXBwaW5nIG1lYW5zIHRoYXQgdGhlIGlucHV0IGRhdGEgaW4gUzMgaXNcbiAgICogb3ZlcndyaXR0ZW4gb24gZXhwb3J0LiBJZiB5b3UgcHJvdmlkZSBhIGN1c3RvbSBwcmVmaXggaW4gdGhlIGV4cG9ydCBwYXRoLCBzdWNoIGFzXG4gICAqIHMzOi8vaW1wb3J0LWJ1Y2tldC9bY3VzdG9tLW9wdGlvbmFsLXByZWZpeF0sIEFtYXpvbiBGU3ggZXhwb3J0cyB0aGUgY29udGVudHMgb2YgeW91ciBmaWxlIHN5c3RlbSB0byB0aGF0IGV4cG9ydFxuICAgKiBwcmVmaXggaW4gdGhlIEFtYXpvbiBTMyBidWNrZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IHMzOi8vaW1wb3J0LWJ1Y2tldC9GU3hMdXN0cmVbY3JlYXRpb24tdGltZXN0YW1wXVxuICAgKi9cbiAgcmVhZG9ubHkgZXhwb3J0UGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogRm9yIGZpbGVzIGltcG9ydGVkIGZyb20gYSBkYXRhIHJlcG9zaXRvcnksIHRoaXMgdmFsdWUgZGV0ZXJtaW5lcyB0aGUgc3RyaXBlIGNvdW50IGFuZCBtYXhpbXVtIGFtb3VudCBvZiBkYXRhIHBlclxuICAgKiBmaWxlIChpbiBNaUIpIHN0b3JlZCBvbiBhIHNpbmdsZSBwaHlzaWNhbCBkaXNrLiBBbGxvd2VkIHZhbHVlcyBhcmUgYmV0d2VlbiAxIGFuZCA1MTIsMDAwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxMDI0XG4gICAqL1xuICByZWFkb25seSBpbXBvcnRlZEZpbGVDaHVua1NpemVNaUI/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXRoIHRvIHRoZSBBbWF6b24gUzMgYnVja2V0IChpbmNsdWRpbmcgdGhlIG9wdGlvbmFsIHByZWZpeCkgdGhhdCB5b3UncmUgdXNpbmcgYXMgdGhlIGRhdGEgcmVwb3NpdG9yeSBmb3JcbiAgICogeW91ciBBbWF6b24gRlN4IGZvciBMdXN0cmUgZmlsZSBzeXN0ZW0uIE11c3QgYmUgb2YgdGhlIGZvcm1hdCBcInMzOi8ve2J1Y2tldE5hbWV9L29wdGlvbmFsLXByZWZpeFwiIGFuZCBjYW5ub3RcbiAgICogZXhjZWVkIDkwMCBjaGFyYWN0ZXJzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGJ1Y2tldCBpcyBpbXBvcnRlZFxuICAgKi9cbiAgcmVhZG9ubHkgaW1wb3J0UGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogUmVxdWlyZWQgZm9yIHRoZSBQRVJTSVNURU5UXzEgZGVwbG95bWVudCB0eXBlLCBkZXNjcmliZXMgdGhlIGFtb3VudCBvZiByZWFkIGFuZCB3cml0ZSB0aHJvdWdocHV0IGZvciBlYWNoIDFcbiAgICogdGViaWJ5dGUgb2Ygc3RvcmFnZSwgaW4gTUIvcy9UaUIuIFZhbGlkIHZhbHVlcyBhcmUgNTAsIDEwMCwgMjAwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRlZmF1bHQsIGNvbmRpdGlvbmFsbHkgcmVxdWlyZWQgZm9yIFBFUlNJU1RFTlRfMSBkZXBsb3ltZW50IHR5cGVcbiAgICovXG4gIHJlYWRvbmx5IHBlclVuaXRTdG9yYWdlVGhyb3VnaHB1dD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHByZWZlcnJlZCBkYXkgYW5kIHRpbWUgdG8gcGVyZm9ybSB3ZWVrbHkgbWFpbnRlbmFuY2UuIFRoZSBmaXJzdCBkaWdpdCBpcyB0aGUgZGF5IG9mIHRoZSB3ZWVrLCBzdGFydGluZyBhdCAwXG4gICAqIGZvciBTdW5kYXksIHRoZW4gdGhlIGZvbGxvd2luZyBhcmUgaG91cnMgYW5kIG1pbnV0ZXMgaW4gdGhlIFVUQyB0aW1lIHpvbmUsIDI0IGhvdXIgY2xvY2suIEZvciBleGFtcGxlOiAnMjoyMDozMCdcbiAgICogaXMgVHVlc2RheXMgYXQgMjA6MzAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gcHJlZmVyZW5jZVxuICAgKi9cbiAgcmVhZG9ubHkgd2Vla2x5TWFpbnRlbmFuY2VTdGFydFRpbWU/OiBMdXN0cmVNYWludGVuYW5jZVRpbWU7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBzcGVjaWZpYyB0byB0aGUgTHVzdHJlIHZlcnNpb24gb2YgdGhlIEZTeCBmaWxlIHN5c3RlbS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMdXN0cmVGaWxlU3lzdGVtUHJvcHMgZXh0ZW5kcyBGaWxlU3lzdGVtUHJvcHMge1xuICAvKipcbiAgICogQWRkaXRpb25hbCBjb25maWd1cmF0aW9uIGZvciBGU3ggc3BlY2lmaWMgdG8gTHVzdHJlLlxuICAgKi9cbiAgcmVhZG9ubHkgbHVzdHJlQ29uZmlndXJhdGlvbjogTHVzdHJlQ29uZmlndXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldCB0aGF0IHRoZSBmaWxlIHN5c3RlbSB3aWxsIGJlIGFjY2Vzc2libGUgZnJvbS5cbiAgICovXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldDogSVN1Ym5ldDtcbn1cblxuLyoqXG4gKiBUaGUgRlN4IGZvciBMdXN0cmUgRmlsZSBTeXN0ZW0gaW1wbGVtZW50YXRpb24gb2YgSUZpbGVTeXN0ZW0uXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtZnN4LWZpbGVzeXN0ZW0uaHRtbFxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkZTeDo6RmlsZVN5c3RlbVxuICovXG5leHBvcnQgY2xhc3MgTHVzdHJlRmlsZVN5c3RlbSBleHRlbmRzIEZpbGVTeXN0ZW1CYXNlIHtcblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIEZTeCBmb3IgTHVzdHJlIGZpbGUgc3lzdGVtIGZyb20gdGhlIGdpdmVuIHByb3BlcnRpZXMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21MdXN0cmVGaWxlU3lzdGVtQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogRmlsZVN5c3RlbUF0dHJpYnV0ZXMpOiBJRmlsZVN5c3RlbSB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgRmlsZVN5c3RlbUJhc2Uge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRuc05hbWUgPSBhdHRycy5kbnNOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGZpbGVTeXN0ZW1JZCA9IGF0dHJzLmZpbGVTeXN0ZW1JZDtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9ucyA9IEx1c3RyZUZpbGVTeXN0ZW0uY29uZmlndXJlQ29ubmVjdGlvbnMoYXR0cnMuc2VjdXJpdHlHcm91cCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBGU3ggZmlsZSBzeXN0ZW0gdHlwZSB1c2VkIGJ5IEZTeCBmb3IgTHVzdHJlLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgREVGQVVMVF9GSUxFX1NZU1RFTV9UWVBFOiBzdHJpbmcgPSAnTFVTVFJFJztcblxuICAvKipcbiAgICogVGhlIGRlZmF1bHQgcG9ydHMgdGhlIGZpbGUgc3lzdGVtIGxpc3RlbnMgb24uIEFjdHVhbCBwb3J0IGxpc3QgaXM6IFs5ODgsIDEwMjEsIDEwMjIsIDEwMjNdXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBERUZBVUxUX1BPUlRfUkFOR0UgPSB7c3RhcnRQb3J0OiA5ODgsIGVuZFBvcnQ6IDEwMjN9O1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmVzIGEgQ29ubmVjdGlvbnMgb2JqZWN0IHdpdGggYWxsIHRoZSBwb3J0cyByZXF1aXJlZCBieSBGU3ggZm9yIEx1c3RyZVxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgY29uZmlndXJlQ29ubmVjdGlvbnMoc2VjdXJpdHlHcm91cDogSVNlY3VyaXR5R3JvdXApOiBDb25uZWN0aW9ucyB7XG4gICAgY29uc3QgY29ubmVjdGlvbnMgPSBuZXcgQ29ubmVjdGlvbnMoe1xuICAgICAgc2VjdXJpdHlHcm91cHM6IFtzZWN1cml0eUdyb3VwXSxcbiAgICAgIGRlZmF1bHRQb3J0OiBQb3J0LnRjcFJhbmdlKFxuICAgICAgICBMdXN0cmVGaWxlU3lzdGVtLkRFRkFVTFRfUE9SVF9SQU5HRS5zdGFydFBvcnQsXG4gICAgICAgIEx1c3RyZUZpbGVTeXN0ZW0uREVGQVVMVF9QT1JUX1JBTkdFLmVuZFBvcnQpLFxuICAgIH0pO1xuXG4gICAgcmV0dXJuIGNvbm5lY3Rpb25zO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBzZWN1cml0eSBncm91cHMvcnVsZXMgdXNlZCB0byBhbGxvdyBuZXR3b3JrIGNvbm5lY3Rpb25zIHRvIHRoZSBmaWxlIHN5c3RlbS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgLyoqXG4gICAqIFRoZSBETlMgbmFtZSBhc3NpZ25lZCB0byB0aGlzIGZpbGUgc3lzdGVtLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRuc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIElEIHRoYXQgQVdTIGFzc2lnbnMgdG8gdGhlIGZpbGUgc3lzdGVtLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZpbGVTeXN0ZW1JZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbW91bnQgbmFtZSBvZiB0aGUgZmlsZSBzeXN0ZW0sIGdlbmVyYXRlZCBieSBGU3hcbiAgICpcbiAgICogQGF0dHJpYnV0ZSBMdXN0cmVNb3VudE5hbWVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtb3VudE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGVuY2Fwc3VsYXRlZCBMMSBmaWxlIHN5c3RlbS5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgZmlsZVN5c3RlbTogQ2ZuRmlsZVN5c3RlbTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTHVzdHJlRmlsZVN5c3RlbVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCB1cGRhdGVkTHVzdHVyZVByb3BzID0ge1xuICAgICAgaW1wb3J0ZWRGaWxlQ2h1bmtTaXplOiBwcm9wcy5sdXN0cmVDb25maWd1cmF0aW9uLmltcG9ydGVkRmlsZUNodW5rU2l6ZU1pQixcbiAgICAgIHdlZWtseU1haW50ZW5hbmNlU3RhcnRUaW1lOiBwcm9wcy5sdXN0cmVDb25maWd1cmF0aW9uLndlZWtseU1haW50ZW5hbmNlU3RhcnRUaW1lPy50b1RpbWVzdGFtcCgpLFxuICAgIH07XG4gICAgY29uc3QgbHVzdHJlQ29uZmlndXJhdGlvbiA9IE9iamVjdC5hc3NpZ24oe30sIHByb3BzLmx1c3RyZUNvbmZpZ3VyYXRpb24sIHVwZGF0ZWRMdXN0dXJlUHJvcHMpO1xuXG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cCA9IChwcm9wcy5zZWN1cml0eUdyb3VwIHx8IG5ldyBTZWN1cml0eUdyb3VwKHRoaXMsICdGc3hMdXN0cmVTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgfSkpO1xuICAgIHNlY3VyaXR5R3JvdXAuYWRkSW5ncmVzc1J1bGUoXG4gICAgICBzZWN1cml0eUdyb3VwLFxuICAgICAgUG9ydC50Y3BSYW5nZShMdXN0cmVGaWxlU3lzdGVtLkRFRkFVTFRfUE9SVF9SQU5HRS5zdGFydFBvcnQsIEx1c3RyZUZpbGVTeXN0ZW0uREVGQVVMVF9QT1JUX1JBTkdFLmVuZFBvcnQpKTtcbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gTHVzdHJlRmlsZVN5c3RlbS5jb25maWd1cmVDb25uZWN0aW9ucyhzZWN1cml0eUdyb3VwKTtcblxuICAgIHRoaXMuZmlsZVN5c3RlbSA9IG5ldyBDZm5GaWxlU3lzdGVtKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGZpbGVTeXN0ZW1UeXBlOiBMdXN0cmVGaWxlU3lzdGVtLkRFRkFVTFRfRklMRV9TWVNURU1fVFlQRSxcbiAgICAgIHN1Ym5ldElkczogW3Byb3BzLnZwY1N1Ym5ldC5zdWJuZXRJZF0sXG4gICAgICBiYWNrdXBJZDogcHJvcHMuYmFja3VwSWQsXG4gICAgICBrbXNLZXlJZDogKHByb3BzLmttc0tleSA/IHByb3BzLmttc0tleS5rZXlJZCA6IHVuZGVmaW5lZCksXG4gICAgICBsdXN0cmVDb25maWd1cmF0aW9uLFxuICAgICAgc2VjdXJpdHlHcm91cElkczogW3NlY3VyaXR5R3JvdXAuc2VjdXJpdHlHcm91cElkXSxcbiAgICAgIHN0b3JhZ2VDYXBhY2l0eTogcHJvcHMuc3RvcmFnZUNhcGFjaXR5R2lCLFxuICAgIH0pO1xuXG4gICAgdGhpcy5maWxlU3lzdGVtSWQgPSB0aGlzLmZpbGVTeXN0ZW0ucmVmO1xuICAgIHRoaXMuZG5zTmFtZSA9IGAke3RoaXMuZmlsZVN5c3RlbUlkfS5mc3guJHt0aGlzLnN0YWNrLnJlZ2lvbn0uJHtBd3MuVVJMX1NVRkZJWH1gO1xuICAgIHRoaXMubW91bnROYW1lID0gdGhpcy5maWxlU3lzdGVtLmF0dHJMdXN0cmVNb3VudE5hbWU7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGVzIHRoZSBwcm9wcyBwcm92aWRlZCBmb3IgYSBuZXcgRlN4IGZvciBMdXN0cmUgZmlsZSBzeXN0ZW0uXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlUHJvcHMocHJvcHM6IEx1c3RyZUZpbGVTeXN0ZW1Qcm9wcykge1xuICAgIGNvbnN0IGx1c3RyZUNvbmZpZ3VyYXRpb24gPSBwcm9wcy5sdXN0cmVDb25maWd1cmF0aW9uO1xuICAgIGNvbnN0IGRlcGxveW1lbnRUeXBlID0gbHVzdHJlQ29uZmlndXJhdGlvbi5kZXBsb3ltZW50VHlwZTtcblxuICAgIC8vIE1ha2Ugc3VyZSB0aGUgaW1wb3J0IHBhdGggaXMgdmFsaWQgYmVmb3JlIHZhbGlkYXRpbmcgdGhlIGV4cG9ydCBwYXRoXG4gICAgdGhpcy52YWxpZGF0ZUltcG9ydFBhdGgobHVzdHJlQ29uZmlndXJhdGlvbi5pbXBvcnRQYXRoKTtcbiAgICB0aGlzLnZhbGlkYXRlRXhwb3J0UGF0aChsdXN0cmVDb25maWd1cmF0aW9uLmV4cG9ydFBhdGgsIGx1c3RyZUNvbmZpZ3VyYXRpb24uaW1wb3J0UGF0aCk7XG5cbiAgICB0aGlzLnZhbGlkYXRlSW1wb3J0ZWRGaWxlQ2h1bmtTaXplKGx1c3RyZUNvbmZpZ3VyYXRpb24uaW1wb3J0ZWRGaWxlQ2h1bmtTaXplTWlCKTtcbiAgICB0aGlzLnZhbGlkYXRlUGVyVW5pdFN0b3JhZ2VUaHJvdWdocHV0KGRlcGxveW1lbnRUeXBlLCBsdXN0cmVDb25maWd1cmF0aW9uLnBlclVuaXRTdG9yYWdlVGhyb3VnaHB1dCk7XG4gICAgdGhpcy52YWxpZGF0ZVN0b3JhZ2VDYXBhY2l0eShkZXBsb3ltZW50VHlwZSwgcHJvcHMuc3RvcmFnZUNhcGFjaXR5R2lCKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIGV4cG9ydCBwYXRoIGlzIGluIHRoZSBjb3JyZWN0IGZvcm1hdCBhbmQgbWF0Y2hlcyB0aGUgaW1wb3J0IHBhdGguXG4gICAqL1xuICBwcml2YXRlIHZhbGlkYXRlRXhwb3J0UGF0aChleHBvcnRQYXRoPzogc3RyaW5nLCBpbXBvcnRQYXRoPzogc3RyaW5nKTogdm9pZCB7XG4gICAgaWYgKGV4cG9ydFBhdGggPT09IHVuZGVmaW5lZCkgeyByZXR1cm47IH1cbiAgICBpZiAoaW1wb3J0UGF0aCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBkZWZpbmUgYW4gZXhwb3J0IHBhdGggd2l0aG91dCBhbHNvIGRlZmluaW5nIGFuIGltcG9ydCBwYXRoJyk7XG4gICAgfVxuXG4gICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZChleHBvcnRQYXRoKSAmJiBUb2tlbi5pc1VucmVzb2x2ZWQoaW1wb3J0UGF0aCkpIHsgcmV0dXJuOyB9XG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGltcG9ydFBhdGgpICE9PSBUb2tlbi5pc1VucmVzb2x2ZWQoZXhwb3J0UGF0aCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIGltcG9ydFBhdGggYW5kIGV4cG9ydFBhdGggbXVzdCBlYWNoIGJlIFRva2VucyBvciBub3QgVG9rZW5zLCB5b3UgY2Fubm90IHVzZSBhIG1peCcpO1xuICAgIH1cbiAgICBpZiAoIWV4cG9ydFBhdGguc3RhcnRzV2l0aChpbXBvcnRQYXRoKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgZXhwb3J0IHBhdGggXCIke2V4cG9ydFBhdGh9XCIgaXMgaW52YWxpZC4gRXhwZWN0aW5nIHRoZSBmb3JtYXQ6IHMzOi8ve0lNUE9SVF9QQVRIfS9vcHRpb25hbC1wcmVmaXhgKTtcbiAgICB9XG4gICAgaWYgKGV4cG9ydFBhdGgubGVuZ3RoID4gOTAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBleHBvcnQgcGF0aCBcIiR7ZXhwb3J0UGF0aH1cIiBleGNlZWRzIHRoZSBtYXhpbXVtIGxlbmd0aCBvZiA5MDAgY2hhcmFjdGVyc2ApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIGltcG9ydGVkRmlsZUNodW5rU2l6ZSBpcyBpbiB0aGUgY29ycmVjdCByYW5nZS5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVJbXBvcnRlZEZpbGVDaHVua1NpemUoaW1wb3J0ZWRGaWxlQ2h1bmtTaXplPzogbnVtYmVyKTogdm9pZCB7XG4gICAgaWYgKGltcG9ydGVkRmlsZUNodW5rU2l6ZSA9PT0gdW5kZWZpbmVkKSB7IHJldHVybjsgfVxuXG4gICAgaWYgKGltcG9ydGVkRmlsZUNodW5rU2l6ZSA8IDEgfHwgaW1wb3J0ZWRGaWxlQ2h1bmtTaXplID4gNTEyMDAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGltcG9ydGVkRmlsZUNodW5rU2l6ZSBjYW5ub3QgYmUgJHtpbXBvcnRlZEZpbGVDaHVua1NpemV9IE1pQi4gSXQgbXVzdCBiZSBhIHZhbHVlIGZyb20gMSB0byA1MTIsMDAwIE1pQmApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIGltcG9ydCBwYXRoIGlzIHRoZSBjb3JyZWN0IGZvcm1hdC5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVJbXBvcnRQYXRoKGltcG9ydFBhdGg/OiBzdHJpbmcpOiB2b2lkIHtcbiAgICBpZiAoaW1wb3J0UGF0aCA9PT0gdW5kZWZpbmVkIHx8IFRva2VuLmlzVW5yZXNvbHZlZChpbXBvcnRQYXRoKSkgeyByZXR1cm47IH1cblxuICAgIGNvbnN0IHJlZ2V4cCA9IC9eczM6XFwvXFwvLztcblxuICAgIGlmIChpbXBvcnRQYXRoLnNlYXJjaChyZWdleHApID09PSAtMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgaW1wb3J0IHBhdGggXCIke2ltcG9ydFBhdGh9XCIgaXMgaW52YWxpZC4gRXhwZWN0aW5nIHRoZSBmb3JtYXQ6IHMzOi8ve0JVQ0tFVF9OQU1FfS9vcHRpb25hbC1wcmVmaXhgKTtcbiAgICB9XG4gICAgaWYgKGltcG9ydFBhdGgubGVuZ3RoID4gOTAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBpbXBvcnQgcGF0aCBcIiR7aW1wb3J0UGF0aH1cIiBleGNlZWRzIHRoZSBtYXhpbXVtIGxlbmd0aCBvZiA5MDAgY2hhcmFjdGVyc2ApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIHBlclVuaXRTdG9yYWdlVGhyb3VnaHB1dCBpcyBkZWZpbmVkIGNvcnJlY3RseSBmb3IgdGhlIGdpdmVuIGRlcGxveW1lbnRUeXBlLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZVBlclVuaXRTdG9yYWdlVGhyb3VnaHB1dChkZXBsb3ltZW50VHlwZTogTHVzdHJlRGVwbG95bWVudFR5cGUsIHBlclVuaXRTdG9yYWdlVGhyb3VnaHB1dD86IG51bWJlcikge1xuICAgIGlmIChwZXJVbml0U3RvcmFnZVRocm91Z2hwdXQgPT09IHVuZGVmaW5lZCkgeyByZXR1cm47IH1cblxuICAgIGlmIChkZXBsb3ltZW50VHlwZSAhPT0gTHVzdHJlRGVwbG95bWVudFR5cGUuUEVSU0lTVEVOVF8xKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3BlclVuaXRTdG9yYWdlVGhyb3VnaHB1dCBjYW4gb25seSBiZSBzZXQgZm9yIHRoZSBQRVJTSVNURU5UXzEgZGVwbG95bWVudCB0eXBlJyk7XG4gICAgfVxuXG4gICAgaWYgKCFbNTAsIDEwMCwgMjAwXS5pbmNsdWRlcyhwZXJVbml0U3RvcmFnZVRocm91Z2hwdXQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3BlclVuaXRTdG9yYWdlVGhyb3VnaHB1dCBtdXN0IGJlIDUwLCAxMDAsIG9yIDIwMCBNQi9zL1RpQicpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZXMgdGhlIHN0b3JhZ2UgY2FwYWNpdHkgaXMgYW4gYWNjZXB0YWJsZSB2YWx1ZSBmb3IgdGhlIGRlcGxveW1lbnQgdHlwZS5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVTdG9yYWdlQ2FwYWNpdHkoZGVwbG95bWVudFR5cGU6IEx1c3RyZURlcGxveW1lbnRUeXBlLCBzdG9yYWdlQ2FwYWNpdHk6IG51bWJlcik6IHZvaWQge1xuICAgIGlmIChkZXBsb3ltZW50VHlwZSA9PT0gTHVzdHJlRGVwbG95bWVudFR5cGUuU0NSQVRDSF8xKSB7XG4gICAgICBpZiAoIVsxMjAwLCAyNDAwLCAzNjAwXS5pbmNsdWRlcyhzdG9yYWdlQ2FwYWNpdHkpICYmIHN0b3JhZ2VDYXBhY2l0eSAlIDM2MDAgIT09IDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdzdG9yYWdlQ2FwYWNpdHkgbXVzdCBiZSAxLDIwMCwgMiw0MDAsIDMsNjAwLCBvciBhIG11bHRpcGxlIG9mIDMsNjAwJyk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmICghWzEyMDAsIDI0MDBdLmluY2x1ZGVzKHN0b3JhZ2VDYXBhY2l0eSkgJiYgc3RvcmFnZUNhcGFjaXR5ICUgMjQwMCAhPT0gMCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ3N0b3JhZ2VDYXBhY2l0eSBtdXN0IGJlIDEsMjAwLCAyLDQwMCwgb3IgYSBtdWx0aXBsZSBvZiAyLDQwMCcpO1xuICAgICAgfVxuICAgIH1cbiAgfVxufVxuIl19