"""
Plugin task
====================================
A core functionality to execute different plugin tasks.
"""
import os
from enum import Enum
from functools import partial


def _add_action(name: str, directory: str = None) -> None:
    """Create a new plugin

    It will generate all the required structure for a new plugin (files and folders).

    Parameters
    ----------
    name : str
        Name of the plugin to create.
    directory : str
        The path to create the plugin.
    """
    path = directory if directory is not None else os.getcwd()
    if os.path.exists(f"{path}/{name}"):
        raise FileExistsError(f"Directory {path}/{name} already exists.")
    os.mkdir(f"{path}/{name}")
    plugin_name = f"{name.capitalize()}Plugin"
    context_name = f"{name.capitalize()}Context"
    with open(f"{path}/{name}/__init__.py", 'w') as init_file:
        init_file.write(f"import .{name} from {plugin_name}\n")
        init_file.write(f"import .{name} from {context_name}\n")
        init_file.close()
    with open(f"{path}/{name}/{name}.py", 'w') as plugin_file:
        plugin_file.write(f"from openvariant.plugins.context import Context\n")
        plugin_file.write(f"from openvariant.plugins.plugin import Plugin\n\n\n")
        plugin_file.write(f"class {context_name}(Context):\n")
        plugin_file.write(f"\t\"\"\"\n")
        plugin_file.write(f"\t{context_name} class generated by OpenVariant\n\n")
        plugin_file.write(f"\tAttributes\n")
        plugin_file.write(f"\t-------\n")
        plugin_file.write(f"\trow : dict\n")
        plugin_file.write(f"\t\tMain method to execute data transformation in each row.\n")
        plugin_file.write(f"\tfield_name : str\n")
        plugin_file.write(f"\t\tName of the corresponding column that was described on the annotation schema.\n")
        plugin_file.write(f"\tfile_path : str\n")
        plugin_file.write(f"\t\tPath of the Input file that is being parsed.\n")
        plugin_file.write(f"\t\"\"\"\n\n")
        plugin_file.write(f"\tdef __init__(self, row: dict, field_name: str, file_path: str) -> None:\n")
        plugin_file.write(f"\t\tsuper().__init__(row, field_name, file_path)\n")
        plugin_file.write(f"\n\n")
        plugin_file.write(f"class {plugin_name}(Plugin):\n")
        plugin_file.write(f"\t\"\"\"\n")
        plugin_file.write(f"\t{plugin_name} class generated by OpenVariant\n\n")
        plugin_file.write(f"\tMethods\n")
        plugin_file.write(f"\t-------\n")
        plugin_file.write(f"\trun(context: {context_name})\n")
        plugin_file.write(f"\t\tMain method to execute data transformation in each row.\n")
        plugin_file.write(f"\t\"\"\"\n\n")
        plugin_file.write(f"\tdef run(self, context: {context_name}) -> dict:\n")
        plugin_file.write(f"\t\t\"\"\"\n")
        plugin_file.write(f"\t\tData transformation of a single row\n\n")
        plugin_file.write(f"\t\tParameters\n")
        plugin_file.write(f"\t\t-------\n")
        plugin_file.write(f"\t\tcontext : {context_name}\n")
        plugin_file.write(f"\t\t\tRepresentation of the row to be parsed.\n")
        plugin_file.write(f"\t\t\n")
        plugin_file.write(f"\t\tReturns\n")
        plugin_file.write(f"\t\t-------\n")
        plugin_file.write(f"\t\tfloat or int or str\n")
        plugin_file.write(f"\t\t\tThe value of the field transformed.\n")
        plugin_file.write(f"\t\t\"\"\"\n")
        plugin_file.write(f"\n")
        plugin_file.write(f"\t\t# This is an example code, modify if as you wish\n")
        plugin_file.write(f"\t\tcontext.row[context.field_name] = \"Hello world! file:\" + context.file_path\n")
        plugin_file.write(f"\n")
        plugin_file.write(f"\t\treturn context.row[context.field_name]\n")


class PluginActions(Enum):
    """Enum to get the function of each action"""

    """Create a new plugin"""
    CREATE = partial(_add_action)
