# -*- coding: utf-8 -*-
# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import warnings
from typing import Awaitable, Callable, Dict, Optional, Sequence, Tuple, Union

from google.api_core import gapic_v1  # type: ignore
from google.api_core import grpc_helpers_async  # type: ignore
from google.api_core import operations_v1  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
import packaging.version

import grpc  # type: ignore
from grpc.experimental import aio  # type: ignore

from google.cloud.dialogflowcx_v3beta1.types import flow
from google.cloud.dialogflowcx_v3beta1.types import flow as gcdc_flow
from google.longrunning import operations_pb2  # type: ignore
from google.protobuf import empty_pb2  # type: ignore
from .base import FlowsTransport, DEFAULT_CLIENT_INFO
from .grpc import FlowsGrpcTransport


class FlowsGrpcAsyncIOTransport(FlowsTransport):
    """gRPC AsyncIO backend transport for Flows.

    Service for managing
    [Flows][google.cloud.dialogflow.cx.v3beta1.Flow].

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _grpc_channel: aio.Channel
    _stubs: Dict[str, Callable] = {}

    @classmethod
    def create_channel(
        cls,
        host: str = "dialogflow.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> aio.Channel:
        """Create and return a gRPC AsyncIO channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            aio.Channel: A gRPC AsyncIO channel object.
        """

        return grpc_helpers_async.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    def __init__(
        self,
        *,
        host: str = "dialogflow.googleapis.com",
        credentials: ga_credentials.Credentials = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: aio.Channel = None,
        api_mtls_endpoint: str = None,
        client_cert_source: Callable[[], Tuple[bytes, bytes]] = None,
        ssl_channel_credentials: grpc.ChannelCredentials = None,
        client_cert_source_for_mtls: Callable[[], Tuple[bytes, bytes]] = None,
        quota_project_id=None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            channel (Optional[aio.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or applicatin default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
            google.auth.exceptions.MutualTlsChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}
        self._operations_client = None

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None
        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                credentials=self._credentials,
                credentials_file=credentials_file,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @property
    def grpc_channel(self) -> aio.Channel:
        """Create the channel designed to connect to this service.

        This property caches on the instance; repeated calls return
        the same channel.
        """
        # Return the channel from cache.
        return self._grpc_channel

    @property
    def operations_client(self) -> operations_v1.OperationsAsyncClient:
        """Create the client designed to process long-running operations.

        This property caches on the instance; repeated calls return the same
        client.
        """
        # Sanity check: Only create a new client if we do not already have one.
        if self._operations_client is None:
            self._operations_client = operations_v1.OperationsAsyncClient(
                self.grpc_channel
            )

        # Return the client from cache.
        return self._operations_client

    @property
    def create_flow(
        self,
    ) -> Callable[[gcdc_flow.CreateFlowRequest], Awaitable[gcdc_flow.Flow]]:
        r"""Return a callable for the create flow method over gRPC.

        Creates a flow in the specified agent.

        Note: You should always train a flow prior to sending it
        queries. See the `training
        documentation <https://cloud.google.com/dialogflow/cx/docs/concept/training>`__.

        Returns:
            Callable[[~.CreateFlowRequest],
                    Awaitable[~.Flow]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_flow" not in self._stubs:
            self._stubs["create_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/CreateFlow",
                request_serializer=gcdc_flow.CreateFlowRequest.serialize,
                response_deserializer=gcdc_flow.Flow.deserialize,
            )
        return self._stubs["create_flow"]

    @property
    def delete_flow(
        self,
    ) -> Callable[[flow.DeleteFlowRequest], Awaitable[empty_pb2.Empty]]:
        r"""Return a callable for the delete flow method over gRPC.

        Deletes a specified flow.

        Returns:
            Callable[[~.DeleteFlowRequest],
                    Awaitable[~.Empty]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_flow" not in self._stubs:
            self._stubs["delete_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/DeleteFlow",
                request_serializer=flow.DeleteFlowRequest.serialize,
                response_deserializer=empty_pb2.Empty.FromString,
            )
        return self._stubs["delete_flow"]

    @property
    def list_flows(
        self,
    ) -> Callable[[flow.ListFlowsRequest], Awaitable[flow.ListFlowsResponse]]:
        r"""Return a callable for the list flows method over gRPC.

        Returns the list of all flows in the specified agent.

        Returns:
            Callable[[~.ListFlowsRequest],
                    Awaitable[~.ListFlowsResponse]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_flows" not in self._stubs:
            self._stubs["list_flows"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/ListFlows",
                request_serializer=flow.ListFlowsRequest.serialize,
                response_deserializer=flow.ListFlowsResponse.deserialize,
            )
        return self._stubs["list_flows"]

    @property
    def get_flow(self) -> Callable[[flow.GetFlowRequest], Awaitable[flow.Flow]]:
        r"""Return a callable for the get flow method over gRPC.

        Retrieves the specified flow.

        Returns:
            Callable[[~.GetFlowRequest],
                    Awaitable[~.Flow]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_flow" not in self._stubs:
            self._stubs["get_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/GetFlow",
                request_serializer=flow.GetFlowRequest.serialize,
                response_deserializer=flow.Flow.deserialize,
            )
        return self._stubs["get_flow"]

    @property
    def update_flow(
        self,
    ) -> Callable[[gcdc_flow.UpdateFlowRequest], Awaitable[gcdc_flow.Flow]]:
        r"""Return a callable for the update flow method over gRPC.

        Updates the specified flow.

        Note: You should always train a flow prior to sending it
        queries. See the `training
        documentation <https://cloud.google.com/dialogflow/cx/docs/concept/training>`__.

        Returns:
            Callable[[~.UpdateFlowRequest],
                    Awaitable[~.Flow]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_flow" not in self._stubs:
            self._stubs["update_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/UpdateFlow",
                request_serializer=gcdc_flow.UpdateFlowRequest.serialize,
                response_deserializer=gcdc_flow.Flow.deserialize,
            )
        return self._stubs["update_flow"]

    @property
    def train_flow(
        self,
    ) -> Callable[[flow.TrainFlowRequest], Awaitable[operations_pb2.Operation]]:
        r"""Return a callable for the train flow method over gRPC.

        Trains the specified flow. Note that only the flow in 'draft'
        environment is trained.

        Note: You should always train a flow prior to sending it
        queries. See the `training
        documentation <https://cloud.google.com/dialogflow/cx/docs/concept/training>`__.

        Returns:
            Callable[[~.TrainFlowRequest],
                    Awaitable[~.Operation]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "train_flow" not in self._stubs:
            self._stubs["train_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/TrainFlow",
                request_serializer=flow.TrainFlowRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["train_flow"]

    @property
    def validate_flow(
        self,
    ) -> Callable[[flow.ValidateFlowRequest], Awaitable[flow.FlowValidationResult]]:
        r"""Return a callable for the validate flow method over gRPC.

        Validates the specified flow and creates or updates
        validation results. Please call this API after the
        training is completed to get the complete validation
        results.

        Returns:
            Callable[[~.ValidateFlowRequest],
                    Awaitable[~.FlowValidationResult]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "validate_flow" not in self._stubs:
            self._stubs["validate_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/ValidateFlow",
                request_serializer=flow.ValidateFlowRequest.serialize,
                response_deserializer=flow.FlowValidationResult.deserialize,
            )
        return self._stubs["validate_flow"]

    @property
    def get_flow_validation_result(
        self,
    ) -> Callable[
        [flow.GetFlowValidationResultRequest], Awaitable[flow.FlowValidationResult]
    ]:
        r"""Return a callable for the get flow validation result method over gRPC.

        Gets the latest flow validation result. Flow
        validation is performed when ValidateFlow is called.

        Returns:
            Callable[[~.GetFlowValidationResultRequest],
                    Awaitable[~.FlowValidationResult]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_flow_validation_result" not in self._stubs:
            self._stubs["get_flow_validation_result"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/GetFlowValidationResult",
                request_serializer=flow.GetFlowValidationResultRequest.serialize,
                response_deserializer=flow.FlowValidationResult.deserialize,
            )
        return self._stubs["get_flow_validation_result"]

    @property
    def import_flow(
        self,
    ) -> Callable[[flow.ImportFlowRequest], Awaitable[operations_pb2.Operation]]:
        r"""Return a callable for the import flow method over gRPC.

        Imports the specified flow to the specified agent from a binary
        file.

        Note: You should always train a flow prior to sending it
        queries. See the `training
        documentation <https://cloud.google.com/dialogflow/cx/docs/concept/training>`__.

        Returns:
            Callable[[~.ImportFlowRequest],
                    Awaitable[~.Operation]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "import_flow" not in self._stubs:
            self._stubs["import_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/ImportFlow",
                request_serializer=flow.ImportFlowRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["import_flow"]

    @property
    def export_flow(
        self,
    ) -> Callable[[flow.ExportFlowRequest], Awaitable[operations_pb2.Operation]]:
        r"""Return a callable for the export flow method over gRPC.

        Exports the specified flow to a binary file.
        Note that resources (e.g. intents, entities, webhooks)
        that the flow references will also be exported.

        Returns:
            Callable[[~.ExportFlowRequest],
                    Awaitable[~.Operation]]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "export_flow" not in self._stubs:
            self._stubs["export_flow"] = self.grpc_channel.unary_unary(
                "/google.cloud.dialogflow.cx.v3beta1.Flows/ExportFlow",
                request_serializer=flow.ExportFlowRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["export_flow"]


__all__ = ("FlowsGrpcAsyncIOTransport",)
