"""Utils for finding the closest match of a string in a list of strings."""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../../pts/api/algos/00_str_matching.pct.py.

# %% auto 0
__all__ = ['fuzzy_match', 'get_vector_dist_matrix', 'embedding_match']

# %% ../../../pts/api/algos/00_str_matching.pct.py 3
try:
    import rapidfuzz
    import numpy as np
except ImportError as e:
    raise ImportError(f"Install adulib[algos] to use this API.") from e

# %% ../../../pts/api/algos/00_str_matching.pct.py 6
def fuzzy_match(query_string, candidate_strings, max_results=5, min_similarity=0, scorer=rapidfuzz.fuzz.ratio):
    """
    Find the closest fuzzy matches to a query string from a list of candidate strings using RapidFuzz.
    
    This is a simplified wrapper around `rapidfuzz.process.extract` to find fuzzy matches. The equivalent of, and much faster version of, `difflib.get_close_matches`.

    Args:
        query_string (str): The string to match against the candidates.
        candidate_strings (Iterable[str]): The list of candidate strings to search.
        max_results (int, optional): Maximum number of matches to return. Defaults to 8.
        min_similarity (float, optional): Minimum similarity threshold (0.0-1.0). Defaults to 0.1.
        scorer (callable, optional): Scoring function from rapidfuzz.fuzz. Defaults to rapidfuzz.fuzz.ratio. See rapidfuzz.fuzz for available scorers.

    Returns:
        List[Tuple[str, float, int]]: List of tuples containing (matched_string, score, index).
    """
    results = rapidfuzz.process.extract(
        query_string,
        candidate_strings,
        scorer=rapidfuzz.fuzz.ratio,
        limit=max_results,
        score_cutoff=min_similarity * 100 # rapidfuzz scores are 0-100, so cutoff needs to be scaled
    )
    return results

# %% ../../../pts/api/algos/00_str_matching.pct.py 9
def get_vector_dist_matrix(
    vectors:list[list[float]],
    metric:str="cosine",
):
    """
    Calculate the pairwise distance matrix for a set of vectors.

    Args:
        vectors (List[List[float]]): List of vectors to calculate distances for.
        metric (str, optional): Distance metric to use. Defaults to "cosine". Options include "euclidean", "manhattan", "cosine", etc. See sklearn.metrics.pairwise_distances for more options.

    Returns:
        np.ndarray: Distance matrix. Each element [i, j] represents the distance between vectors[i] and vectors[j].
    """
    from sklearn.metrics import pairwise_distances
    return pairwise_distances(vectors, metric=metric)

# %% ../../../pts/api/algos/00_str_matching.pct.py 12
def embedding_match(embedding_index, dist_matrix, num_matches=5):
    """
    Find the indices and distances of the closest matches to a given embedding. Use it in combination with `get_vector_dist_matrix` to find the closest embeddings.

    Args:
        embedding_index (int): Index of the embedding to match.
        dist_matrix (np.ndarray): Pairwise distance matrix of embeddings. Computed using `get_vector_dist_matrix`.
        num_matches (int, optional): Number of closest matches to return (excluding self). Defaults to 5.

    Returns:
        Tuple[np.ndarray, np.ndarray]: Tuple of (indices of closest matches, distances to those matches).
    """
    distances = dist_matrix[embedding_index]
    closest_indices = np.argsort(distances)[1:num_matches+1]
    closest_distances = [distances[i] for i in closest_indices]
    return closest_indices, np.array(closest_distances)
