# Copyright 2008 Canonical Ltd.  All rights reserved.

"""Test for the WADL generation."""

from __future__ import absolute_import, print_function

__metaclass__ = type

from contextlib import contextmanager
from io import (
    BytesIO,
    StringIO,
    )
from lxml import etree
import collections
import logging
import random
import re
import simplejson
import unittest

import six
from zope.component import (
    eventtesting,
    getGlobalSiteManager,
    getUtility,
    )
from zope.interface import implementer, Interface
from zope.interface.interface import InterfaceClass
from zope.publisher.browser import TestRequest
from zope.schema import Choice, Date, Datetime, TextLine
from zope.schema.interfaces import ITextLine
from zope.security.management import (
    endInteraction,
    newInteraction,
    queryInteraction,
    )
from zope.traversing.browser.interfaces import IAbsoluteURL

from lazr.enum import EnumeratedType, Item
from lazr.restful import (
    EntryField,
    ResourceOperation,
    )
from lazr.restful.fields import Reference
from lazr.restful.interfaces import (
    ICollection,
    IEntry,
    IFieldHTMLRenderer,
    INotificationsProvider,
    IResourceGETOperation,
    IServiceRootResource,
    IWebBrowserOriginatingRequest,
    IWebServiceConfiguration,
    IWebServiceClientRequest,
    IWebServiceVersion,
    )
from lazr.restful import (
    EntryFieldResource,
    EntryResource,
    ResourceGETOperation,
    )
from lazr.restful.declarations import (
    export_read_operation,
    exported,
    exported_as_webservice_entry,
    LAZR_WEBSERVICE_NAME,
    scoped,
    )
from lazr.restful.testing.webservice import (
    create_web_service_request,
    DummyAbsoluteURL,
    IGenericCollection,
    IGenericEntry,
    simple_renderer,
    WebServiceTestCase,
    )
from lazr.restful.testing.tales import test_tales
from lazr.restful.utils import (
    get_current_browser_request, get_current_web_service_request,
    tag_request_with_version_name)
from lazr.restful._resource import CollectionResource, BatchingResourceMixin


def get_resource_factory(model_interface, resource_interface):
    """Return the autogenerated adapter class for a model_interface.

    :param model_interface: the annnotated interface for which we are looking
        for the web service resource adapter
    :param resource_interface: the method provided by the resource, usually
        `IEntry` or `ICollection`.
    :return: the resource factory (the autogenerated adapter class.
    """
    request_interface = getUtility(IWebServiceVersion, name='2.0')
    return getGlobalSiteManager().adapters.lookup(
        (model_interface, request_interface), resource_interface)


def get_operation_factory(model_interface, name):
    """Find the factory for a GET operation adapter.

    :param model_interface: the model interface on which the operation is
        defined.
    :param name: the name of the exported method.
    :return: the factory (autogenerated class) that implements the operation
        on the webservice.
    """
    request_interface = getUtility(IWebServiceVersion, name='2.0')
    return getGlobalSiteManager().adapters.lookup(
        (model_interface, request_interface),
        IResourceGETOperation, name=name)


class IHas_getitem(Interface):
    pass


@implementer(IHas_getitem)
class Has_getitem:
    def __getitem__(self, item):
        return "wibble"


class ResourceOperationTestCase(unittest.TestCase):
    """A test case for resource operations."""

    def test_object_with_getitem_should_not_batch(self):
        """Test ResourceOperation.should_batch().

        Custom operations returning a Reference to objects that
        implement __getitem__ should not batch the results (iter() on
        such objects does not fail).
        """
        return_type = Reference(IHas_getitem)
        result = Has_getitem()

        operation = ResourceGETOperation("fake context", "fake request")
        operation.return_type = return_type

        self.assertFalse(
            operation.should_batch(result),
            "Batching should not happen for Reference return types.")


class EntryTestCase(WebServiceTestCase):
    """A test suite that defines an entry class."""

    WADL_NS = "{http://research.sun.com/wadl/2006/10}"

    default_media_type = "application/json"

    @implementer(IWebBrowserOriginatingRequest)
    class DummyWebsiteRequest:
        """A request to the website, as opposed to the web service."""

    class DummyWebsiteURL(DummyAbsoluteURL):
        """A web-centric implementation of the dummy URL."""
        URL = 'http://www.website.url/'

    @contextmanager
    def request(self, media_type=None):
        media_type = media_type or self.default_media_type
        request = getUtility(IWebServiceConfiguration).createRequest(
            BytesIO(b""), {'HTTP_ACCEPT' : media_type})
        newInteraction(request)
        yield request
        endInteraction()

    @property
    def wadl(self):
        """Get a parsed WADL description of the web service."""
        with self.request() as request:
            return request.publication.application.toWADL()

    @contextmanager
    def entry_resource(self, entry_interface, entry_implementation,
                       *implementation_args):
        """Create a request to an entry resource, and yield the resource."""
        entry_class = get_resource_factory(entry_interface, IEntry)
        data_object = entry_implementation(*implementation_args)

        with self.request() as request:
            entry = entry_class(data_object, request)
            resource = EntryResource(data_object, request)
            yield resource

    @contextmanager
    def entry_field_resource(self, entry_interface, entry_implementation,
                             field_name, *implementation_args):
        entry_class = get_resource_factory(entry_interface, IEntry)
        data_object = entry_implementation(*implementation_args)
        with self.request() as request:
            entry = entry_class(data_object, request)
            field = entry.schema.get(field_name)
            entry_field = EntryField(entry, field, field_name)
            resource = EntryFieldResource(entry_field, request)
            yield resource

    def register_html_field_renderer(self, entry_interface, field_interface,
                                     render_function, name=''):
        """Register an HTML representation for a field or class of field."""
        def renderer(context, field, request):
            return render_function

        getGlobalSiteManager().registerAdapter(
            renderer,
            (entry_interface, field_interface, IWebServiceClientRequest),
            IFieldHTMLRenderer, name=name)

    def _register_url_adapter(self, entry_interface):
        """Register an IAbsoluteURL implementation for an interface."""
        getGlobalSiteManager().registerAdapter(
            DummyAbsoluteURL, [entry_interface, IWebServiceClientRequest],
            IAbsoluteURL)

    def _register_website_url_space(self, entry_interface):
        """Simulates a service where an entry corresponds to a web page."""
        self._register_url_adapter(entry_interface)

        # First, create a converter from web service requests to
        # web page requests.
        def web_service_request_to_website_request(service_request):
            """Create a corresponding request to the website."""
            return self.DummyWebsiteRequest()

        getGlobalSiteManager().registerAdapter(
            web_service_request_to_website_request,
            [IWebServiceClientRequest], IWebBrowserOriginatingRequest)

        # Next, set up a distinctive URL, and register it as the
        # website URL for the given entry interface.
        getGlobalSiteManager().registerAdapter(
            self.DummyWebsiteURL,
            [entry_interface, IWebBrowserOriginatingRequest],
            IAbsoluteURL)


@exported_as_webservice_entry()
class IHasOneField(Interface):
    """An entry with a single field."""
    a_field = exported(TextLine(title=u"A field."))


@implementer(IHasOneField)
class HasOneField:
    """An implementation of IHasOneField."""
    def __init__(self, value):
        self.a_field = value


@exported_as_webservice_entry()
class IHasTwoFields(Interface):
    """An entry with two fields."""
    a_field = exported(TextLine(title=u"A field."))
    another_field = exported(TextLine(title=u"Another field."))


@implementer(IHasTwoFields)
class HasTwoFields:
    """An implementation of IHasTwoFields."""
    def __init__(self, value1, value2):
        self.a_field = value1
        self.another_field = value2


class TestEntryWebLink(EntryTestCase):

    testmodule_objects = [HasOneField, IHasOneField]

    def test_entry_includes_web_link_when_available(self):
        # If a web service request can be adapted to a web*site* request,
        # the representation of an entry will include a link to the
        # corresponding entry on the website.
        #
        # This is useful when each entry published by the web service
        # has a human-readable page on some corresponding website. The
        # web service can publish links to the website for use by Ajax
        # clients or for other human-interaction purposes.
        self._register_website_url_space(IHasOneField)

        # Now a representation of IHasOneField includes a
        # 'web_link'.
        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            representation = resource.toDataForJSON()
            self.assertEqual(representation['self_link'], DummyAbsoluteURL.URL)
            self.assertEqual(
                representation['web_link'], self.DummyWebsiteURL.URL)

    def test_wadl_includes_web_link_when_available(self):
        # If an entry includes a web_link, this information will
        # show up in the WADL description of the entry.
        self._register_website_url_space(IHasOneField)

        doc = etree.parse(StringIO(self.wadl))
        # Verify that the 'has_one_field-full' representation includes
        # a 'web_link' param.
        representation = [
            rep for rep in doc.findall('%srepresentation' % self.WADL_NS)
            if rep.get('id') == 'has_one_field-full'][0]
        param = [
            param for param in representation.findall(
                '%sparam' % self.WADL_NS)
            if param.get('name') == 'web_link'][0]

        # Verify that the 'web_link' param includes a 'link' tag.
        self.assertFalse(param.find('%slink' % self.WADL_NS) is None)

    def test_entry_omits_web_link_when_not_available(self):
        # When there is no way of turning a webservice request into a
        # website request, the 'web_link' attribute is missing from
        # entry representations.
        self._register_url_adapter(IHasOneField)

        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            representation = resource.toDataForJSON()
            self.assertEqual(
                representation['self_link'], DummyAbsoluteURL.URL)
            self.assertFalse('web_link' in representation)

    def test_wadl_omits_web_link_when_not_available(self):
        # When there is no way of turning a webservice request into a
        # website request, the 'web_link' attribute is missing from
        # WADL descriptions of entries.
        self._register_url_adapter(IHasOneField)
        self.assertFalse('web_link' in self.wadl)


@exported_as_webservice_entry(publish_web_link=False)
class IHasNoWebLink(Interface):
    """An entry that does not publish a web_link."""
    a_field = exported(TextLine(title=u"A field."))


@implementer(IHasNoWebLink)
class HasNoWebLink:
    """An implementation of IHasNoWebLink."""
    def __init__(self, value):
        self.a_field = value


class TestSuppressWebLink(EntryTestCase):
    """Test the ability to suppress web_link on a per-entry basis."""

    testmodule_objects = [IHasNoWebLink, HasNoWebLink]

    def test_entry_omits_web_link_when_suppressed(self):
        self._register_website_url_space(IHasNoWebLink)

        with self.entry_resource(IHasNoWebLink, HasNoWebLink, "") as (
            resource):
            representation = resource.toDataForJSON()
            self.assertEqual(
                representation['self_link'], DummyAbsoluteURL.URL)
            self.assertFalse('web_link' in representation)


class InterfaceRestrictedField(TextLine):
    """A field that must be exported from one kind of interface."""

    def __init__(self, restrict_to_interface, *args, **kwargs):
        self.restrict_to_interface = restrict_to_interface
        super(InterfaceRestrictedField, self).__init__(*args, **kwargs)

    def bind(self, context):
        if not self.restrict_to_interface.providedBy(context):
            raise AssertionError(
                "InterfaceRestrictedField can only be used with %s"
                % self.restrict_to_interface.__name__)
        return super(InterfaceRestrictedField, self).bind(context)


@exported_as_webservice_entry()
class IHasRestrictedField(Interface):
    """An entry with an InterfaceRestrictedField."""
    a_field = exported(InterfaceRestrictedField(Interface))


@implementer(IHasRestrictedField)
class HasRestrictedField:
    """An implementation of IHasRestrictedField."""
    def __init__(self, value):
        self.a_field = value


@exported_as_webservice_entry()
class IHasFieldExportedAsDifferentName(Interface):
    """An entry with a field exported as a different name."""
    a_field = exported(TextLine(title=u"A field."), exported_as='field')


@implementer(IHasFieldExportedAsDifferentName)
class HasFieldExportedAsDifferentName:
    """An implementation of IHasFieldExportedAsDifferentName."""
    def __init__(self, value):
        self.a_field = value


class TestEntryWrite(EntryTestCase):

    testmodule_objects = [
        IHasOneField, HasOneField,
        IHasFieldExportedAsDifferentName, HasFieldExportedAsDifferentName]

    def test_applyChanges_rejects_nonexistent_web_link(self):
        # If web_link is not published, applyChanges rejects a request
        # that references it.
        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            errors = resource.applyChanges({'web_link': u'some_value'})
            self.assertEqual(
                errors,
                'web_link: You tried to modify a nonexistent attribute.')

    def test_applyChanges_rejects_changed_web_link(self):
        """applyChanges rejects an attempt to change web_link ."""
        self._register_website_url_space(IHasOneField)

        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            errors = resource.applyChanges({'web_link': u'some_value'})
            self.assertEqual(
                errors,
                'web_link: You tried to modify a read-only attribute.')

    def test_applyChanges_accepts_unchanged_web_link(self):
        # applyChanges accepts a reference to web_link, as long as the
        # value isn't actually being changed.
        self._register_website_url_space(IHasOneField)

        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            existing_web_link = resource.toDataForJSON()['web_link']
            representation = simplejson.loads(
                resource.applyChanges({'web_link': existing_web_link}))
            self.assertEqual(representation['web_link'], existing_web_link)

    def test_applyChanges_returns_representation_on_empty_changeset(self):
        # Even if the changeset is empty, applyChanges returns a
        # representation of the (unchanged) resource.
        self._register_website_url_space(IHasOneField)

        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            existing_representation = resource.toDataForJSON()
            representation = simplejson.loads(resource.applyChanges({}))
            self.assertEqual(representation, existing_representation)

    def test_applyChanges_returns_modified_representation_on_change(self):
        # If a change is made, the representation that is returned
        # represents the new state.  Bug fix: this should happen even
        # if the resource and its context have different names for the
        # same value.
        self._register_website_url_space(IHasFieldExportedAsDifferentName)

        with self.entry_resource(
            IHasFieldExportedAsDifferentName,
            HasFieldExportedAsDifferentName,
            u"initial value") as resource:
            # We have an entry that has a different name on the
            # entry than on its context.
            entry = resource.entry
            self.assertEqual(entry.field, entry.context.a_field)
            # This populates the cache.
            self.assertEqual(
                u'initial value', resource.toDataForJSON()['field'])
            # This returns the changed value.
            representation = simplejson.loads(
                resource.applyChanges(dict(field=u'new value')))
            self.assertEqual(
                u'new value', representation['field'])


class TestEntryWriteForRestrictedField(EntryTestCase):

    testmodule_objects = [IHasRestrictedField, HasRestrictedField]

    def test_applyChanges_binds_to_resource_context(self):
        """Make sure applyChanges binds fields to the resource context.

        This case verifies that applyChanges binds fields to the entry
        resource's context, not the resource itself. If an
        InterfaceRestrictedField is bound to an object that doesn't
        expose the right interface, it will raise an exception.
        """
        self._register_url_adapter(IHasRestrictedField)
        with self.entry_resource(
            IHasRestrictedField, HasRestrictedField, "") as resource:
            entry = resource.entry
            entry.schema['a_field'].restrict_to_interface = IHasRestrictedField
            self.assertEqual(entry.a_field, '')
            resource.applyChanges({'a_field': u'a_value'})
            self.assertEqual(entry.a_field, 'a_value')

            # Make sure that IHasRestrictedField itself works correctly.
            class IOtherInterface(Interface):
                """An interface not provided by IHasRestrictedField."""
                pass
            entry.schema['a_field'].restrict_to_interface = IOtherInterface
            self.assertRaises(AssertionError, resource.applyChanges,
                              {'a_field': u'a_new_value'})
            self.assertEqual(resource.entry.a_field, 'a_value')


class HTMLRepresentationTest(EntryTestCase):

    testmodule_objects = [HasOneField, IHasOneField]
    default_media_type = "application/xhtml+xml"

    def setUp(self):
        super(HTMLRepresentationTest, self).setUp()
        self._register_url_adapter(IHasOneField)
        self.unicode_message = u"Hello from a \N{SNOWMAN}"
        self.utf8_message = self.unicode_message.encode('utf-8')

    def test_entry_html_representation_is_utf8(self):
        with self.entry_resource(
            IHasOneField, HasOneField, self.unicode_message) as resource:
            html = resource.do_GET()
            self.assertTrue(self.utf8_message in html)

    def test_field_html_representation_is_utf8(self):
        with self.entry_field_resource(
            IHasOneField, HasOneField, "a_field",
            self.unicode_message) as resource:
            html = resource.do_GET()
            self.assertTrue(html == self.utf8_message)


class JSONPlusHTMLRepresentationTest(EntryTestCase):

    testmodule_objects = [HasTwoFields, IHasTwoFields]

    def setUp(self):
        super(JSONPlusHTMLRepresentationTest, self).setUp()
        self.default_media_type = "application/json;include=lp_html"
        self._register_url_adapter(IHasTwoFields)

    def register_html_field_renderer(self, name=''):
        """Simplify the register_html_field_renderer call."""
        super(JSONPlusHTMLRepresentationTest,
              self).register_html_field_renderer(
            IHasTwoFields, ITextLine, simple_renderer, name)

    @contextmanager
    def resource(self, value_1="value 1", value_2="value 2"):
        """Simplify the entry_resource call."""
        with self.entry_resource(
                IHasTwoFields, HasTwoFields,
                six.text_type(value_1), six.text_type(value_2)) as resource:
            yield resource

    def test_web_layer_json_representation_omits_lp_html(self):
        self.register_html_field_renderer()
        with self.resource() as resource:
            tales_string = test_tales(
                "entry/webservice:json", entry=resource.entry.context)
            self.assertFalse("lp_html" in tales_string)

    def test_normal_json_representation_omits_lp_html(self):
        self.default_media_type = "application/json"
        self.register_html_field_renderer()
        with self.resource() as resource:
            json = simplejson.loads(resource.do_GET())
            self.assertFalse('lp_html' in json)

    def test_entry_with_no_html_renderers_omits_lp_html(self):
        with self.resource() as resource:
            json = simplejson.loads(resource.do_GET())
            self.assertFalse('lp_html' in json)
            self.assertEqual(
                resource.request.response.getHeader("Content-Type"),
                "application/json")

    def test_field_specific_html_renderer_shows_up_in_lp_html(self):
        self.register_html_field_renderer("a_field")
        with self.resource() as resource:
            json = simplejson.loads(resource.do_GET())
            html = json['lp_html']
            self.assertEqual(
                html['a_field'], simple_renderer(resource.entry.a_field))
            self.assertEqual(
                resource.request.response.getHeader("Content-Type"),
                resource.JSON_PLUS_XHTML_TYPE)

    def test_html_renderer_for_class_renders_all_fields_of_that_class(self):
        self.register_html_field_renderer()
        with self.resource() as resource:
            json = simplejson.loads(resource.do_GET())
            html = json['lp_html']
            self.assertEqual(
                html['a_field'], simple_renderer(resource.entry.a_field))
            self.assertEqual(
                html['another_field'],
                simple_renderer(resource.entry.another_field))

    def test_json_plus_html_etag_is_json_etag(self):
        self.register_html_field_renderer()
        with self.resource() as resource:
            etag_1 = resource.getETag(resource.JSON_TYPE)
            etag_2 = resource.getETag(resource.JSON_PLUS_XHTML_TYPE)
            self.assertEqual(etag_1, etag_2)

    def test_acceptchanges_ignores_lp_html_for_json_plus_html_type(self):
        # The lp_html portion of the representation is ignored during
        # writes.
        self.register_html_field_renderer()
        json = None
        with self.resource() as resource:
            json_plus_xhtml = resource.JSON_PLUS_XHTML_TYPE
            json = simplejson.loads(six.text_type(
                    resource._representation(json_plus_xhtml)))
            resource.applyChanges(json, json_plus_xhtml)
            self.assertEqual(resource.request.response.getStatus(), 209)

    def test_acceptchanges_does_not_ignore_lp_html_for_bare_json_type(self):
        self.register_html_field_renderer()
        json = None
        with self.resource() as resource:
            json = simplejson.loads(six.text_type(
                    resource._representation(resource.JSON_PLUS_XHTML_TYPE)))
            resource.applyChanges(json, resource.JSON_TYPE)
            self.assertEqual(resource.request.response.getStatus(), 400)


class UnicodeChoice(EnumeratedType):
    """A choice between an ASCII value and a Unicode value."""
    ASCII = Item("Ascii", "Ascii choice")
    UNICODE = Item(u"Uni\u00e7ode", "Uni\u00e7ode choice")


@exported_as_webservice_entry()
class ICanBeSetToUnicodeValue(Interface):
    """An entry with an InterfaceRestrictedField."""
    a_field = exported(Choice(
        vocabulary=UnicodeChoice,
        title=u"A value that might be ASCII or Unicode.",
        required=False, default=None))


@implementer(ICanBeSetToUnicodeValue)
class CanBeSetToUnicodeValue:
    """An implementation of ICanBeSetToUnicodeValue."""
    def __init__(self, value):
        self.a_field = value


class UnicodeErrorTestCase(EntryTestCase):
    """Test that Unicode error strings are properly passed through."""

    testmodule_objects = [CanBeSetToUnicodeValue, ICanBeSetToUnicodeValue]

    def setUp(self):
        super(UnicodeErrorTestCase, self).setUp()
        self._register_url_adapter(ICanBeSetToUnicodeValue)

    def test_unicode_error(self):
        with self.entry_resource(
            ICanBeSetToUnicodeValue, CanBeSetToUnicodeValue, "") as resource:

            # This will raise an exception, which will cause the request
            # to fail with a 400 error code.
            error = resource.applyChanges({'a_field': u'No such value'})
            self.assertEqual(resource.request.response.getStatus(), 400)

            # The error message is a Unicode string which mentions both
            # the ASCII value and the Unicode value,
            expected_error = (
                u'a_field: Invalid value "No such value". Acceptable values '
                u'are: Ascii, Uni\u00e7ode')
            self.assertEqual(error, expected_error)


class WadlAPITestCase(WebServiceTestCase):
    """Test the docstring generation."""

    @exported_as_webservice_entry()
    class IScopedEntry(Interface):
        @scoped('test-scope')
        @export_read_operation()
        def test():
            """A method with a scope."""

    # This one is used to test when docstrings are missing.
    @exported_as_webservice_entry()
    class IUndocumentedEntry(Interface):
        a_field = exported(TextLine())

    testmodule_objects = [
        IGenericEntry, IGenericCollection, IScopedEntry, IUndocumentedEntry]

    def test_wadl_field_type(self):
        """Test the generated XSD field types for various fields."""
        self.assertEqual(test_tales("field/wadl:type", field=TextLine()),
                          None)
        self.assertEqual(test_tales("field/wadl:type", field=Date()),
                          "xsd:date")
        self.assertEqual(test_tales("field/wadl:type", field=Datetime()),
                          "xsd:dateTime")

    def test_wadl_entry_doc(self):
        """Test the wadl:doc generated for an entry adapter."""
        entry = get_resource_factory(IGenericEntry, IEntry)
        doclines = test_tales(
            'entry/wadl_entry:doc', entry=entry).splitlines()
        self.assertEqual([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>A simple, reusable entry interface for use in tests.</p>',
            '<p>The entry publishes one field and one named operation.</p>',
            '',
            '</wadl:doc>'], doclines)

    def test_empty_wadl_entry_doc(self):
        """Test that no docstring on an entry results in no wadl:doc."""
        entry = get_resource_factory(self.IUndocumentedEntry, IEntry)
        self.assertEqual(
            None, test_tales('entry/wadl_entry:doc', entry=entry))

    def test_wadl_collection_doc(self):
        """Test the wadl:doc generated for a collection adapter."""
        collection = get_resource_factory(IGenericCollection, ICollection)
        doclines = test_tales(
            'collection/wadl_collection:doc', collection=collection
            ).splitlines()
        self.assertEqual([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            'A simple collection containing IGenericEntry, for use in tests.',
            '</wadl:doc>'], doclines)

    def test_field_wadl_doc (self):
        """Test the wadl:doc generated for an exported field."""
        entry = get_resource_factory(IGenericEntry, IEntry)
        field = entry.schema['a_field']
        doclines = test_tales(
            'field/wadl:doc', field=field).splitlines()
        self.assertEqual([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>A &quot;field&quot;</p>',
            '<p>The only field that can be &lt;&gt; 0 in the entry.</p>',
            '',
            '</wadl:doc>'], doclines)

    def test_field_empty_wadl_doc(self):
        """Test that no docstring on a collection results in no wadl:doc."""
        entry = get_resource_factory(self.IUndocumentedEntry, IEntry)
        field = entry.schema['a_field']
        self.assertEqual(None, test_tales('field/wadl:doc', field=field))

    def test_wadl_operation_doc(self):
        """Test the wadl:doc generated for an operation adapter."""
        operation = get_operation_factory(IGenericEntry, 'greet')
        doclines = test_tales(
            'operation/wadl_operation:doc', operation=operation).splitlines()
        # Only compare the first 2 lines and the last one.
        # we dont care about the formatting of the parameters table.
        self.assertEqual([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>Print an appropriate greeting based on the message.</p>',],
            doclines[0:2])
        self.assertEqual('</wadl:doc>', doclines[-1])
        self.assertTrue(len(doclines) > 3,
            'Missing the parameter table: %s' % "\n".join(doclines))

    def test_wadl_operation_with_scopes_doc(self):
        """Test the wadl:doc generated for an operation adapter."""
        operation = get_operation_factory(self.IScopedEntry, 'test')
        doclines = test_tales(
            'operation/wadl_operation:doc', operation=operation).splitlines()
        # Only compare the first three lines and the last one.
        # we dont care about the formatting of the parameters table.
        self.assertEqual([
            '<wadl:doc xmlns="http://www.w3.org/1999/xhtml">',
            '<p>A method with a scope.</p>',
            '<p>Scopes: <tt class="rst-docutils literal"><span class="pre">'
            'test-scope</span></tt></p>',
            ], doclines[0:3])
        self.assertEqual('</wadl:doc>', doclines[-1])
        self.assertTrue(len(doclines) > 3,
            'Missing the parameter table: %s' % "\n".join(doclines))


class DuplicateNameTestCase(WebServiceTestCase):
    """Test AssertionError when two resources expose the same name.

    This class contains no tests of its own. It's up to the subclass
    to define IDuplicate and call doDuplicateTest().
    """

    def doDuplicateTest(self, expected_error_message):
        """Try to generate a WADL representation of the root.

        This will fail due to a name conflict.
        """
        resource = getUtility(IServiceRootResource)
        request = create_web_service_request('/2.0')
        request.traverse(resource)
        try:
            resource.toWADL()
            self.fail('Expected toWADL to fail with an AssertionError')
        except AssertionError as e:
            self.assertEqual(str(e), expected_error_message)


def make_entry(name):
    """Make an entity with some attibutes to expose as a web service."""
    fields = {
        '%s_field' % letter: exported(TextLine(title=u'Field %s' % letter))
        for letter in 'rstuvwxyz'}
    cls = InterfaceClass(name, bases=(Interface,), attrs=fields)
    return exported_as_webservice_entry(singular_name=name)(cls)


class TestWadlDeterminism(WebServiceTestCase):
    """We want the WADL generation to be consistent for a given input."""

    def __init__(self, *args, **kwargs):
        # make some -- randomly ordered -- objects to use to build the WADL
        self.testmodule_objects = [make_entry(name) for name in 'abcdefghijk']
        random.shuffle(self.testmodule_objects)
        super(TestWadlDeterminism, self).__init__(*args, **kwargs)

    @property
    def wadl(self):
        resource = getUtility(IServiceRootResource)
        request = create_web_service_request('/2.0')
        request.traverse(resource)
        return resource.toWADL()

    def test_entity_order(self):
        # The entities should be listed in alphabetical order by class.
        self.assertEqual(
            re.findall(r'<wadl:resource_type id="(.)">', self.wadl),
            ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k'])

    def test_attribute_order(self):
        # The individual entity attributes should be listed in alphabetical
        # order by class.
        self.assertEqual(
            re.findall(
                r'<wadl:param (?:[^>]* )*name="(.)_field"(?: [^>]*)*>',
                self.wadl)[:9],
            ['r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'])


class DuplicateSingularNameTestCase(DuplicateNameTestCase):
    """Test AssertionError when resource types share a singular name."""

    @exported_as_webservice_entry('generic_entry')
    class IDuplicate(Interface):
        """An entry that reuses the singular name of IGenericEntry."""

    testmodule_objects = [IGenericEntry, IDuplicate]

    def test_duplicate_singular(self):
        self.doDuplicateTest("Both IDuplicate and IGenericEntry expose the "
                             "singular name 'generic_entry'.")


class DuplicatePluralNameTestCase(DuplicateNameTestCase):
    """Test AssertionERror when resource types share a plural name."""

    @exported_as_webservice_entry(plural_name='generic_entrys')
    class IDuplicate(Interface):
        """An entry that reuses the plural name of IGenericEntry."""

    testmodule_objects = [IGenericEntry, IDuplicate]

    def test_duplicate_plural(self):
        self.doDuplicateTest("Both IDuplicate and IGenericEntry expose the "
                             "plural name 'generic_entrys'.")


class GetCurrentWebServiceRequestTestCase(WebServiceTestCase):
    """Test the get_current_web_service_request utility function."""

    testmodule_objects = [IGenericEntry]

    def test_web_service_request_is_versioned(self):
        """Ensure the get_current_web_service_request() result is versioned.

        When a normal browser request is turned into a web service
        request, it needs to have a version associated with it.
        lazr.restful associates the new request with the latest
        version of the web service: in this case, version 2.0.
        """

        # When there's no interaction setup, get_current_web_service_request()
        # returns None.
        self.assertEqual(None, queryInteraction())
        self.assertEqual(None, get_current_web_service_request())

        # Set up an interaction.
        request = TestRequest()
        newInteraction(request)

        # A normal web browser request isn't associated with any version.
        website_request = get_current_browser_request()
        self.assertRaises(AttributeError, getattr, website_request, 'version')

        # But the result of get_current_web_service_request() is
        # associated with version 2.0.
        webservice_request = get_current_web_service_request()
        self.assertEqual("2.0", webservice_request.version)
        marker_20 = getUtility(IWebServiceVersion, "2.0")
        self.assertTrue(marker_20.providedBy(webservice_request))

        # We can use tag_request_with_version_name to change the
        # version of a request object.
        tag_request_with_version_name(webservice_request, '1.0')
        self.assertEqual("1.0", webservice_request.version)
        marker_10 = getUtility(IWebServiceVersion, "1.0")
        self.assertTrue(marker_10.providedBy(webservice_request))

        tag_request_with_version_name(webservice_request, '2.0')
        self.assertEqual("2.0", webservice_request.version)
        self.assertTrue(marker_20.providedBy(webservice_request))

        endInteraction()


@exported_as_webservice_entry()
class ITestEntry(IEntry):
    """Interface for a test entry."""


@implementer(ITestEntry)
class TestEntry:
    def __init__(self, context, request):
        pass


class BaseBatchingTest:
    """A base class which tests BatchingResourceMixin and subclasses."""

    testmodule_objects = [HasRestrictedField, IHasRestrictedField]

    def setUp(self):
        super(BaseBatchingTest, self).setUp()
        # Register TestEntry as the IEntry implementation for ITestEntry.
        getGlobalSiteManager().registerAdapter(
            TestEntry, [ITestEntry, IWebServiceClientRequest], provided=IEntry)
        # Is doing this by hand the right way?
        ITestEntry.setTaggedValue(
            LAZR_WEBSERVICE_NAME,
            dict(singular='test_entity', plural='test_entities'))


    def make_instance(self, entries, request):
        raise NotImplementedError('You have to make your own instances.')

    def test_getting_a_batch(self):
        entries = [1, 2, 3]
        request = create_web_service_request('/devel')
        instance = self.make_instance(entries, request)
        total_size = instance.get_total_size(entries)
        self.assertEqual(total_size, '3')


class TestBatchingResourceMixin(BaseBatchingTest, WebServiceTestCase):
    """Test that BatchingResourceMixin does batching correctly."""

    def make_instance(self, entries, request):
        return BatchingResourceMixin()


class TestCollectionResourceBatching(BaseBatchingTest, WebServiceTestCase):
    """Test that CollectionResource does batching correctly."""

    def make_instance(self, entries, request):
        @implementer(ICollection)
        class Collection:
            entry_schema = ITestEntry

            def __init__(self, entries):
                self.entries = entries

            def find(self):
                return self.entries

        return CollectionResource(Collection(entries), request)


class TestResourceOperationBatching(BaseBatchingTest, WebServiceTestCase):
    """Test that ResourceOperation does batching correctly."""

    def make_instance(self, entries, request):
        # constructor parameters are ignored
        return ResourceOperation(None, request)


Notification = collections.namedtuple('Notification', ['level', 'message'])


class NotificationsProviderTest(EntryTestCase):
    """Test that notifcations are included in the response headers."""

    testmodule_objects = [HasOneField, IHasOneField]

    @implementer(INotificationsProvider)
    class DummyWebsiteRequestWithNotifications:
        """A request to the website, as opposed to the web service."""

        @property
        def notifications(self):
            return [Notification(logging.INFO, "Informational"),
                    Notification(logging.WARNING, "Warning")
                    ]
    def setUp(self):
        super(NotificationsProviderTest, self).setUp()
        self.default_media_type = "application/json;include=lp_html"
        self._register_website_url_space(IHasOneField)
        self._register_notification_adapter()

    def _register_notification_adapter(self):
        """Simulates a service where an entry corresponds to a web page."""

        # First, create a converter from web service requests to
        # web service requests with notifications.
        def web_service_request_to_notification_request(service_request):
            """Create a corresponding request to the website."""
            return self.DummyWebsiteRequestWithNotifications()

        getGlobalSiteManager().registerAdapter(
            web_service_request_to_notification_request,
            [IWebServiceClientRequest], INotificationsProvider)

    @contextmanager
    def resource(self):
        """Simplify the entry_resource call."""
        with self.entry_resource(IHasOneField, HasOneField, "") as resource:
            yield resource

    def test_response_notifications(self):
        with self.resource() as resource:
            resource.request.publication.callObject(
                resource.request, resource)
            notifications = resource.request.response.getHeader(
                "X-Lazr-Notifications")
            self.assertFalse(notifications is None)
            notifications = simplejson.loads(notifications)
            expected_notifications = [
                [logging.INFO, "Informational"], [logging.WARNING, "Warning"]]
            self.assertEqual(notifications, expected_notifications)

class EventTestCase(EntryTestCase):

    testmodule_objects = [IHasOneField]

    def setUp(self):
        super(EventTestCase, self).setUp()
        self._register_url_adapter(IHasOneField)
        eventtesting.setUp()

    def test_event_fired_when_changeset_is_not_empty(self):
        # Passing in a non-empty changeset spawns an
        # IObjectModifiedEvent.
        with self.entry_resource(
                IHasOneField, HasOneField, "") as resource:
            resource.applyChanges({'a_field': u'Some value'})
        events = eventtesting.getEvents()
        self.assertEqual(len(events), 1)
        event = events[0]
        self.assertEqual(event.object_before_modification.a_field, "")
        self.assertEqual(event.object.a_field, "Some value")

    def test_event_not_fired_when_changeset_is_empty(self):
        # Passing in an empty changeset does not spawn an
        # IObjectModifiedEvent.
        with self.entry_resource(
                IHasOneField, HasOneField, "") as resource:
            resource.applyChanges({})
        self.assertEqual(len(eventtesting.getEvents()), 0)


class MalformedRequest(EntryTestCase):

    testmodule_objects = [HasOneField, IHasOneField]
    default_media_type = "application/xhtml+xml"

    def setUp(self):
        super(MalformedRequest, self).setUp()
        self._register_url_adapter(IHasOneField)
        self.unicode_message = u"Hello from a \N{SNOWMAN}"

    def test_multiple_named_operations_generate_error_on_GET(self):
        with self.entry_resource(
                IHasOneField, HasOneField, self.unicode_message) as resource:
            resource.request.form['ws.op'] = ['foo', 'bar']
            result = resource.do_GET()
        self.assertEqual(resource.request.response.getStatus(), 400)
        self.assertEqual(
            result, "Expected a single operation: ['foo', 'bar']")

    def test_multiple_named_operations_generate_error_on_POST(self):
        with self.entry_resource(
                IHasOneField, HasOneField, self.unicode_message) as resource:
            resource.request.form['ws.op'] = ['foo', 'bar']
            result = resource.do_POST()
        self.assertEqual(resource.request.response.getStatus(), 400)
        self.assertEqual(
            result, "Expected a single operation: ['foo', 'bar']")
