import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { Duration, IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Endpoint } from './endpoint';
import { InstanceType } from './instance';
import { IClusterParameterGroup, IParameterGroup } from './parameter-group';
import { ISubnetGroup } from './subnet-group';
/**
 * Possible Instances Types to use in Neptune cluster
 * used for defining {@link DatabaseClusterProps.engineVersion}.
 */
export declare class EngineVersion {
    readonly version: string;
    /**
     * Neptune engine version 1.0.1.0
     */
    static readonly V1_0_1_0: EngineVersion;
    /**
     * Neptune engine version 1.0.1.1
     */
    static readonly V1_0_1_1: EngineVersion;
    /**
     * Neptune engine version 1.0.1.2
     */
    static readonly V1_0_1_2: EngineVersion;
    /**
     * Neptune engine version 1.0.2.1
     */
    static readonly V1_0_2_1: EngineVersion;
    /**
     * Neptune engine version 1.0.2.2
     */
    static readonly V1_0_2_2: EngineVersion;
    /**
     * Neptune engine version 1.0.3.0
     */
    static readonly V1_0_3_0: EngineVersion;
    /**
     * Neptune engine version 1.0.4.0
     */
    static readonly V1_0_4_0: EngineVersion;
    /**
     * Neptune engine version 1.0.4.1
     */
    static readonly V1_0_4_1: EngineVersion;
    /**
     * Neptune engine version 1.0.5.0
     */
    static readonly V1_0_5_0: EngineVersion;
    /**
     * Neptune engine version 1.1.0.0
     */
    static readonly V1_1_0_0: EngineVersion;
    /**
     * Constructor for specifying a custom engine version
     * @param version the engine version of Neptune
     */
    constructor(version: string);
}
/**
 * Properties for a new database cluster
 */
export interface DatabaseClusterProps {
    /**
     * What version of the database to start
     *
     * @default - The default engine version.
     */
    readonly engineVersion?: EngineVersion;
    /**
     * The port the Neptune cluster will listen on
     *
     * @default - The default engine port
     */
    readonly port?: number;
    /**
     * How many days to retain the backup
     *
     * @default - cdk.Duration.days(1)
     */
    readonly backupRetention?: Duration;
    /**
     * A daily time range in 24-hours UTC format in which backups preferably execute.
     *
     * Must be at least 30 minutes long.
     *
     * Example: '01:00-02:00'
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region. To see the time blocks available, see
     */
    readonly preferredBackupWindow?: string;
    /**
     * The KMS key for storage encryption.
     *
     * @default - default master key.
     */
    readonly kmsKey?: kms.IKey;
    /**
     * Whether to enable storage encryption
     *
     * @default true
     */
    readonly storageEncrypted?: boolean;
    /**
     * Number of Neptune compute instances
     *
     * @default 1
     */
    readonly instances?: number;
    /**
     * An optional identifier for the cluster
     *
     * @default - A name is automatically generated.
     */
    readonly dbClusterName?: string;
    /**
     * Map AWS Identity and Access Management (IAM) accounts to database accounts
     *
     * @default - `false`
     */
    readonly iamAuthentication?: boolean;
    /**
     * Base identifier for instances
     *
     * Every replica is named by appending the replica number to this string, 1-based.
     *
     * @default - `dbClusterName` is used with the word "Instance" appended. If `dbClusterName` is not provided, the
     * identifier is automatically generated.
     */
    readonly instanceIdentifierBase?: string;
    /**
     * What type of instance to start for the replicas
     */
    readonly instanceType: InstanceType;
    /**
     * A list of AWS Identity and Access Management (IAM) role that can be used by the cluster to access other AWS services.
     *
     * @default - No role is attached to the cluster.
     */
    readonly associatedRoles?: iam.IRole[];
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if ``removalPolicy`` is RETAIN, false otherwise
     */
    readonly deletionProtection?: boolean;
    /**
     * A weekly time range in which maintenance should preferably execute.
     *
     * Must be at least 30 minutes long.
     *
     * Example: 'tue:04:17-tue:04:47'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * Additional parameters to pass to the database engine
     *
     * @default - No parameter group.
     */
    readonly clusterParameterGroup?: IClusterParameterGroup;
    /**
     * The DB parameter group to associate with the instance.
     *
     * @default no parameter group
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     */
    readonly subnetGroup?: ISubnetGroup;
    /**
     * What subnets to run the Neptune instances in.
     *
     * Must be at least 2 subnets in two different AZs.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the instances within the VPC
     *
     * @default private subnets
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Security group.
     *
     * @default a new security group is created.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The removal policy to apply when the cluster and its instances are removed
     * or replaced during a stack update, or when the stack is deleted. This
     * removal policy also applies to the implicit security group created for the
     * cluster if one is not supplied as a parameter.
     *
     * @default - Retain cluster.
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * If set to true, Neptune will automatically update the engine of the entire
     * cluster to the latest minor version after a stabilization window of 2 to 3 weeks.
     *
     * @default - false
     */
    readonly autoMinorVersionUpgrade?: boolean;
}
/**
 * Create a clustered database with a given number of instances.
 */
export interface IDatabaseCluster extends IResource, ec2.IConnectable {
    /**
     * Identifier of the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * Resource identifier of the cluster
     * @attribute ClusterResourceId
     */
    readonly clusterResourceIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     * @attribute Endpoint,Port
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     * @attribute ReadEndpoint
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Grant the given identity connection access to the database.
     */
    grantConnect(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Properties that describe an existing cluster instance
 */
export interface DatabaseClusterAttributes {
    /**
     * The database port
     */
    readonly port: number;
    /**
     * The security group of the database cluster
     */
    readonly securityGroup: ec2.ISecurityGroup;
    /**
     * Identifier for the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * Resource Identifier for the cluster
     */
    readonly clusterResourceIdentifier: string;
    /**
     * Cluster endpoint address
     */
    readonly clusterEndpointAddress: string;
    /**
     * Reader endpoint address
     */
    readonly readerEndpointAddress: string;
}
/**
 * A new or imported database cluster.
 */
export declare abstract class DatabaseClusterBase extends Resource implements IDatabaseCluster {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope: Construct, id: string, attrs: DatabaseClusterAttributes): IDatabaseCluster;
    /**
     * Identifier of the cluster
     */
    abstract readonly clusterIdentifier: string;
    /**
     * Resource identifier of the cluster
     */
    abstract readonly clusterResourceIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * The connections object to implement IConnectable
     */
    abstract readonly connections: ec2.Connections;
    protected abstract enableIamAuthentication?: boolean;
    grantConnect(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::Neptune::DBCluster
 */
export declare class DatabaseCluster extends DatabaseClusterBase implements IDatabaseCluster {
    /**
     * The default number of instances in the Neptune cluster if none are
     * specified
     */
    static readonly DEFAULT_NUM_INSTANCES = 1;
    readonly clusterIdentifier: string;
    readonly clusterEndpoint: Endpoint;
    readonly clusterReadEndpoint: Endpoint;
    readonly connections: ec2.Connections;
    /**
     * The resource id for the cluster; for example: cluster-ABCD1234EFGH5678IJKL90MNOP. The cluster ID uniquely
     * identifies the cluster and is used in things like IAM authentication policies.
     * @attribute ClusterResourceId
     */
    readonly clusterResourceIdentifier: string;
    /**
     * The VPC where the DB subnet group is created.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The subnets used by the DB subnet group.
     */
    readonly vpcSubnets: ec2.SubnetSelection;
    /**
     * Subnet group used by the DB
     */
    readonly subnetGroup: ISubnetGroup;
    /**
     * Identifiers of the instance
     */
    readonly instanceIdentifiers: string[];
    /**
     * Endpoints which address each individual instance.
     */
    readonly instanceEndpoints: Endpoint[];
    protected enableIamAuthentication?: boolean;
    constructor(scope: Construct, id: string, props: DatabaseClusterProps);
}
