"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FsxLifecycleStatusMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
class FsxLifecycleStatusMonitor extends constructs_1.Construct {
    /**
     * Creates an instance of FsxLifecycleStatusMonitor.
     * @param {Construct} scope - parent construct
     * @param {string} id - unique id
     * @memberof FsxLifecycleStatusMonitor - class instance
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.topic = this.createSNSTopic();
        this.policy = this.createIamPolicy();
        this.fn = this.createLambdaFunction();
        this.fn.role?.attachInlinePolicy(this.policy);
        this.logGroup = new logs.LogGroup(this, 'fsx-health-loggroup', {
            logGroupName: this.fn.functionName,
            retention: props.logRetentionDays ??
                FsxLifecycleStatusMonitor.DEFAULT_LOG_RETENTION_PERIOD,
        });
        this.policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [this.logGroup.logGroupArn],
        }));
        this.rule = new events.Rule(this, 'rule', {
            ruleName: 'fsx-health-trigger',
            description: 'Trigger the FSx health check based on the underlying cron expression',
            schedule: props.schedule ?? FsxLifecycleStatusMonitor.DEFAULT_SCHEDULE,
            targets: [new targets.LambdaFunction(this.fn)],
        });
    }
    /**
     * Topic linked to the Lambda function.
     *
     * @return {*}  {sns.Topic} - sns topic
     * @memberof FsxLifecycleStatusMonitor - class instance
     */
    createSNSTopic() {
        return new sns.Topic(this, 'fsx-health-sns-topic', {
            displayName: 'fsx-lifecycle-monitor',
            topicName: 'fsx-lifecycle-monitor',
            masterKey: kms.Alias.fromAliasName(this, 'alias', 'alias/aws/sns'),
        });
    }
    createIamPolicy() {
        return new iam.Policy(this, 'policy', {
            policyName: 'fsx-lambda-sns',
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['fsx:DescribeFileSystems'],
                    resources: ['*'],
                }),
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['sns:Publish'],
                    resources: ['*'],
                }),
            ],
        });
    }
    /**
     *
     *
     * @return {*}  {lambda.Function}
     * @memberof FsxLifecycleStatusMonitor
     */
    createLambdaFunction() {
        return new lambda.Function(this, 'fsx-lifecycle-monitor', {
            runtime: lambda.Runtime.PYTHON_3_10,
            handler: 'fsx-health.lambda_handler',
            architecture: lambda.Architecture.X86_64,
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda')),
            insightsVersion: lambda.LambdaInsightsVersion.VERSION_1_0_229_0,
            environment: {
                LambdaSNSTopic: this.topic.topicArn,
            },
        });
    }
}
exports.FsxLifecycleStatusMonitor = FsxLifecycleStatusMonitor;
_a = JSII_RTTI_SYMBOL_1;
FsxLifecycleStatusMonitor[_a] = { fqn: "aws-fsx-lifecycle-status-monitor.FsxLifecycleStatusMonitor", version: "0.0.32" };
/**
 * Default schedule for the FSx Lifecycle Status Monitor.
 *
 * @static
 * @memberof FsxLifecycleStatusMonitor
 */
FsxLifecycleStatusMonitor.DEFAULT_SCHEDULE = events.Schedule.cron({
    minute: '0/10',
    hour: '*',
    day: '*',
    month: '*',
    year: '*',
});
/**
 * Default log retention for the FSx Lifecycle Status Monitor.
 *
 * @static
 * @memberof FsxLifecycleStatusMonitor
 */
FsxLifecycleStatusMonitor.DEFAULT_LOG_RETENTION_PERIOD = logs.RetentionDays.ONE_YEAR;
//# sourceMappingURL=data:application/json;base64,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