#!/usr/bin/env python3
"""
Script to export the metadata upgrade status table from Redshift to CSV.
This script pulls the table generated by the sync process and saves it locally.
"""

import logging
from datetime import datetime
from aind_data_access_api.rds_tables import RDSCredentials, Client

# Configure logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)

# Redshift settings (same as sync.py)
REDSHIFT_SECRETS = "/aind/prod/redshift/credentials/readwrite"
RDS_TABLE_NAME = "metadata_upgrade_status_prod"


def export_table_to_csv():
    """Export the Redshift table to CSV"""

    try:
        # Initialize RDS client
        logger.info("Connecting to Redshift...")
        rds_client = Client(
            credentials=RDSCredentials(aws_secrets_name=REDSHIFT_SECRETS),
        )

        # Read the table
        logger.info(f"Reading table '{RDS_TABLE_NAME}' from Redshift...")
        df = rds_client.read_table(RDS_TABLE_NAME)

        if df is None or len(df) == 0:
            logger.warning("Table is empty or could not be read")
            return

        logger.info(f"Successfully read {len(df)} rows from table")
        logger.info(f"Table columns: {list(df.columns)}")

        # Generate filename with timestamp
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        filename = f"metadata_upgrade_status_{timestamp}.csv"

        # Save to CSV
        logger.info(f"Saving to CSV file: {filename}")
        df.to_csv(filename, index=False)

        logger.info(f"Successfully exported {len(df)} rows to {filename}")

        # Print summary statistics
        logger.info("Summary statistics:")
        if "status" in df.columns:
            status_counts = df["status"].value_counts()
            for status, count in status_counts.items():
                logger.info(f"  {status}: {count}")

        if "upgrader_version" in df.columns:
            version_counts = df["upgrader_version"].value_counts()
            logger.info("Upgrader versions:")
            for version, count in version_counts.items():
                logger.info(f"  {version}: {count}")

    except Exception as e:
        logger.error(f"Error exporting table: {e}")
        raise


if __name__ == "__main__":
    export_table_to_csv()
