import * as glue from '@aws-cdk/aws-glue-alpha';
import * as athena from 'aws-cdk-lib/aws-athena';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { DataProduct } from './data-product';
import { DataSet } from './data-sets/data-set';
import { KinesisStream } from './data-streams/kinesis-stream';
/**
 * @stability stable
 */
export interface CrossAccountProperties {
    /**
     * @stability stable
     */
    readonly consumerAccountIds: string[];
    /**
     * @stability stable
     */
    readonly dataCatalogOwnerAccountId: string;
}
/**
 * @stability stable
 */
export interface DataLakeProperties {
    /**
     * The name of the DataLake.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The Stage the DataLake will be deployed.
     *
     * @stability stable
     */
    readonly stageName: Stage;
    /**
     * The List of DataProducts for this account.
     *
     * @default - No data products
     * @stability stable
     */
    readonly dataProducts?: DataProduct[];
    /**
     * The Type of DataLake this instance is.
     *
     * This can be a DATA_PRODUCT only, CENTRAL_CATALOG, CONSUMER, or DATA_PRODUCT_AND_CATALOG type.
     *
     * @stability stable
     */
    readonly lakeType: LakeType;
    /**
     * VPC for Glue jobs.
     *
     * @default - No vpc
     * @stability stable
     * @description - The VPC that will be used if the Glue job needs access to resources within the account or internet access
     */
    readonly vpc?: ec2.Vpc;
    /**
     * List of Lake Formation TBAC policy tags.
     *
     * @default - No tags
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/TBAC-section.html
     * @stability stable
     * @description - Define the tag taxonomy needed for the DataLake
     */
    readonly policyTags?: {
        [name: string]: string;
    };
    /**
     * Cross account AWS account IDs.
     *
     * @default - No cross account ids
     * @see https://aws.amazon.com/premiumsupport/knowledge-center/glue-data-catalog-cross-account-access/
     * @stability stable
     * @description - The cross account ids needed for setting up the Glue resource policy
     */
    readonly crossAccountAccess?: CrossAccountProperties;
    /**
     * Security group to attach to Glue jobs.
     *
     * @default - No security group
     * @see https://docs.aws.amazon.com/glue/latest/dg/setup-vpc-for-glue-access.html
     * @stability stable
     * @description - Security Group that will be used to allow port access in the VPC
     */
    readonly glueSecurityGroup?: ec2.SecurityGroup;
    /**
     * Data Lake Admin role.
     *
     * @default - Admin role created based on best practices
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/permissions-reference.html
     * @stability stable
     * @description - IAM Role for DataLake admin access
     */
    readonly datalakeAdminRole?: iam.Role;
    /**
     * Data Lake Database Creator role.
     *
     * @default - Database creator role created based on best practices
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/permissions-reference.html
     * @stability stable
     * @description - IAM Role for DataLake database creator access
     */
    readonly datalakeCreatorRole?: iam.Role;
    /**
     * Create default Glue Database for DataLake.
     *
     * @default - false
     * @stability stable
     */
    readonly createDefaultDatabase: Boolean;
    /**
     * @stability stable
     */
    readonly logBucketProps?: s3.BucketProps;
    /**
     * Create default Athena workgroup for querying data lake resources.
     *
     * @default - false
     * @stability stable
     */
    readonly createAthenaWorkgroup?: Boolean;
}
/**
 * @stability stable
 */
export declare enum Stage {
    /**
     * @stability stable
     */
    ALPHA = "alpha",
    /**
     * @stability stable
     */
    BETA = "beta",
    /**
     * @stability stable
     */
    GAMMA = "gamma",
    /**
     * @stability stable
     */
    PROD = "prod"
}
/**
 * @stability stable
 */
export declare enum Permissions {
    /**
     * @stability stable
     */
    ALTER = "ALTER",
    /**
     * @stability stable
     */
    CREATE_DATABASE = "CREATE_DATABASE",
    /**
     * @stability stable
     */
    CREATE_TABLE = "CREATE_TABLE",
    /**
     * @stability stable
     */
    DATA_LOCATION_ACCESS = "DATA_LOCATION_ACCESS",
    /**
     * @stability stable
     */
    DELETE = "DELETE",
    /**
     * @stability stable
     */
    DESCRIBE = "DESCRIBE",
    /**
     * @stability stable
     */
    DROP = "DROP",
    /**
     * @stability stable
     */
    INSERT = "INSERT",
    /**
     * @stability stable
     */
    SELECT = "SELECT",
    /**
     * @stability stable
     */
    ASSOCIATE = "ASSOCIATE"
}
/**
 * @stability stable
 */
export declare enum LakeType {
    /**
     * @stability stable
     */
    DATA_PRODUCT = "DATA_PRODUCT",
    /**
     * @stability stable
     */
    CENTRAL_CATALOG = "CENTRAL_CATALOG",
    /**
     * @stability stable
     */
    CONSUMER = "CONSUMER",
    /**
     * @stability stable
     */
    DATA_PRODUCT_AND_CATALOG = "DATA_PRODUCT_AND_CATALOG"
}
/**
 * A CDK construct to create a DataLake.
 *
 * @stability stable
 */
export declare class DataLake extends Construct {
    /**
     * @stability stable
     */
    readonly dataSets: {
        [schemaName: string]: DataSet;
    };
    /**
     * @stability stable
     */
    readonly dataStreams: {
        [schemaName: string]: KinesisStream;
    };
    /**
     * @stability stable
     */
    readonly databases: {
        [name: string]: glue.Database;
    };
    /**
     * @stability stable
     */
    readonly datalakeAdminRole: iam.IRole;
    /**
     * @stability stable
     */
    readonly datalakeDbCreatorRole: iam.IRole;
    /**
     * @stability stable
     */
    readonly logBucket: s3.Bucket;
    /**
     * @stability stable
     */
    readonly stageName: Stage;
    /**
     * @stability stable
     */
    readonly vpc?: ec2.Vpc;
    /**
     * @stability stable
     */
    readonly athenaWorkgroup?: athena.CfnWorkGroup;
    /**
     * @stability stable
     */
    readonly lakeType: LakeType;
    private readonly glueSecurityGroup?;
    private readonly crossAccountAccess?;
    private readonly downloadLocations;
    private readonly logBucketProps;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: DataLakeProperties);
    /**
     * @stability stable
     */
    createDownloaderCustomResource(stageName: string): void;
    private createDatabase;
    private addDataStream;
    private addPipeline;
    private createPipelineResources;
    private createJDBCConnection;
    private createPolicyTagsCustomResource;
    /**
     * @stability stable
     */
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds: string[], dataCatalogOwnerAccountId: string): void;
}
