"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLake = exports.LakeType = exports.Permissions = exports.Stage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const glue = require("@aws-cdk/aws-glue-alpha");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const athena = require("aws-cdk-lib/aws-athena");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lf = require("aws-cdk-lib/aws-lakeformation");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const data_set_1 = require("./data-sets/data-set");
const kinesis_ops_1 = require("./data-streams/kinesis-ops");
const kinesis_stream_1 = require("./data-streams/kinesis-stream");
const s3_delivery_stream_1 = require("./data-streams/s3-delivery-stream");
const glue_crawler_1 = require("./etl/glue-crawler");
const glue_job_1 = require("./etl/glue-job");
const glue_job_ops_1 = require("./etl/glue-job-ops");
const glue_table_1 = require("./etl/glue-table");
const data_lake_admin_1 = require("./personas/data-lake-admin");
const data_lake_creator_1 = require("./personas/data-lake-creator");
const pipeline_1 = require("./pipeline");
const utils_1 = require("./utils");
/**
 * @stability stable
 */
var Stage;
(function (Stage) {
    Stage["ALPHA"] = "alpha";
    Stage["BETA"] = "beta";
    Stage["GAMMA"] = "gamma";
    Stage["PROD"] = "prod";
})(Stage = exports.Stage || (exports.Stage = {}));
/**
 * @stability stable
 */
var Permissions;
(function (Permissions) {
    Permissions["ALTER"] = "ALTER";
    Permissions["CREATE_DATABASE"] = "CREATE_DATABASE";
    Permissions["CREATE_TABLE"] = "CREATE_TABLE";
    Permissions["DATA_LOCATION_ACCESS"] = "DATA_LOCATION_ACCESS";
    Permissions["DELETE"] = "DELETE";
    Permissions["DESCRIBE"] = "DESCRIBE";
    Permissions["DROP"] = "DROP";
    Permissions["INSERT"] = "INSERT";
    Permissions["SELECT"] = "SELECT";
    Permissions["ASSOCIATE"] = "ASSOCIATE";
})(Permissions = exports.Permissions || (exports.Permissions = {}));
/**
 * @stability stable
 */
var LakeType;
(function (LakeType) {
    LakeType["DATA_PRODUCT"] = "DATA_PRODUCT";
    LakeType["CENTRAL_CATALOG"] = "CENTRAL_CATALOG";
    LakeType["CONSUMER"] = "CONSUMER";
    LakeType["DATA_PRODUCT_AND_CATALOG"] = "DATA_PRODUCT_AND_CATALOG";
})(LakeType = exports.LakeType || (exports.LakeType = {}));
/**
 * A CDK construct to create a DataLake.
 *
 * @stability stable
 */
class DataLake extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.dataSets = {};
        /**
         * @stability stable
         */
        this.dataStreams = {};
        /**
         * @stability stable
         */
        this.databases = {};
        this.downloadLocations = {}; //used for the Custom Resource to allow downloading of existing datasets into datalake
        this.stageName = props.stageName;
        this.crossAccountAccess = props.crossAccountAccess ? props.crossAccountAccess : undefined;
        this.vpc = props.vpc ? props.vpc : undefined;
        this.lakeType = props.lakeType;
        if (props.logBucketProps) {
            this.logBucketProps = props.logBucketProps;
        }
        else {
            this.logBucketProps = {
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(30),
                    },
                ],
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: true,
            };
        }
        if (this.vpc) {
            const securityGroupName = utils_1.buildUniqueName({
                name: 'glue',
                resourceUse: 'datalake',
                stage: this.stageName,
            }, 80);
            this.glueSecurityGroup = new ec2.SecurityGroup(this, 'glue-sg', {
                description: 'Glue self referential allow in out',
                vpc: this.vpc,
                securityGroupName: securityGroupName,
            });
            this.glueSecurityGroup.connections.allowFrom(this.glueSecurityGroup, ec2.Port.allTcp());
            this.glueSecurityGroup.connections.allowTo(this.glueSecurityGroup, ec2.Port.allTcp());
            new aws_cdk_lib_1.CfnOutput(this, 'GlueSecurityGroupName', { value: securityGroupName });
        }
        // make this optional
        this.logBucket = new s3.Bucket(this, 'datalake-log-bucket', {
            bucketName: utils_1.buildS3BucketName({
                stage: props.stageName,
                resourceUse: 'log-bucket',
                name: props.name,
            }),
            ...this.logBucketProps,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'DataLakeLogBucket', { value: this.logBucket.bucketName });
        if (props.datalakeAdminRole) {
            this.datalakeAdminRole = props.datalakeAdminRole;
        }
        else {
            this.datalakeAdminRole = new data_lake_admin_1.DataLakeAdministrator(this, `${props.name}-datalake-admin-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-admin',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.datalakeCreatorRole) {
            this.datalakeDbCreatorRole = props.datalakeCreatorRole;
        }
        else {
            this.datalakeDbCreatorRole = new data_lake_creator_1.DataLakeCreator(this, `${props.name}-datalake-creator-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-creator',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.createDefaultDatabase) {
            this.databases[props.name] = this.createDatabase(props.name);
            new aws_cdk_lib_1.CfnOutput(this, 'DataLakeDefaultDatabase', { value: props.name });
        }
        if (this.crossAccountAccess) {
            this.createCrossAccountGlueCatalogResourcePolicy(this.crossAccountAccess.consumerAccountIds, this.crossAccountAccess.dataCatalogOwnerAccountId);
        }
        if (props.createAthenaWorkgroup) {
            this.athenaWorkgroup = new athena.CfnWorkGroup(this, 'workgroup', {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'workgroup',
                    stage: this.stageName,
                }, 60),
                description: 'Default Data Lake Workgroup',
                state: 'ENABLED',
                recursiveDeleteOption: true,
                workGroupConfiguration: {
                    enforceWorkGroupConfiguration: true,
                    resultConfiguration: {
                        outputLocation: `s3://${this.logBucket.bucketName}/results/`,
                    },
                    engineVersion: {
                        selectedEngineVersion: 'Athena engine version 2',
                        effectiveEngineVersion: 'Athena engine version 2',
                    },
                },
            });
            new aws_cdk_lib_1.CfnOutput(this, 'DataLakeAthenaWorkgroup', { value: this.athenaWorkgroup.name });
        }
        if (props.policyTags) {
            this.createPolicyTagsCustomResource(props.policyTags, this.datalakeAdminRole);
        }
        if (props.dataProducts && props.dataProducts.length > 0) {
            props.dataProducts.forEach((product) => {
                this.databases[product.databaseName] = this.createDatabase(product.databaseName);
                product.pipelines.forEach((pipe) => {
                    this.addPipeline(pipe, product);
                });
            });
        }
    }
    /**
     * @stability stable
     */
    createDownloaderCustomResource(stageName) {
        // download the data sets with the custom resource after successfull creation of resource
        const onEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'DataloaderHandler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/download-data'),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            functionName: utils_1.buildLambdaFunctionName({
                name: 'load-data',
                resourceUse: 'cr',
                stage: stageName,
            }),
        });
        // create readable and writable buckets for the datasets and set the appropriate S3 access
        onEvent.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: ['*'],
        }));
        const dataLoadProvider = new cr.Provider(this, 'DataloaderProvider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // CR to download the static datasets form the dataSets var passed in.
        new aws_cdk_lib_1.CustomResource(this, 'LoadDatalakeCustomResource', {
            serviceToken: dataLoadProvider.serviceToken,
            properties: {
                dataSets: this.downloadLocations,
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
            },
        });
    }
    createDatabase(databaseName) {
        const db = new glue.Database(this, `${databaseName}-database`, {
            databaseName: `${databaseName}`,
        });
        const dbPerm = new lf.CfnPermissions(this, `${databaseName}-lf-db-creator-permission`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.datalakeDbCreatorRole.roleArn,
            },
            resource: {
                databaseResource: {
                    name: databaseName,
                },
            },
            permissions: [
                Permissions.ALTER,
                Permissions.CREATE_TABLE,
                Permissions.DROP,
            ],
        });
        dbPerm.node.addDependency(db);
        return db;
    }
    addDataStream(pipeline, dataSet) {
        const schemaName = pipeline.name;
        const dataStreamStack = new aws_cdk_lib_1.NestedStack(aws_cdk_lib_1.Stack.of(this), `${schemaName}-datastream-stack`);
        if (!pipeline.streamProperties) {
            throw Error("Cannot create a stream pipeline without 'sreamProperties'");
        }
        this.dataStreams[schemaName] = new kinesis_stream_1.KinesisStream(dataStreamStack, 'DataStream', {
            shardCount: 1,
            streamName: pipeline.streamProperties.streamName,
        });
        const deliveryStream = new s3_delivery_stream_1.S3DeliveryStream(dataStreamStack, 'deliveryStream', {
            compression: s3_delivery_stream_1.CompressionType.UNCOMPRESSED,
            kinesisStream: this.dataStreams[schemaName].stream,
            s3Bucket: s3.Bucket.fromBucketName(this, 'get-bucket-for-kinesis', dataSet.getDataSetBucketName(pipeline.dataSetDropTier)),
            s3Prefix: pipeline.destinationPrefix,
        });
        new kinesis_ops_1.KinesisOps(dataStreamStack, 'kinesis-ops', {
            stream: this.dataStreams[schemaName],
            deliveryStream: deliveryStream,
        });
        if (pipeline.streamProperties.lambdaDataGenerator) {
            const dataGeneratorFunction = new lambda.Function(dataStreamStack, 'data-generator-function', {
                code: pipeline.streamProperties.lambdaDataGenerator.code,
                handler: pipeline.streamProperties.lambdaDataGenerator.handler,
                timeout: pipeline.streamProperties.lambdaDataGenerator.timeout,
                runtime: pipeline.streamProperties.lambdaDataGenerator.runtime,
                functionName: pipeline.streamProperties.lambdaDataGenerator.functionName,
                environment: {
                    KINESIS_STREAM: this.dataStreams[schemaName].stream.streamName,
                },
            });
            this.dataStreams[schemaName].stream.grantWrite(dataGeneratorFunction);
            const rule = new events.Rule(this, 'Rule', {
                schedule: pipeline.streamProperties.lambdaDataGenerator.schedule,
                ruleName: pipeline.streamProperties.lambdaDataGenerator.ruleName,
            });
            rule.addTarget(new targets.LambdaFunction(dataGeneratorFunction));
        }
        return this.dataStreams[schemaName];
    }
    addPipeline(pipeline, dataProduct) {
        const schemaName = pipeline.name;
        const dataSetStack = dataProduct.accountId == aws_cdk_lib_1.Aws.ACCOUNT_ID ? new aws_cdk_lib_1.NestedStack(aws_cdk_lib_1.Stack.of(this), `${schemaName}-dataset-stack`) : this;
        // create the dataSet
        this.dataSets[schemaName] = new data_set_1.DataSet(dataSetStack, schemaName, {
            pipeline: pipeline,
            dataProduct: dataProduct,
            logBucket: this.logBucket,
            stage: this.stageName,
            s3BucketProps: dataProduct.s3BucketProps,
            lakeType: this.lakeType,
            dataTiers: [data_set_1.DataTier.RAW, data_set_1.DataTier.TRUSTED, data_set_1.DataTier.REFINED],
            datalakeAdminRole: this.datalakeAdminRole,
            datalakeDbCreatorRole: this.datalakeDbCreatorRole,
        });
        const ds = this.dataSets[schemaName];
        const catelogAccountId = dataProduct.dataCatalogAccountId ? dataProduct.dataCatalogAccountId : aws_cdk_lib_1.Aws.ACCOUNT_ID;
        if (this.lakeType === LakeType.DATA_PRODUCT || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG) {
            this.createPipelineResources(pipeline, dataProduct, ds);
        }
        // only create the table if the lake has a catelog
        if (pipeline.table && (this.lakeType === LakeType.CENTRAL_CATALOG || this.lakeType === LakeType.DATA_PRODUCT_AND_CATALOG)) {
            const table = new glue_table_1.GlueTable(this, `${pipeline.name}-table`, {
                catalogId: pipeline.table.catalogId,
                columns: pipeline.table.columns,
                databaseName: this.databases[dataProduct.databaseName].databaseName,
                description: pipeline.table.description,
                inputFormat: pipeline.table.inputFormat,
                outputFormat: pipeline.table.outputFormat,
                parameters: pipeline.table.parameters,
                partitionKeys: pipeline.table.partitionKeys,
                s3Location: `s3://${ds.getDataSetBucketName(pipeline.dataSetDropTier)}/${pipeline.destinationPrefix}`,
                serdeParameters: pipeline.table.serdeParameters,
                serializationLibrary: pipeline.table.serializationLibrary,
                tableName: pipeline.table.tableName,
            });
            table.node.addDependency(this.databases[dataProduct.databaseName]);
        }
        // find the correct metadata catalog account
        if (catelogAccountId == aws_cdk_lib_1.Aws.ACCOUNT_ID) {
            // refactor to only register the needed buckets from the data product account
            if (!pipeline.table) {
                const bucketName = ds.getDataSetBucketName(pipeline.dataSetDropTier);
                // still dirty needs more refactoring
                if (bucketName) {
                    const name = bucketName.replace(/\W/g, '');
                    // only create a crawler for the drop location of the data in the data product of the pipeline
                    const crawler = new glue_crawler_1.GlueCrawler(this, `data-lake-crawler-${name}`, {
                        name: utils_1.buildGlueCrawlerName({
                            stage: this.stageName,
                            resourceUse: 'crawler',
                            name: pipeline.name,
                        }),
                        databaseName: dataProduct.databaseName,
                        bucketName: bucketName,
                        bucketPrefix: pipeline.destinationPrefix,
                        roleName: utils_1.buildRoleName({
                            stage: this.stageName,
                            resourceUse: 'crawler-role',
                            name: pipeline.name,
                        }),
                    });
                    ds.locationRegistry.forEach(r => {
                        crawler.node.addDependency(r);
                    });
                }
            }
        }
    }
    // this is a jumbled mess clean up once refecto
    createPipelineResources(pipeline, dataProduct, ds) {
        switch (pipeline.type) {
            case pipeline_1.DataPipelineType.S3: {
                if (ds.downloadLocations) {
                    this.downloadLocations[pipeline.name] = ds.downloadLocations;
                }
                break;
            }
            case pipeline_1.DataPipelineType.STREAM: {
                this.addDataStream(pipeline, ds);
                break;
            }
            case pipeline_1.DataPipelineType.JDBC: {
                this.createJDBCConnection(pipeline);
                break;
            }
        }
        // rethink this whole section
        if (pipeline.job) {
            const jobScript = utils_1.packageAsset(this, `${pipeline.name}Script`, pipeline.job.jobScript);
            pipeline.job.jobArgs['--TempDir'] = `s3://${this.logBucket.bucketName}/temp/`;
            pipeline.job.jobArgs['--spark-event-logs-path'] = `s3://${this.logBucket.bucketName}/logs/`;
            let s3Location = ds.getDataSetBucketName(pipeline.job.destinationLocation);
            if (pipeline.job.destinationLocation && s3Location) {
                pipeline.job.jobArgs['--DESTINATION_BUCKET'] = s3Location;
                const job = new glue_job_1.GlueJob(this, `${pipeline.name}-etl-job`, {
                    deploymentBucket: jobScript.bucket,
                    jobScript: utils_1.toS3Path(jobScript),
                    name: pipeline.job.name,
                    workerType: pipeline.job.workerType,
                    description: pipeline.job.description,
                    glueVersion: pipeline.job.glueVersion,
                    jobArgs: pipeline.job.jobArgs,
                    maxCapacity: pipeline.job.maxCapacity,
                    maxConcurrentRuns: pipeline.job.maxConcurrentRuns,
                    maxRetries: pipeline.job.maxRetries,
                    numberOfWorkers: pipeline.job.numberOfWorkers,
                    roleName: pipeline.job.roleName,
                    timeout: pipeline.job.timeout,
                    jobType: pipeline.job.jobType,
                    readAccessBuckets: [
                        this.logBucket,
                    ],
                    writeAccessBuckets: [
                        this.logBucket,
                        s3.Bucket.fromBucketName(this, 'raw-bucket-role', s3Location),
                    ],
                });
                new glue_job_ops_1.GlueJobOps(this, `${pipeline.name}-etl-job-ops`, {
                    job: job,
                });
                if (pipeline.streamProperties) {
                    this.dataStreams[pipeline.name].stream.grantRead(job.role);
                }
                new lf.CfnPermissions(this, `${pipeline.name}-create-table-perm`, {
                    dataLakePrincipal: {
                        dataLakePrincipalIdentifier: job.role.roleArn,
                    },
                    resource: {
                        databaseResource: {
                            name: dataProduct.databaseName,
                        },
                    },
                    permissions: [
                        Permissions.ALTER,
                        Permissions.CREATE_TABLE,
                        Permissions.DESCRIBE,
                    ],
                });
                if (pipeline.table) {
                    new lf.CfnPermissions(this, `${pipeline.name}-access-table-perm`, {
                        dataLakePrincipal: {
                            dataLakePrincipalIdentifier: job.role.roleArn,
                        },
                        resource: {
                            tableResource: {
                                databaseName: dataProduct.databaseName,
                                name: pipeline.table.tableName,
                            },
                        },
                        permissions: [
                            Permissions.SELECT,
                            Permissions.DESCRIBE,
                        ],
                    });
                }
            }
        }
    }
    createJDBCConnection(pipeline) {
        if (this.vpc && this.glueSecurityGroup) {
            new glue.Connection(this, `${pipeline.name}-glue-connection`, {
                type: glue.ConnectionType.JDBC,
                connectionName: `${pipeline.name}-jdbc`,
                description: `JDBC connection for glue to use on pipeline ${pipeline.name}`,
                subnet: this.vpc.isolatedSubnets[0],
                securityGroups: [this.glueSecurityGroup],
                properties: {
                    JDBC_CONNECTION_URL: pipeline.jdbcProperties.jdbc,
                    USERNAME: pipeline.jdbcProperties.username,
                    PASSWORD: pipeline.jdbcProperties.password,
                },
            });
        }
        else {
            throw new Error('VPC required to create a JDBC pipeline.');
        }
    }
    createPolicyTagsCustomResource(policyTags, datalakeAdminRole) {
        const onEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'create-policy-tags-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/create-tags-handler'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-tags',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const myProvider = new cr.Provider(this, 'policy-tags-provider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const outputs = new aws_cdk_lib_1.CustomResource(this, 'tag-creation-custom-resource', {
            serviceToken: myProvider.serviceToken,
            properties: {
                policyTags: policyTags,
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
            },
        });
        outputs.node.addDependency(datalakeAdminRole);
    }
    /**
     * @stability stable
     */
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds, dataCatalogOwnerAccountId) {
        const onCatalogEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'enable-hybrid-catalog-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/enable-hybrid-catalog'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-catalog',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const catalogProvider = new cr.Provider(this, 'hybrid-catalog-provider', {
            onEventHandler: onCatalogEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        new aws_cdk_lib_1.CustomResource(this, 'hybrid-catalog-custom-resource', {
            serviceToken: catalogProvider.serviceToken,
            properties: {
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
                consumerAccountIds: consumerAccountIds,
                producerAccountId: dataCatalogOwnerAccountId,
            },
        });
    }
}
exports.DataLake = DataLake;
_a = JSII_RTTI_SYMBOL_1;
DataLake[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.DataLake", version: "0.0.44" };
//# sourceMappingURL=data:application/json;base64,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