"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataSet = exports.DataTier = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lf = require("aws-cdk-lib/aws-lakeformation");
const s3sns = require("aws-cdk-lib/aws-s3-notifications");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
const __1 = require("..");
const data_lake_bucket_1 = require("../data-lake-bucket");
const utils_1 = require("../utils");
/**
 * @stability stable
 */
var DataTier;
(function (DataTier) {
    DataTier["RAW"] = "raw";
    DataTier["REFINED"] = "refined";
    DataTier["TRUSTED"] = "trusted";
})(DataTier = exports.DataTier || (exports.DataTier = {}));
/**
 * @stability stable
 */
class DataSet extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.name = props.pipeline.name;
        this.dropLocation = props.pipeline.dataSetDropTier;
        this.pipeline = props.pipeline;
        this.dataProduct = props.dataProduct;
        this.lakeType = props.lakeType;
        this.locationRegistry = [];
        const dataCatalogAccountId = props.dataProduct.dataCatalogAccountId ?
            props.dataProduct.dataCatalogAccountId : props.dataProduct.accountId;
        const crossAccount = props.dataProduct.dataCatalogAccountId ?
            props.dataProduct.dataCatalogAccountId != props.dataProduct.accountId ? true : false : false;
        if (props.dataTiers.includes(DataTier.RAW)) {
            this.rawBucketName = utils_1.buildS3BucketName({
                name: props.pipeline.name,
                accountId: this.dataProduct.accountId,
                resourceUse: 'raw',
                stage: props.stage,
            });
        }
        if (props.dataTiers.includes(DataTier.TRUSTED)) {
            this.trustedBucketName = utils_1.buildS3BucketName({
                name: props.pipeline.name,
                accountId: this.dataProduct.accountId,
                resourceUse: 'trusted',
                stage: props.stage,
            });
        }
        if (props.dataTiers.includes(DataTier.REFINED)) {
            this.refinedBucketName = utils_1.buildS3BucketName({
                name: props.pipeline.name,
                accountId: this.dataProduct.accountId,
                resourceUse: 'refined',
                stage: props.stage,
            });
        }
        props.dataTiers.forEach(r => {
            this.createDataTierBucket({
                bucketName: this.getDataSetBucketName(r),
                crossAccount: crossAccount,
                dataCatalogAccountId: dataCatalogAccountId,
                lakeType: props.lakeType,
                logBucket: props.logBucket,
                pipelineName: props.pipeline.name,
                s3BucketProps: props.s3BucketProps,
                s3NotificationProps: props.pipeline.s3NotificationProps,
                datalakeAdminRole: props.datalakeAdminRole,
                datalakeDbCreatorRole: props.datalakeDbCreatorRole,
                tier: r,
            });
        });
        // revisit this
        if (this.dropLocation) {
            this.downloadLocations = {
                destinationPrefix: props.pipeline.destinationPrefix,
                destinationBucketName: this.getDataSetBucketName(this.dropLocation),
                sourceBucketName: props.pipeline.s3Properties ? props.pipeline.s3Properties.sourceBucketName : undefined,
                sourceKeys: props.pipeline.s3Properties ? props.pipeline.s3Properties.sourceKeys : undefined,
            };
        }
    }
    createS3NotificationTopic(s3NotificationProps, bucket) {
        this.s3NotificationTopic = new sns.Topic(this, 'sns-datalake-notification-topic');
        bucket.addEventNotification(s3NotificationProps.event, new s3sns.SnsDestination(this.s3NotificationTopic), {
            prefix: s3NotificationProps.prefix,
            suffix: s3NotificationProps.suffix,
        });
    }
    createDataTierBucket(props) {
        if (props.lakeType === __1.LakeType.DATA_PRODUCT || props.lakeType === __1.LakeType.DATA_PRODUCT_AND_CATALOG && props.bucketName) {
            const bucket = new data_lake_bucket_1.DataLakeBucket(this, `s3-${props.tier}-bucket-${props.pipelineName}`, {
                bucketName: props.bucketName,
                dataCatalogAccountId: props.dataCatalogAccountId,
                logBucket: props.logBucket,
                crossAccount: props.crossAccount,
                s3Properties: props.s3BucketProps,
            }).bucket;
            if (props.s3NotificationProps) {
                this.createS3NotificationTopic(props.s3NotificationProps, bucket);
            }
            if (props.dataCatalogAccountId == aws_cdk_lib_1.Aws.ACCOUNT_ID) {
                this.locationRegistry.push(this.registerDataLakeLocation(props.datalakeAdminRole.roleArn, props.datalakeDbCreatorRole.roleArn, props.bucketName));
            }
        }
    }
    registerDataLakeLocation(dataLakeAdminRoleArn, dataLakeDbCreatorRoleArn, bucketName) {
        const name = bucketName.replace(/\W/g, '');
        const dlResource = new lf.CfnResource(this, `lf-resource-${name}`, {
            resourceArn: `arn:aws:s3:::${bucketName}`,
            useServiceLinkedRole: false,
            roleArn: dataLakeDbCreatorRoleArn,
        });
        this.createDataLocationAccessPermission(`${name}-creator`, dataLakeDbCreatorRoleArn, bucketName, dlResource);
        this.createDataLocationAccessPermission(`${name}-admin`, dataLakeAdminRoleArn, bucketName, dlResource);
        return dlResource;
    }
    createDataLocationAccessPermission(name, roleArn, bucketName, resource) {
        const perm = new lf.CfnPermissions(this, `datalake-creator-perm-${name}`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${bucketName}`,
                },
            },
            permissions: [
                __1.Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        perm.node.addDependency(resource);
        return perm;
    }
    /**
     * @stability stable
     */
    getDataSetBucketName(dataTier) {
        return dataTier == DataTier.RAW ? this.rawBucketName :
            dataTier == DataTier.REFINED ? this.refinedBucketName :
                dataTier == DataTier.TRUSTED ? this.trustedBucketName : undefined;
    }
}
exports.DataSet = DataSet;
_a = JSII_RTTI_SYMBOL_1;
DataSet[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.DataSet", version: "0.0.44" };
//# sourceMappingURL=data:application/json;base64,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