"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueJobOps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
/**
 * @stability stable
 */
class GlueJobOps extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.job = props.job;
        this.dashboard = new cloudwatch.Dashboard(this, 'dashboard', {
            dashboardName: `ETL_${this.job.name}`,
        });
        this.jvmHeapSizeExceeding80PercentAlarm = new cloudwatch.Alarm(this, 'jvm-heapSize-exceeding80percent-alarm', {
            alarmName: `${this.job.name} JvmHeapSizeExceeding80`,
            alarmDescription: `Jvm Heap Size exceeding 80% glue job (${this.job.name})`,
            metric: this.job.jvmHeapUsageMetric({
                period: aws_cdk_lib_1.Duration.days(1),
                statistic: cloudwatch.Statistic.MAXIMUM,
            }),
            threshold: 0.8,
            evaluationPeriods: 1,
            datapointsToAlarm: 1,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            ...(props.jvmHeapSizeExceeding80percent || {}),
        });
        this.jvmHeapSizeExceeding90PercentAlarm = new cloudwatch.Alarm(this, 'jvm-heapSize-exceeding90Percent-alarm', {
            alarmName: `${this.job.name} JvmHeapSizeExceeding90`,
            alarmDescription: `Jvm Heap Size exceeding 90% glue job (${this.job.name})`,
            metric: this.job.jvmHeapUsageMetric({
                period: aws_cdk_lib_1.Duration.days(1),
                statistic: cloudwatch.Statistic.MAXIMUM,
            }),
            threshold: 0.9,
            evaluationPeriods: 1,
            datapointsToAlarm: 1,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            ...(props.jvmHeapSizeExceeding90percent || {}),
        });
        this.metricExecutionFailureAlarm = new cloudwatch.Alarm(this, 'metric-execution-failure-alarm', {
            alarmName: `${this.job.name} ExecutionFailure`,
            alarmDescription: `Error while running the Glue job ${this.job.name} on the current attempt. There might be job retries after this error.`,
            metric: this.job.metricFailure(),
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            threshold: 1,
            evaluationPeriods: 1,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.metricExecutionFailure || {}),
        });
        this.metricAllExecutionAttemptsFailedAlarm = new cloudwatch.Alarm(this, 'metric-all-execution-attempts-failed-alarm', {
            alarmName: `${this.job.name} AllExecutionAttemptsFailed`,
            alarmDescription: `Error while running the Glue job ${this.job.name} on the last attempt. There will be no retries of the job after this error.`,
            metric: this.job.metricAllExecutionAttemptsFailed(),
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            threshold: 1,
            evaluationPeriods: 1,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.metricAllExecutionAttemptsFailed || {}),
        });
        this.alarmsSev2 = [
            this.jvmHeapSizeExceeding90PercentAlarm,
            this.metricAllExecutionAttemptsFailedAlarm,
        ];
        this.alarmsSev3 = [
            this.jvmHeapSizeExceeding80PercentAlarm,
            this.metricExecutionFailureAlarm,
        ];
        this.setupDashboard();
    }
    addWidgets(widgets) {
        for (let i = 0; i < widgets.length; i += 4) {
            this.dashboard.addWidgets(...widgets.slice(i, i + 4));
        }
    }
    setupDashboard() {
        this.dashboard.addWidgets(new cloudwatch.TextWidget({
            markdown: `# ${this.job.name} Job Result`,
            height: 1,
            width: 24,
        }));
        const jobResultWidgets = [
            new cloudwatch.GraphWidget({
                left: [
                    new cloudwatch.MathExpression({
                        expression: `SEARCH('MetricName="TriggeredRules" RuleName="${this.job.metricSuccessRule.ruleName}"', 'Sum', 900)`,
                        usingMetrics: {},
                        label: 'Success Count',
                    }),
                ],
                title: 'Success Count',
                height: 6,
                width: 6,
            }),
            new cloudwatch.GraphWidget({
                left: [
                    new cloudwatch.MathExpression({
                        expression: `SEARCH('MetricName="TriggeredRules" RuleName="${this.job.metricFailureRule.ruleName}"', 'Sum', 900)`,
                        usingMetrics: {},
                        label: 'Failure Count',
                    }),
                ],
                title: 'Failure Count',
                height: 6,
                width: 6,
            }),
            new cloudwatch.GraphWidget({
                left: [
                    new cloudwatch.MathExpression({
                        expression: `SEARCH('MetricName="TriggeredRules" RuleName="${this.job.metricTimeoutRule.ruleName}"', 'Sum', 900)`,
                        usingMetrics: {},
                        label: 'Timeout Count',
                    }),
                ],
                title: 'Timeout Count',
                height: 6,
                width: 6,
            }),
        ];
        this.dashboard.addWidgets(...jobResultWidgets);
        this.dashboard.addWidgets(new cloudwatch.TextWidget({
            markdown: `# ${this.job.name} JVM Glue Driver Stats Alarms`,
            height: 1,
            width: 24,
        }));
        this.dashboard.addWidgets(...[
            new cloudwatch.GraphWidget({
                left: [this.job.diskSpaceUsedMbMetric()],
                title: `${this.job.diskSpaceUsedMbMetric().metricName} (${this.job.diskSpaceUsedMbMetric().statistic})`,
                height: 6,
                width: 6,
            }),
            new cloudwatch.GraphWidget({
                left: [this.job.elapsedTimeMetric()],
                title: `${this.job.elapsedTimeMetric().metricName} (${this.job.elapsedTimeMetric().statistic})`,
                height: 6,
                width: 6,
            }),
        ]);
        const sev2AlarmWidgets = this.alarmsSev2.map(this.alarmWidget);
        const sev3AlarmWidgets = this.alarmsSev3.map(this.alarmWidget);
        this.addWidgets(sev2AlarmWidgets);
        this.addWidgets(sev3AlarmWidgets);
    }
    alarmWidget(alarm) {
        return new cloudwatch.AlarmWidget({
            alarm: alarm,
            title: `${alarm.alarmName}`,
            height: 6,
            width: 6,
        });
    }
}
exports.GlueJobOps = GlueJobOps;
_a = JSII_RTTI_SYMBOL_1;
GlueJobOps[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.GlueJobOps", version: "0.0.44" };
//# sourceMappingURL=data:application/json;base64,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