"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueJob = exports.GlueJobType = exports.GlueVersion = exports.GlueWorkerType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const eventtargets = require("aws-cdk-lib/aws-events-targets");
const glue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * @stability stable
 */
var GlueWorkerType;
(function (GlueWorkerType) {
    GlueWorkerType["STANDARD"] = "Standard";
    GlueWorkerType["G1_X"] = "G.1X";
    GlueWorkerType["G2_X"] = "G.2X";
})(GlueWorkerType = exports.GlueWorkerType || (exports.GlueWorkerType = {}));
/**
 * @stability stable
 */
var GlueVersion;
(function (GlueVersion) {
    GlueVersion["V_0"] = "0.9";
    GlueVersion["V_1"] = "1.0";
    GlueVersion["V_2"] = "2.0";
    GlueVersion["V_3"] = "3.0";
})(GlueVersion = exports.GlueVersion || (exports.GlueVersion = {}));
/**
 * @stability stable
 */
var GlueJobType;
(function (GlueJobType) {
    GlueJobType["GLUE_ETL"] = "glueetl";
    GlueJobType["GLUE_STREAMING"] = "gluestreaming";
})(GlueJobType = exports.GlueJobType || (exports.GlueJobType = {}));
/**
 * @stability stable
 */
class GlueJob extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.allExecutionAttemptsFailedEventSource = 'custom.aws.glue.allExecutionAttemptsFailed';
        /**
         * @stability stable
         */
        this.allExecutionAttemptsFailedEventDetailType = 'All Execution Attempts Failed';
        this.role = this.createGlueJobRole(props);
        this.job = new glue.CfnJob(this, `${props.name}-glue-job`, {
            name: props.name,
            description: props.description,
            workerType: props.workerType,
            numberOfWorkers: props.numberOfWorkers,
            role: this.role.roleName,
            maxRetries: props.maxRetries || 0,
            executionProperty: {
                maxConcurrentRuns: props.maxConcurrentRuns || 3,
            },
            glueVersion: props.glueVersion || GlueVersion.V_1,
            command: {
                pythonVersion: '3',
                scriptLocation: props.jobScript,
                name: props.jobType,
            },
            timeout: props.timeout || GlueJob.DAY_IN_MINUTES,
            defaultArguments: {
                '--job-language': 'python',
                '--enable-metrics': true,
                '--enable-continuous-cloudwatch-log': true,
                '--region': aws_cdk_lib_1.Stack.of(this).region,
                '--enable-glue-datacatalog': true,
                '--enable-continuous-log-filter': true,
                '--enable-spark-ui': true,
                ...props.jobArgs,
            },
        });
        this.name = props.name;
        this.metricSuccessRule = this.jobRule('SuccessRule', this.name, 'SUCCEEDED');
        this.metricFailureRule = this.jobRule('FailureRule', this.name, 'FAILED');
        this.metricTimeoutRule = this.jobRule('TimeoutRule', this.name, 'TIMEOUT');
        this.executionFailureRule = new events.Rule(scope, `${this.name}-execution-failure-rule`, {
            description: `Glue job ${this.name} failed or timed out on an attempt. There might be job retries after this error.`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Job State Change'],
                detail: {
                    state: ['FAILED', 'TIMEOUT'],
                    jobName: [this.name],
                },
            },
        });
        this.lambdaFunction = this.createLambdaFunction();
        this.executionFailureRule.addTarget(new eventtargets.LambdaFunction(this.lambdaFunction));
        this.allExecutionAttemptsFailedRule = new events.Rule(this, `${this.name}-all-execution-attempts-failed-rule`, {
            description: `Glue job ${this.name} failed or timed out on the last attempt. There will be no retries of the job after this error.`,
            eventPattern: {
                source: [this.allExecutionAttemptsFailedEventSource],
                detailType: [this.allExecutionAttemptsFailedEventDetailType],
                detail: {
                    jobName: [this.name],
                },
            },
        });
    }
    createGlueJobRole(props) {
        const role = new iam.Role(this, 'Role', {
            roleName: props.roleName || props.name + 'Role',
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AWSGlueConsoleFullAccess'),
            ],
        });
        role.addToPolicy(new iam.PolicyStatement({ actions: ['lakeformation:GetDataAccess'], resources: ['*'] }));
        props.deploymentBucket.grantRead(role);
        if (props.readAccessBuckets) {
            props.readAccessBuckets.forEach(bucket => {
                bucket.grantRead(role);
            });
        }
        if (props.writeAccessBuckets) {
            props.writeAccessBuckets.forEach(bucket => {
                bucket.grantWrite(role);
            });
        }
        return role;
    }
    jobRule(id, jobName, ...states) {
        return new events.Rule(this, id, {
            ruleName: jobName + states.join(''),
            description: `Event triggered when Glue job ${jobName} is in ${states.join(' or ')} state(s)`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Job State Change'],
                detail: {
                    state: states,
                    jobName: [jobName],
                },
            },
        });
    }
    /**
     * @stability stable
     */
    metricSuccess(props) {
        return this.ruleMetric(this.metricSuccessRule, props);
    }
    /**
     * @stability stable
     */
    metricFailure(props) {
        return this.ruleMetric(this.metricFailureRule, props);
    }
    /**
     * @stability stable
     */
    metricAllExecutionAttemptsFailed(props) {
        return new cloudwatch.Metric({
            metricName: 'TriggeredRules',
            namespace: 'AWS/Events',
            dimensionsMap: {
                RuleName: this.allExecutionAttemptsFailedRule.ruleName,
            },
            statistic: 'Sum',
            period: aws_cdk_lib_1.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * @stability stable
     */
    metricTimeout(props) {
        return this.ruleMetric(this.metricTimeoutRule, props);
    }
    ruleMetric({ ruleName }, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Events',
            metricName: 'TriggeredRules',
            dimensionsMap: { RuleName: ruleName },
            statistic: cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    /**
     * @stability stable
     */
    metric(metricName, dimensionType, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Glue',
            metricName,
            dimensionsMap: {
                JobName: this.name,
                JobRunId: 'ALL',
                Type: dimensionType,
            },
            ...props,
        });
    }
    /**
     * @stability stable
     */
    jvmHeapUsageMetric(props) {
        return this.metric('glue.ALL.jvm.heap.usage', 'gauge', props);
    }
    /**
     * @stability stable
     */
    elapsedTimeMetric(props) {
        return this.metric('glue.driver.aggregate.elapsedTime', 'count', props);
    }
    /**
     * @stability stable
     */
    diskSpaceUsedMbMetric(props) {
        return this.metric('glue.driver.BlockManager.disk.diskSpaceUsed_MB', 'gauge', props);
    }
    /**
     * @stability stable
     */
    runTimeInMiliseconds(props) {
        return this.metric('glue.driver.aggregate.elapsedTime', 'gauge', props);
    }
    createLambdaFunction() {
        const lambdaFunction = new lambda.SingletonFunction(this, `GlueExecutionFailListenerLambdaSingleton${this.name}`, {
            description: 'Checks if an error of a Glue job was on the last attempt (no more retries) in which case the function sends out an event.',
            environment: {
                eventToSendSource: this.allExecutionAttemptsFailedEventSource,
                eventToSendDetailType: this.allExecutionAttemptsFailedEventDetailType,
            },
            uuid: 'GlueExecutionFailListenerLambda',
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(1),
            code: lambda.Code.fromInline(`
import boto3
import json
import os
import re

def handler(event, context):
    try:
        jobRunId = event['detail']['jobRunId']
        jobName = event['detail']['jobName']
    except:
        raise Exception(f'Received an malformed event. ({event})')
        
    # get the current execution attempt, we parse it from the jobRunId which has a _attempt_# suffix on retries
    try:
        curExecutionAttempt = int(re.findall('_attempt_(\\d*)$', jobRunId)[0])
    except IndexError:
        curExecutionAttempt = 0
    
    # get the number of MaxRetries for this glue job
    try:
        glue_client = boto3.client('glue')
        maxRetries = glue_client.get_job(JobName=jobName)['Job']['MaxRetries']
    except Exception as e:
        raise Exception(f'Failed to access the Glue API to get the MaxRetries parameter. ({e})')
        
    # is this the last execution? if yes we send out the event
    isLastExecutionAttempt = curExecutionAttempt == maxRetries
    print(f'Job name: {jobName}, is last execution attempt: {isLastExecutionAttempt}, current attempt: {curExecutionAttempt}, max retry attempts: {maxRetries}')
    if isLastExecutionAttempt:
        event_client = boto3.client('events')
        event_client.put_events(Entries=[{
            'Source': os.environ['eventToSendSource'],
            'Detail': json.dumps(event['detail']),
            'DetailType': os.environ['eventToSendDetailType']
        }])
    `),
        });
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const accountId = aws_cdk_lib_1.Stack.of(this).account;
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [`arn:aws:events:${region}:${accountId}:event-bus/default`],
        }));
        lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['glue:GetJob'],
            resources: [`arn:aws:glue:${region}:${accountId}:job/${this.name}`],
        }));
        return lambdaFunction;
    }
}
exports.GlueJob = GlueJob;
_a = JSII_RTTI_SYMBOL_1;
GlueJob[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.GlueJob", version: "0.0.44" };
GlueJob.DAY_IN_MINUTES = 1440;
//# sourceMappingURL=data:application/json;base64,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