"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScheduledJobWorkflow = void 0;
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
;
const utils_1 = require("../utils");
class ScheduledJobWorkflow extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stateMachineRole = new iam.Role(scope, 'StateMachineJobExecutionRole', {
            roleName: utils_1.buildRoleName({
                name: props.name,
                resourceUse: 'datalake',
                stage: props.stageName,
            }),
            assumedBy: new iam.ServicePrincipal('states'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3FullAccess'),
            ],
        });
        const jobState = new tasks.GlueStartJobRun(this, 'GlueExecutionStep', {
            glueJobName: props.jobName,
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
            arguments: sfn.TaskInput.fromObject(props.jobArguments),
            timeout: props.jobTimeout,
            resultPath: '$.jobOutput',
        });
        const stateMachineDefinition = sfn.Chain.start(jobState).toSingleState('Run Job pipeline', {
            comment: 'Container for glue job states',
        }).addCatch(this.getStateMachineFailureHandlerState(), {});
        this.stateMachine = new sfn.StateMachine(this, 'GlueStateMachine', {
            definition: stateMachineDefinition,
            logs: {
                destination: new logs.LogGroup(this, `DataLakeWorkflow-${props.name}`, {
                    retention: logs.RetentionDays.SIX_MONTHS,
                }),
                includeExecutionData: true,
                level: sfn.LogLevel.ERROR,
            },
            tracingEnabled: true,
            role: stateMachineRole,
        });
        this.rule = new events.Rule(this, 'Rule', {
            schedule: props.schedule,
            ruleName: utils_1.buildEventRuleName({
                name: props.name,
                resourceUse: 'datalake',
                stage: props.stageName,
            }),
        });
        this.rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
    }
    getStateMachineFailureHandlerState() {
        return new sfn.Fail(this, 'Handle failures', {
            comment: 'Handle failures for entire state machine',
        });
    }
}
exports.ScheduledJobWorkflow = ScheduledJobWorkflow;
//# sourceMappingURL=data:application/json;base64,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