
__all__ = ['generate_array_stub']

import textwrap

from matrixtype import get_matrix_types, inspect_matrix
from quaterniontype import get_quaternion_types, inspect_quaternion
from stub import union, vector_name
from typemap import DATA_TYPE_SIZE_PREFIX_TO_C_TYPE
from vectortype import get_vector_types, inspect_vector

_INT_CTYPES = [
    'ctypes.c_bool',
    'ctypes.c_byte',
    'ctypes.c_int8',
    'ctypes.c_ubyte',
    'ctypes.c_uint8',
    'ctypes.c_short',
    'ctypes.c_int16',
    'ctypes.c_ushort',
    'ctypes.c_uint16',
    'ctypes.c_int',
    'ctypes.c_int32',
    'ctypes.c_uint',
    'ctypes.c_uint32',
    'ctypes.c_long',
    'ctypes.c_int64',
    'ctypes.c_ulong',
    'ctypes.c_uint64',
    'ctypes.c_longlong',
    'ctypes.c_ulonglong',
    'ctypes.c_size_t',
    'ctypes.c_ssize_t',
]

_FLOAT_CTYPES = [
    'ctypes.c_float',
    'ctypes.c_double',
    'ctypes.c_longdouble',
]

_CTYPES = [*_INT_CTYPES, *_FLOAT_CTYPES]

_GLM_TYPES = [
    *get_matrix_types(),
    *get_quaternion_types(),
    *get_vector_types(),
]

_ARRAY_TYPES = [*_CTYPES, *_GLM_TYPES]


_SPLIT_COMPONENTS = {}
for name in get_vector_types():
    vector = inspect_vector(name)
    _SPLIT_COMPONENTS[name] = f'''Tuple[{', '.join(f'array[ctypes.{DATA_TYPE_SIZE_PREFIX_TO_C_TYPE[(vector.data_type, vector.data_size)].__name__}]' for i in range(vector.size))}]'''
for name in get_quaternion_types():
    quaternion = inspect_quaternion(name)
    _SPLIT_COMPONENTS[name] = f'''Tuple[{', '.join(f'array[ctypes.{DATA_TYPE_SIZE_PREFIX_TO_C_TYPE[(quaternion.data_type, quaternion.data_size)].__name__}]' for i in range(4))}]'''
for name in get_matrix_types():
    matrix = inspect_matrix(name)
    _SPLIT_COMPONENTS[name] = f'''Tuple[{', '.join(f'array[{vector_name(matrix.data_type, matrix.columns)}]' for i in range(matrix.rows))}]'''


# this would be used to over the ctype property, but properties can't actually
# be overloaded
_CTYPE_PROPERTY = {t: t for t in _CTYPES}
for name in get_vector_types():
    vector = inspect_vector(name)
    _CTYPE_PROPERTY[name] = f'ctypes.{DATA_TYPE_SIZE_PREFIX_TO_C_TYPE[(vector.data_type, vector.data_size)].__name__}'
for name in get_quaternion_types():
    quaternion = inspect_quaternion(name)
    _CTYPE_PROPERTY[name] = f'ctypes.{DATA_TYPE_SIZE_PREFIX_TO_C_TYPE[(quaternion.data_type, quaternion.data_size)].__name__}'
for name in get_matrix_types():
    matrix = inspect_matrix(name)
    _CTYPE_PROPERTY[name] = f'ctypes.{DATA_TYPE_SIZE_PREFIX_TO_C_TYPE[(matrix.data_type, matrix.data_size)].__name__}'


def generate_array_stub():
    return ['array'], textwrap.dedent(f"""

    _AT = TypeVar('_AT', {", ".join(_ARRAY_TYPES)})
    _AT2 = TypeVar('_AT2', {", ".join(_ARRAY_TYPES)})
    _GLMT = TypeVar('_GLMT', {", ".join(_GLM_TYPES)})
    _CT = TypeVar('_CT', {", ".join(_CTYPES)})
    _ICT = TypeVar('_ICT', {", ".join(_INT_CTYPES)})
    _FCT = TypeVar('_FCT', {", ".join(_INT_CTYPES)})

    class array(Generic[_AT]):

        nbytes: int
        typecode: str
        element_type: Type[_AT]
        itemsize: int
        dt_size: int
        address: int
        length: int
        readonly: bool
        reference: Any

        @property
        def ptr(self) -> ctypes.c_void_p: ...

        @property
        def dtype(self) -> str: ...

        @property
        def ctype(self) -> Type[{union(_CTYPES)}]: ...

        @overload
        def __init__(self, _1: _AT, /, *_: _AT) -> None: ...
        @overload
        def __init__(self, _: array[_AT], /) -> None: ...
        @overload
        def __init__(self, _: Iterable[_AT], /) -> None: ...

        def __len__(self) -> int: ...
        def __contains__(self, value: Any) -> bool: ...

        @overload
        def __iter__(self: array[_GLMT]) -> Generator[_GLMT, None, None]: ...
        @overload
        def __iter__(self: {union(f'array[{t}]'for t in _INT_CTYPES)}) -> Generator[int, None, None]: ...
        @overload
        def __iter__(self: {union(f'array[{t}]'for t in _FLOAT_CTYPES)}) -> Generator[float, None, None]: ...

        @overload
        def __getitem__(self: array[_GLMT], index: int) -> _GLMT: ...
        @overload
        def __getitem__(self: {union(f'array[{t}]'for t in _INT_CTYPES)}, index: int) -> int: ...
        @overload
        def __getitem__(self: {union(f'array[{t}]'for t in _FLOAT_CTYPES)}, index: int) -> float: ...
        @overload
        def __getitem__(self, index: slice) -> array[_AT]: ...

        @overload
        def __setitem__(self: array[_GLMT], index: int, value: _GLMT) -> None: ...
        @overload
        def __setitem__(self: array[_ICT], index: int, value: _ICT) -> None: ...
        @overload
        def __setitem__(self: array[_FCT], index: int, value: _FCT) -> None: ...
        @overload
        def __setitem__(self, index: slice, value: array[_AT]) -> None: ...

        @staticmethod
        def from_bytes(bytes: bytes, type: Type[_AT], /) -> array[_AT]: ...

        @staticmethod
        def from_numbers(type: Type[_CT], /, *numbers: glm_typing.Number) -> array[_CT]: ...

        @staticmethod
        def zeros(length: int, type: Type[_AT], /) -> array[_AT]: ...

        @overload
        def to_list(self: array[_GLMT]) -> List[_GLMT]: ...
        @overload
        def to_list(self: {union(f'array[{t}]'for t in _INT_CTYPES)}) -> List[int]: ...
        @overload
        def to_list(self: {union(f'array[{t}]'for t in _FLOAT_CTYPES)}) -> List[float]: ...

        @overload
        def to_tuple(self: array[_GLMT]) -> Tuple[_GLMT, ...]: ...
        @overload
        def to_tuple(self: {union(f'array[{t}]'for t in _INT_CTYPES)}) -> Tuple[int, ...]: ...
        @overload
        def to_tuple(self: {union(f'array[{t}]'for t in _FLOAT_CTYPES)}) -> Tuple[float, ...]: ...
        {"".join(f'''
        @overload
        def split_components(self: array[{glm_type}]) -> {component_type}: ...''' for glm_type, component_type in _SPLIT_COMPONENTS.items())}

        def to_bytes(self) -> bytes: ...

        def reduce(self, func: Callable[[_AT, _AT], _AT], init: Optional[_AT] = ..., /) -> _AT: ...
        def filter(self, _: Callable[[_AT], Any], /) -> array[_AT]: ...
        def map(self, _: Callable[[_AT], _AT2], /) -> array[_AT2]: ...
        def sort(self, _: Callable[[_AT], SupportsInt], /) -> array[_AT]: ...
        def concat(self, other: array[_AT], /) -> array[_AT]: ...
        def iconcat(self, other: array[_AT], /) -> None: ...
        def repeat(self, count: int, /) -> array[_AT]: ...
        def irepeat(self, count: int, /) -> None: ...

        @overload
        @staticmethod
        def as_reference(array: array[_AT], /) -> array[_AT]: ...
        @overload
        @staticmethod
        def as_reference(obj: _GLMT, /) -> array[_GLMT]: ...

        def reinterpret_cast(self, type: Type[_AT], /) -> array[_AT]: ...

        def __eq__(self, other: Any) -> bool: ...
        def __ne__(self, other: Any) -> bool: ...

        def __add__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __radd__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __iadd__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __sub__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rsub__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __isub__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __mul__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rmul__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __imul__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __mod__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rmod__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __imod__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __pow__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rpow__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __ipow__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __lshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rlshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __ilshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rrshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __irshift__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __and__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rand__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __iand__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __xor__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rxor__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __ixor__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __or__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __ror__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __ior__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __truediv__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __rtruediv__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...
        def __itruediv__(self, other: Union[glm_typing.Number, _AT, array[_AT]]) -> array[_AT]: ...

        def __neg__(self) -> array[_AT]: ...
        def __pos__(self) -> array[_AT]: ...
        def __abs__(self) -> array[_AT]: ...
        def __inv__(self) -> array[_AT]: ...
    """)
