"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDestination = void 0;
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination.
 *
 * @stability stable
 */
class LambdaDestination {
    /**
     * @stability stable
     */
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration.
     *
     * @stability stable
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn,
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload'),
                }),
            ],
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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