import numpy as np


def interp2d(x, xf, f):
    """
    Can interpolate a table to get an array of values in 2D

    Parameters
    ----------
    x: array_like
        1d array of values to be interpolated
    xf: 1d array of values
    f: array_like
        2d array of function values size=(len(xf), n)

    Returns
    -------

    Examples
    --------
    >>> f = np.array([[0, 0, 0],
    >>>              [0, 1, 4],
    >>>              [2, 6, 2],
    >>>              [10, 10, 10]
    >>>              ])
    >>> xf = np.array([0, 1, 2, 3])

    >>> x = np.array([0.5, 1, 2.2, 2.5])
    >>> f_interp = interp2d(x, xf, f)
    >>> print(f_interp[0][0])
    0.0
    >>> print(f_interp[0][1])
    0.5
    >>> print(f_interp[0][2])
    2.0
    """
    ind = np.argmin(np.abs(x[:, np.newaxis] - xf), axis=1)
    x_ind = xf[ind]
    ind0 = np.where(x_ind > x, ind - 1, ind)
    ind1 = np.where(x_ind > x, ind, ind + 1)
    ind0 = np.clip(ind0, 0, None)
    ind1 = np.clip(ind1, None, len(xf) - 1)
    f0 = f[ind0]
    f1 = f[ind1]
    a0 = xf[ind0]
    a1 = xf[ind1]
    denom = (a1 - a0)
    denom_adj = np.clip(denom, 1e-10, None)  # to avoid divide by zero warning
    s0 = np.where(denom > 0, (x - a0) / denom_adj, 1)  # if denom less than 0, then out of bounds
    s1 = 1 - s0
    return s1[:, np.newaxis] * f0 + s0[:, np.newaxis] * f1


def interp_twoways(x0, y0, xs, ys, fs, logy=False):
    if logy:
        f_interp = interp2d(np.array([np.log(y0)]), np.log(ys), fs)[0]
    else:
        f_interp = interp2d(np.array([y0]), ys, fs)[0]
    return np.interp(x0, xs, f_interp)


def get_kz_gazetas_v_lte_0p4(a0, lob):
    lobs = np.array([1.2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    kzs = np.array([
        [1, 1, 0.9885, 0.9677, 0.9216, 0.8732, 0.8147, 0.7504, 0.6791, 0.6001],
        [1, 1.012, 1.032, 1.042, 1.023, 0.986, 0.9348, 0.8665, 0.7861, 0.6985],
        [1, 1.069, 1.142, 1.199, 1.196, 1.135, 1.043, 0.9441, 0.8508, 0.7575],
        [1, 1.124, 1.225, 1.275, 1.252, 1.184, 1.094, 0.9916, 0.8906, 0.8016],
        [1, 1.231, 1.348, 1.356, 1.309, 1.233, 1.143, 1.047, 0.9523, 0.8579],
    ])
    return interp_twoways(a0, lob, a0s, lobs, kzs, logy=True)


def get_kz_gazetas_v_gt_0p4(a0, lob):
    lobs = np.array([1, 6, 1000])
    a0s = np.linspace(0, 2, 10)
    kzs = np.array([
        [1, 1.007, 0.9804, 0.9235, 0.853, 0.7575, 0.6508, 0.5295, 0.3859, 0.2301],
        [1, 1.03, 1.043, 0.9886, 0.8232, 0.567, 0.2662, -0.1174, -0.5485, -0.5485],
        [1, 1.227, 1.266, 1.117, 0.805, 0.3359, -0.2942, -0.7138, -0.7138, -0.7138],
        ])
    return interp_twoways(a0, lob, a0s, lobs, kzs, logy=True)


def get_ky_gazetas(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    kys = np.array([
        [1, 1.033, 1.035, 1.014, 0.9868, 0.9596, 0.9346, 0.9174, 0.9025, 0.8892],
        [1, 1.05, 1.087, 1.09, 1.09, 1.088, 1.083, 1.081, 1.083, 1.092],
        [1, 1.099, 1.165, 1.214, 1.248, 1.272, 1.28, 1.275, 1.264, 1.263],
        [1, 1.142, 1.245, 1.317, 1.366, 1.4, 1.418, 1.423, 1.423, 1.421],
        [1, 1.193, 1.344, 1.47, 1.539, 1.564, 1.572, 1.577, 1.575, 1.573],
        [1, 1.237, 1.485, 1.631, 1.699, 1.723, 1.726, 1.729, 1.728, 1.725],
        ])
    return interp_twoways(a0, lob, a0s, lobs, kys, logy=True)


def get_cz_gazetas_v_lte_0p4(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    czs = np.array([
        [0.9178, 0.9252, 0.9327, 0.9402, 0.9478, 0.9555, 0.9633, 0.9711, 0.9856, 1.001],
        [0.9716, 0.9784, 0.9852, 0.9921, 0.9988, 1.005, 1.011, 1.017, 1.025, 1.037],
        [1.109, 1.102, 1.095, 1.088, 1.08, 1.07, 1.061, 1.057, 1.059, 1.061],
        [1.486, 1.29, 1.174, 1.114, 1.085, 1.072, 1.063, 1.06, 1.06, 1.061],
        [1.773, 1.416, 1.252, 1.176, 1.143, 1.121, 1.106, 1.096, 1.092, 1.088],
        [2.003, 1.579, 1.344, 1.235, 1.183, 1.152, 1.132, 1.122, 1.117, 1.115],
    ])
    return interp_twoways(a0, lob, a0s, lobs, czs, logy=True)


def get_czf_gazetas_v_gt_0p4(a0, lob):
    lobs = np.array([1, 2, 4, 6, 1000])
    a0s = np.linspace(0, 2, 10)
    czfs = np.array([
        [1.038, 1.03, 1.022, 1.015, 1.009, 1.006, 1.003, 1.003, 1.004, 1.004],
        [1.038, 1.03, 1.022, 1.015, 1.009, 1.006, 1.003, 1.003, 1.004, 1.004],
        [1.077, 1.061, 1.047, 1.035, 1.028, 1.023, 1.019, 1.017, 1.017, 1.017],
        [1.12, 1.096, 1.077, 1.062, 1.052, 1.045, 1.04, 1.036, 1.034, 1.036],
        [1.223, 1.158, 1.11, 1.078, 1.062, 1.053, 1.047, 1.046, 1.046, 1.049],
        ])
    return interp_twoways(a0, lob, a0s, lobs, czfs, logy=True)


def get_cy_gazetas_v_e_0p3(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    cys = np.array([
        [0.867, 0.823, 0.7952, 0.7786, 0.7699, 0.7627, 0.7586, 0.7624, 0.7748, 0.8119],
        [0.9977, 0.9977, 0.9963, 0.9901, 0.9632, 0.9317, 0.8953, 0.8737, 0.8619, 0.8601],
        [1.19, 1.185, 1.164, 1.103, 1.038, 0.9992, 0.9608, 0.9327, 0.909, 0.922],
        [1.369, 1.337, 1.255, 1.102, 1.02, 0.9654, 0.9254, 0.8997, 0.89, 0.8876],
        [1.899, 1.537, 1.289, 1.112, 1.029, 0.9787, 0.9361, 0.9051, 0.8951, 0.8945],
        [2.422, 1.787, 1.398, 1.179, 1.064, 0.9984, 0.9547, 0.9207, 0.9022, 0.9151],
        ])
    return interp_twoways(a0, lob, a0s, lobs, cys, logy=True)


def get_cy_gazetas_v_e_0p5(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    cys = np.array([
        [0.867, 0.823, 0.7952, 0.7786, 0.7699, 0.7627, 0.7586, 0.7624, 0.7748, 0.8119],
        [0.9977, 0.9977, 0.9963, 0.9901, 0.9632, 0.9317, 0.8953, 0.8737, 0.8619, 0.8601],
        [1.19, 1.185, 1.164, 1.103, 1.038, 0.9992, 0.9608, 0.9327, 0.909, 0.922],
        [1.369, 1.337, 1.255, 1.102, 1.02, 0.9654, 0.9254, 0.8997, 0.89, 0.8876],
        [1.899, 1.537, 1.289, 1.112, 1.029, 0.9787, 0.9361, 0.9051, 0.8951, 0.8945],
        [3.007, 2.148, 1.57, 1.292, 1.15, 1.062, 1.023, 0.9965, 0.9647, 0.9358],
        ])
    return interp_twoways(a0, lob, a0s, lobs, cys, logy=True)


def get_crx_gazetas(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10])
    a0s = np.linspace(0, 2, 10)
    crxs = np.array([
        [0.001, 0.04695, 0.107, 0.1845, 0.2618, 0.3281, 0.3844, 0.4283, 0.4682, 0.5073],
        [0.001, 0.07291, 0.1533, 0.2436, 0.3237, 0.3839, 0.4322, 0.4749, 0.5138, 0.5523],
        [0.001, 0.09579, 0.192, 0.2875, 0.3575, 0.4117, 0.4584, 0.4976, 0.5355, 0.5814],
        [0.001, 0.1211, 0.2245, 0.3158, 0.3921, 0.4433, 0.4834, 0.523, 0.5611, 0.6017],
        [0.001, 0.1527, 0.2606, 0.3435, 0.4129, 0.47, 0.5168, 0.5536, 0.593, 0.6308],
        ])
    return interp_twoways(a0, lob, a0s, lobs, crxs, logy=True)


def get_cry_gazetas(a0, lob):
    lobs = np.array([1, 2, 4, 6, 10, 1000])
    a0s = np.linspace(0, 2, 10)
    crys = np.array([
        [-0.002241, 0.04273, 0.1082, 0.1805, 0.2693, 0.346, 0.4115, 0.4635, 0.5075, 0.5493],
        [0.005605, 0.1354, 0.2878, 0.4248, 0.5328, 0.6157, 0.6876, 0.7459, 0.7954, 0.8386],
        [0.00897, 0.5096, 0.7022, 0.7836, 0.8182, 0.85, 0.8825, 0.9124, 0.9444, 0.9798],
        [0.02242, 0.721, 0.905, 0.9614, 0.9652, 0.9741, 0.9849, 0.9965, 0.9982, 1],
        [0.02018, 0.8797, 0.9786, 0.9983, 1, 1, 1, 1, 1, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        ])
    return interp_twoways(a0, lob, a0s, lobs, crys, logy=True)


def get_ct_gazetas(a0, lob):
    lobs = np.array([1, 2, 3, 4, 6, 1000])
    a0s = np.linspace(0, 2, 10)
    cts = np.array([
        [0.001, 0.03851, 0.09381, 0.1696, 0.256, 0.3491, 0.4404, 0.5306, 0.621, 0.7171],
        [0.001, 0.09122, 0.2596, 0.4182, 0.5407, 0.6424, 0.7191, 0.7833, 0.8374, 0.8866],
        [0.001, 0.1896, 0.4323, 0.6588, 0.784, 0.8414, 0.8719, 0.8898, 0.9095, 0.9309],
        [0.001, 0.3115, 0.6502, 0.8543, 0.9193, 0.9532, 0.9651, 0.9658, 0.9677, 0.9713],
        [0.001, 0.5007, 0.853, 0.9474, 0.9834, 1, 1, 1, 1, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        ])
    return interp_twoways(a0, lob, a0s, lobs, cts, logy=True)


def show_plots():
    import matplotlib.pyplot as plt
    a0s = np.linspace(0, 2, 10)
    lobs = [0.5, 1, 4, 1000]
    lss = ['-', '--', ':', '-.']
    for i, lob in enumerate(lobs):
        f = []
        for a0 in a0s:
            # f.append(get_kz_gazetas_v_gt_0p4(a0, lob))
            f.append(get_crx_gazetas(a0, lob))
        plt.plot(a0s, f, label=str(lob), ls=lss[i])
    plt.legend()
    plt.show()


if __name__ == '__main__':
    # print(get_kz_gazetas_v_lte_0p4(1, 1.5))
    show_plots()
