import mediapipe as mp
import cv2
import numpy as np
import math
from google.protobuf.json_format import MessageToDict
mp_face_detection = mp.solutions.face_detection
mp_drawing = mp.solutions.drawing_utils
mp_pose = mp.solutions.pose
mp_drawing_styles = mp.solutions.drawing_styles


class HumanDetector():

    def __init__(self):
        
        self.mp_hands = mp.solutions.hands
        self.mp_drawing = mp.solutions.drawing_utils
        # Initialize MediaPipe Hands.
        self.hands = self.mp_hands.Hands(static_image_mode=True,max_num_hands=2,min_detection_confidence=0.4)
        self.mp_pose = mp.solutions.pose
        self.pose = self.mp_pose.Pose(min_detection_confidence=0.5, min_tracking_confidence=0.5)
        self.face_detection = mp_face_detection.FaceDetection(model_selection=0, min_detection_confidence=0.5)
        self.skeleton_detection = mp_pose.Pose(static_image_mode=True,model_complexity=2,enable_segmentation=True,min_detection_confidence=0.5)


        self.list_points = ["WRIST",
          "THUMB_CMC",
          "THUMB_MCP" ,
          "THUMB_IP",
          "THUMB_TIP",
          "INDEX_FINGER_MCP",
          "INDEX_FINGER_PIP",
          "INDEX_FINGER_DIP",
          "INDEX_FINGER_TIP",
          "MIDDLE_FINGER_MCP",
          "MIDDLE_FINGER_PIP",
          "MIDDLE_FINGER_DIP",
          "MIDDLE_FINGER_TIP",
          "RING_FINGER_MCP",
          "RING_FINGER_PIP",
          "RING_FINGER_DIP",
          "RING_FINGER_TIP",
          "PINKY_MCP",
          "PINKY_PIP",
          "PINKY_DIP",
          "PINKY_TIP"]

        self.dict_points = {"WRIST":{},"THUMB_CMC":{},"THUMB_MCP":{},
          "THUMB_IP":{},"THUMB_TIP":{},
          "INDEX_FINGER_MCP":{},"INDEX_FINGER_PIP":{},
          "INDEX_FINGER_DIP":{},"INDEX_FINGER_TIP":{},
          "MIDDLE_FINGER_MCP":{},"MIDDLE_FINGER_PIP":{},
          "MIDDLE_FINGER_DIP":{},"MIDDLE_FINGER_TIP":{},
          "RING_FINGER_MCP":{},"RING_FINGER_PIP":{},
          "RING_FINGER_DIP":{},"RING_FINGER_TIP":{},
          "PINKY_MCP":{},"PINKY_PIP":{},
          "PINKY_DIP":{},"PINKY_TIP":{}}

    def getBody(self,image_):
        image = image_.copy()
        image.flags.writeable = False
        image = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
        results = self.skeleton_detection.process(image)
        y_resolution, x_resolution, c = image.shape

        # Draw the pose annotation on the image.
        image.flags.writeable = True
        image = cv2.cvtColor(image, cv2.COLOR_RGB2BGR)
        mp_drawing.draw_landmarks(
            image,
            results.pose_landmarks,
            mp_pose.POSE_CONNECTIONS,
            landmark_drawing_spec=mp_drawing_styles.get_default_pose_landmarks_style())
        # Flip the image horizontally for a selfie-view display.

        list_pose = ["NOSE",
            "LEFT_EYE_INNER",
            "LEFT_EYE",
            "LEFT_EYE_OUTER",
            "RIGHT_EYE_INNER",
            "RIGHT_EYE",
            "RIGHT_EYE_OUTER",
            "LEFT_EAR",
            "RIGHT_EAR",
            "MOUTH_LEFT",
            "MOUTH_RIGHT",
            "LEFT_SHOULDER",
            "RIGHT_SHOULDER",
            "LEFT_ELBOW",
            "RIGHT_ELBOW",
            "LEFT_WRIST",
            "RIGHT_WRIST",
            "LEFT_PINKY",
            "RIGHT_PINKY",
            "LEFT_INDEX",
            "RIGHT_INDEX",
            "LEFT_THUMB",
            "RIGHT_THUMB",
            "LEFT_HIP",
            "RIGHT_HIP",
            "LEFT_KNEE",
            "RIGHT_KNEE",
            "LEFT_ANKLE",
            "RIGHT_ANKLE",
            "LEFT_HEEL",
            "RIGHT_HEEL",
            "LEFT_FOOT_INDEX",
            "RIGHT_FOOT_INDEX"]

        init_value = {"x":0,"y":0, "detected":False}
        pose_values = {};

        for element in list_pose:
            pose_values[element] = init_value
            #print(element)
            try:
                value = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark[element]]
                value.x, value.y = self.normalized_to_pixel_coordinates(value.x, value.y, x_resolution,y_resolution)
                pose_values[element] = {"x":value.x,"y":value.y, "detected":True}
            except:
                pass
    
        return cv2.flip(image, 1), pose_values

    def getFace(self,image_):
        image = image_.copy()
        image.flags.writeable = False
        image = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
        results = self.face_detection.process(image)
        y_resolution, x_resolution, c = image.shape
     
        image.flags.writeable = True
        image = cv2.cvtColor(image, cv2.COLOR_RGB2BGR)
        faces = {}
        if results.detections:
            for detection in results.detections:
                data = detection.location_data
                xmin = data.relative_bounding_box.xmin
                ymin = data.relative_bounding_box.ymin
                width = data.relative_bounding_box.width
                height= data.relative_bounding_box.height
                xmin, ymin = self.normalized_to_pixel_coordinates(xmin, ymin, x_resolution,y_resolution)
                width, height = self.normalized_to_pixel_coordinates(width, height, x_resolution,y_resolution)

                faces= {detection.label_id[0]:{"xmin":xmin, "ymin":ymin, "width":width, "height":height}}

                #print(detection.label_id[0])
                #print(data.relative_bounding_box.xmin)
                
   

                mp_drawing.draw_detection(image, detection)
        
        return cv2.flip(image, 1), faces


    def getDistance(self,elementA,elementB):
        pixelX = elementA.x - elementB.x
        pixelY = elementA.y - elementB.y

        
        distance = (pixelX**2 + pixelY**2)**.5
        
        return distance
    

    def getHandFromPoseIndexes(self,wrist, elbow, shoulder, threshold = 0.03):

        
        ratioWristElbow = 0.33
        handRectangle_x = wrist.x + ratioWristElbow * (wrist.x - elbow.x)
        handRectangle_y = wrist.y + ratioWristElbow * (wrist.y - elbow.y)
        distanceWristElbow = self.getDistance(wrist, elbow)
        distanceElbowShoulder = self.getDistance(elbow, shoulder)
        handRectangle_width = 1.5 * max(distanceWristElbow, 0.9 * distanceElbowShoulder)



        handRectangle_height = handRectangle_width;
        #x-y refers to the center --> offset to topLeft point
        handRectangle_x = handRectangle_x - handRectangle_width / 2.0
        handRectangle_y = handRectangle_y - handRectangle_height / 2.0


        return int(handRectangle_x), int(handRectangle_y), int(handRectangle_width), int(handRectangle_height)


    def normalized_to_pixel_coordinates(self, normalized_x, normalized_y, image_width,image_height):

      x_px = min(math.floor(normalized_x * image_width), image_width - 1)
      y_px = min(math.floor(normalized_y * image_height), image_height - 1)
      
      return x_px, y_px

        

    def get2DBodyPose(self, image):
        # To improve performance, optionally mark the image as not writeable to
        # pass by reference.
        y_resolution, x_resolution, c = image.shape
        image.flags.writeable = False
        results = self.pose.process(image)
        rigth_value = {}
        left_value = {}
        body_pose = {}

        # Draw the pose annotation on the image.
        #image.flags.writeable = True
        #new_image = cv2.cvtColor(image, cv2.COLOR_RGB2BGR)
        #self.mp_drawing.draw_landmarks(new_image, results.pose_landmarks, self.mp_pose.POSE_CONNECTIONS)


        is_l = False
        is_r = False

      
        try:

            RIGHT_WRIST = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_WRIST]
            RIGHT_WRIST.x, RIGHT_WRIST.y = self.normalized_to_pixel_coordinates(RIGHT_WRIST.x, RIGHT_WRIST.y, x_resolution,y_resolution)

            RIGHT_ELBOW = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_ELBOW]
            RIGHT_ELBOW.x, RIGHT_ELBOW.y = self.normalized_to_pixel_coordinates(RIGHT_ELBOW.x, RIGHT_ELBOW.y, x_resolution,y_resolution)
            

            RIGHT_SHOULDER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_SHOULDER]
            RIGHT_SHOULDER.x, RIGHT_SHOULDER.y = self.normalized_to_pixel_coordinates(RIGHT_SHOULDER.x, RIGHT_SHOULDER.y, x_resolution,y_resolution)

            is_r = True
            
        except:
            print("Not complete right hand")
            pass
        
        
        try:
            LEFT_WRIST = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_WRIST]
            LEFT_WRIST.x, LEFT_WRIST.y = self.normalized_to_pixel_coordinates(LEFT_WRIST.x, LEFT_WRIST.y, x_resolution,y_resolution)

            LEFT_ELBOW = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_ELBOW]
            LEFT_ELBOW.x, LEFT_ELBOW.y = self.normalized_to_pixel_coordinates(LEFT_ELBOW.x, LEFT_ELBOW.y, x_resolution,y_resolution)
 
            LEFT_SHOULDER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_SHOULDER]
            LEFT_SHOULDER.x, LEFT_SHOULDER.y = self.normalized_to_pixel_coordinates(LEFT_SHOULDER.x, LEFT_SHOULDER.y, x_resolution,y_resolution)

            is_l = True
        except:
            print("Not complete left hand")
            pass


        try:

            NOSE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.NOSE]
            NOSE.x, NOSE.y = self.normalized_to_pixel_coordinates(NOSE.x, NOSE.y, x_resolution,y_resolution)

            LEFT_EYE_INNER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_EYE_INNER]
            LEFT_EYE_INNER.x, LEFT_EYE_INNER.y = self.normalized_to_pixel_coordinates(LEFT_EYE_INNER.x, LEFT_EYE_INNER.y, x_resolution,y_resolution)

            LEFT_EYE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_EYE]
            LEFT_EYE.x, LEFT_EYE.y = self.normalized_to_pixel_coordinates(LEFT_EYE.x, LEFT_EYE.y, x_resolution,y_resolution)

            LEFT_EYE_OUTER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_EYE_OUTER]
            LEFT_EYE_OUTER.x, LEFT_EYE_OUTER.y = self.normalized_to_pixel_coordinates(LEFT_EYE_OUTER.x, LEFT_EYE_OUTER.y, x_resolution,y_resolution)

            
            RIGHT_EYE_INNER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_EYE_INNER]
            RIGHT_EYE_INNER.x, RIGHT_EYE_INNER.y = self.normalized_to_pixel_coordinates(RIGHT_EYE_INNER.x, RIGHT_EYE_INNER.y, x_resolution,y_resolution)

            RIGHT_EYE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_EYE]
            RIGHT_EYE.x, RIGHT_EYE.y = self.normalized_to_pixel_coordinates(RIGHT_EYE.x, RIGHT_EYE.y, x_resolution,y_resolution)

            
            RIGHT_EYE_OUTER = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_EYE_OUTER]
            RIGHT_EYE_OUTER.x, RIGHT_EYE_OUTER.y = self.normalized_to_pixel_coordinates(RIGHT_EYE_OUTER.x, RIGHT_EYE_OUTER.y, x_resolution,y_resolution)

                        
            LEFT_EAR = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_EAR]
            LEFT_EAR.x, LEFT_EAR.y = self.normalized_to_pixel_coordinates(LEFT_EAR.x, LEFT_EAR.y, x_resolution,y_resolution)

      
            RIGHT_EAR = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_EAR]
            RIGHT_EAR.x, RIGHT_EAR.y = self.normalized_to_pixel_coordinates(RIGHT_EAR.x, RIGHT_EAR.y, x_resolution,y_resolution)

            MOUTH_LEFT = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.MOUTH_LEFT]
            MOUTH_LEFT.x, MOUTH_LEFT.y = self.normalized_to_pixel_coordinates(MOUTH_LEFT.x, MOUTH_LEFT.y, x_resolution,y_resolution)

            MOUTH_RIGHT = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.MOUTH_RIGHT]
            MOUTH_RIGHT.x, MOUTH_RIGHT.y = self.normalized_to_pixel_coordinates(MOUTH_RIGHT.x, MOUTH_RIGHT.y, x_resolution,y_resolution)

            LEFT_HIP = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_HIP]
            LEFT_HIP.x, LEFT_HIP.y = self.normalized_to_pixel_coordinates(LEFT_HIP.x, LEFT_HIP.y, x_resolution,y_resolution)

            RIGHT_HIP = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_HIP]
            RIGHT_HIP.x, RIGHT_HIP.y = self.normalized_to_pixel_coordinates(RIGHT_HIP.x, RIGHT_HIP.y, x_resolution,y_resolution)

            LEFT_PINKY = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_PINKY]
            LEFT_PINKY.x, LEFT_PINKY.y = self.normalized_to_pixel_coordinates(LEFT_PINKY.x, LEFT_PINKY.y, x_resolution,y_resolution)

            RIGHT_PINKY = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_PINKY]
            RIGHT_PINKY.x, RIGHT_PINKY.y = self.normalized_to_pixel_coordinates(RIGHT_PINKY.x, RIGHT_PINKY.y, x_resolution,y_resolution)

            LEFT_INDEX = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_INDEX]
            LEFT_INDEX.x, LEFT_INDEX.y = self.normalized_to_pixel_coordinates(LEFT_INDEX.x, LEFT_INDEX.y, x_resolution,y_resolution)

            RIGHT_INDEX = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_INDEX]
            RIGHT_INDEX.x, RIGHT_INDEX.y = self.normalized_to_pixel_coordinates(RIGHT_INDEX.x, RIGHT_INDEX.y, x_resolution,y_resolution)

            
            LEFT_THUMB = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_THUMB]
            LEFT_THUMB.x, LEFT_THUMB.y = self.normalized_to_pixel_coordinates(LEFT_THUMB.x, LEFT_THUMB.y, x_resolution,y_resolution)

            RIGHT_THUMB = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_THUMB]
            RIGHT_THUMB.x, RIGHT_THUMB.y = self.normalized_to_pixel_coordinates(RIGHT_THUMB.x, RIGHT_THUMB.y, x_resolution,y_resolution)

            LEFT_HIP = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_HIP]
            LEFT_HIP.x, LEFT_HIP.y = self.normalized_to_pixel_coordinates(LEFT_HIP.x, LEFT_HIP.y, x_resolution,y_resolution)

            RIGHT_HIP = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_HIP]
            RIGHT_HIP.x, RIGHT_HIP.y = self.normalized_to_pixel_coordinates(RIGHT_HIP.x, RIGHT_HIP.y, x_resolution,y_resolution)

            LEFT_KNEE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_KNEE]
            LEFT_KNEE.x, LEFT_KNEE.y = self.normalized_to_pixel_coordinates(LEFT_KNEE.x, LEFT_KNEE.y, x_resolution,y_resolution)

            RIGHT_KNEE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_KNEE]
            RIGHT_KNEE.x, RIGHT_KNEE.y = self.normalized_to_pixel_coordinates(RIGHT_KNEE.x, RIGHT_KNEE.y, x_resolution,y_resolution)

            LEFT_ANKLE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_ANKLE]
            LEFT_ANKLE.x, LEFT_ANKLE.y = self.normalized_to_pixel_coordinates(LEFT_ANKLE.x, LEFT_ANKLE.y, x_resolution,y_resolution)

            RIGHT_ANKLE = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_ANKLE]
            RIGHT_ANKLE.x, RIGHT_ANKLE.y = self.normalized_to_pixel_coordinates(RIGHT_ANKLE.x, RIGHT_ANKLE.y, x_resolution,y_resolution)

            LEFT_HEEL = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_HEEL]
            LEFT_HEEL.x, LEFT_HEEL.y = self.normalized_to_pixel_coordinates(LEFT_HEEL.x, LEFT_HEEL.y, x_resolution,y_resolution)

            RIGHT_HEEL = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_HEEL]
            RIGHT_HEEL.x, RIGHT_HEEL.y = self.normalized_to_pixel_coordinates(RIGHT_HEEL.x, RIGHT_HEEL.y, x_resolution,y_resolution)

            LEFT_FOOT_INDEX = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.LEFT_FOOT_INDEX]
            LEFT_FOOT_INDEX.x, LEFT_FOOT_INDEX.y = self.normalized_to_pixel_coordinates(LEFT_FOOT_INDEX.x, LEFT_FOOT_INDEX.y, x_resolution,y_resolution)

            RIGHT_FOOT_INDEX = results.pose_landmarks.landmark[self.mp_pose.PoseLandmark.RIGHT_FOOT_INDEX]
            RIGHT_FOOT_INDEX.x, RIGHT_FOOT_INDEX.y = self.normalized_to_pixel_coordinates(RIGHT_FOOT_INDEX.x, RIGHT_FOOT_INDEX.y, x_resolution,y_resolution)

        except:
            print("Not complete left hand")
            pass




        if is_l == True and is_r == True:


            x,y,w,h = self.getHandFromPoseIndexes(RIGHT_WRIST, RIGHT_ELBOW, RIGHT_SHOULDER)
            rigth_value = {"x":x, "y":y, "w":w, "h":h}

            RIGHT_WRIST = {"x":RIGHT_WRIST.x,"y":RIGHT_WRIST.y}
            RIGHT_ELBOW = {"x":RIGHT_ELBOW.x,"y":RIGHT_ELBOW.y}
            RIGHT_SHOULDER = {"x":RIGHT_SHOULDER.x,"y":RIGHT_SHOULDER.y}
        
            #cv2.rectangle(image,(x,y),(x+w,y+h), (255,0,0), 2)
            #cv2.imshow("hand_rect", image)


            x,y,w,h = self.getHandFromPoseIndexes(LEFT_WRIST, LEFT_ELBOW, LEFT_SHOULDER)
            left_value = {"x":x, "y":y, "w":w, "h":h}


            LEFT_WRIST = {"x":LEFT_WRIST.x,"y":LEFT_WRIST.y}
            LEFT_ELBOW = {"x":LEFT_ELBOW.x,"y":LEFT_ELBOW.y}
            LEFT_SHOULDER = {"x":LEFT_SHOULDER.x,"y":LEFT_SHOULDER.y}
        
            body_pose = {"RIGHT_WRIST":RIGHT_WRIST, "RIGHT_ELBOW":RIGHT_ELBOW, "RIGHT_SHOULDER":RIGHT_SHOULDER, "LEFT_WRIST":LEFT_WRIST, "LEFT_ELBOW":LEFT_ELBOW, "LEFT_SHOULDER":LEFT_SHOULDER, "LEFT_HIP":LEFT_HIP, "RIGHT_HIP":RIGHT_HIP, "NOSE":NOSE,
            LEFT_EYE_INNER:"LEFT_EYE_INNER",   "LEFT_PINKY":LEFT_PINKY, "RIGHT_PINKY":RIGHT_PINKY, "LEFT_INDEX":LEFT_INDEX, "RIGHT_INDEX":RIGHT_INDEX, "LEFT_THUMB":LEFT_THUMB,
            "RIGHT_THUMB" :RIGHT_THUMB,"LEFT_HIP":LEFT_HIP,"RIGHT_HIP":RIGHT_HIP,"LEFT_KNEE":LEFT_KNEE,"RIGHT_KNEE":RIGHT_KNEE,"LEFT_ANKLE":LEFT_ANKLE,"RIGHT_ANKLE":RIGHT_ANKLE,"LEFT_HEEL":LEFT_HEEL,"RIGHT_HEEL":RIGHT_HEEL,
            "LEFT_FOOT_INDEX":LEFT_FOOT_INDEX, "RIGHT_FOOT_INDEX":RIGHT_FOOT_INDEX, "NOSE":NOSE,"LEFT_EYE_INNER":LEFT_EYE_INNER,"LEFT_EYE":LEFT_EYE,"LEFT_EYE_OUTER":LEFT_EYE_OUTER,"RIGHT_EYE_INNER":RIGHT_EYE_INNER,
            "RIGHT_EYE":RIGHT_EYE,"RIGHT_EYE_OUTER":RIGHT_EYE_OUTER,"LEFT_EAR":LEFT_EAR,"RIGHT_EAR":RIGHT_EAR,"MOUTH_LEFT":MOUTH_LEFT,"MOUTH_RIGHT":MOUTH_RIGHT}
     

        return rigth_value, left_value, body_pose


        

    def get2Dhands(self,image_):

        image = image_.copy()
        y_resolution, x_resolution, c = image.shape
        # Convert the BGR image to RGB, flip the image around y-axis for correct 
        # handedness output and process it with MediaPipe Hands.
        results = self.hands.process(cv2.flip(cv2.cvtColor(image, cv2.COLOR_BGR2RGB), 1))
        # Print handedness (left v.s. right hand).
        # print(results.multi_handedness)

        
        self.data = {"Right": self.dict_points.copy(),"Left": self.dict_points.copy(), "l_data":False ,"r_data":False}

        annotated_image = cv2.flip(image.copy(), 1)
        k = 0
        hand_ = "left"

        
        if results.multi_hand_landmarks:

            hand_dict = []
            for idx, hand_handedness in enumerate(results.multi_handedness):
                handedness_dict = MessageToDict(hand_handedness)
                hand_dict.append(handedness_dict)

            
            for hand_landmarks in results.multi_hand_landmarks:
                j = 0
                for data_point in hand_landmarks.landmark:

                    hand_ = hand_dict[k]["classification"][0]["label"]
                    hand_value = hand_.split("\r")
                    if hand_value[0] == "Left":
                        self.data["l_data"] = True
                    else:
                        self.data["r_data"] = True
                        
                        
                    self.data[hand_value[0]][self.list_points[j]] = {'x': data_point.x*x_resolution,'y': data_point.y*y_resolution }
                    j = j + 1

                self.mp_drawing.draw_landmarks(annotated_image, hand_landmarks, self.mp_hands.HAND_CONNECTIONS)
                k = k + 1
                
        return annotated_image, self.data
 
