"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationSource = exports.ConfigurationContent = exports.LambdaValidator = exports.JsonSchemaValidator = exports.ConfigurationSourceType = exports.ValidatorType = exports.ConfigurationType = exports.SourcedConfiguration = exports.HostedConfiguration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const mimeTypes = require("mime-types");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const deployment_strategy_1 = require("./deployment-strategy");
const extension_1 = require("./extension");
const hash_1 = require("./private/hash");
class ConfigurationBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 128,
            separator: '-',
        });
        this.application = props.application;
        this.applicationId = this.application.applicationId;
        this.type = props.type;
        this.validators = props.validators;
        this.description = props.description;
        this.deployTo = props.deployTo;
        this.deploymentKey = props.deploymentKey;
        this.deploymentStrategy = props.deploymentStrategy || new deployment_strategy_1.DeploymentStrategy(this, 'DeploymentStrategy', {
            rolloutStrategy: deployment_strategy_1.RolloutStrategy.CANARY_10_PERCENT_20_MINUTES,
        });
    }
    /**
     * Adds an extension defined by the action point and event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param actionPoint The action point which triggers the event
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    /**
     * Adds a PRE_CREATE_HOSTED_CONFIGURATION_VERSION extension with the
     * provided event destination and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    /**
     * Adds a PRE_START_DEPLOYMENT extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_START extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_STEP extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_BAKING extension with the provided event destination and
     * also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_COMPLETE extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_ROLLED_BACK extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    /**
     * Adds an extension association to the configuration profile.
     *
     * @param extension The extension to create an association for
     */
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
    addExistingEnvironmentsToApplication() {
        this.deployTo?.forEach((environment) => {
            if (!this.application.environments.includes(environment)) {
                this.application.addExistingEnvironment(environment);
            }
        });
    }
    deployConfigToEnvironments() {
        if (!this.deployTo || !this.versionNumber) {
            return;
        }
        this.application.environments.forEach((environment) => {
            if ((this.deployTo && !this.deployTo.includes(environment))) {
                return;
            }
            const logicalId = `Deployment${this.getDeploymentHash(environment)}`;
            new aws_appconfig_1.CfnDeployment(this, logicalId, {
                applicationId: this.application.applicationId,
                configurationProfileId: this.configurationProfileId,
                deploymentStrategyId: this.deploymentStrategy.deploymentStrategyId,
                environmentId: environment.environmentId,
                configurationVersion: this.versionNumber,
                description: this.description,
                kmsKeyIdentifier: this.deploymentKey?.keyArn,
            });
        });
    }
}
class HostedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_HostedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, HostedConfiguration);
            }
            throw error;
        }
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'ConfigurationProfile', {
            applicationId: this.applicationId,
            locationUri: 'hosted',
            name: this.name,
            description: this.description,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.content = props.content.content;
        this.contentType = props.content.contentType;
        this.latestVersionNumber = props.latestVersionNumber;
        this.versionLabel = props.versionLabel;
        this._cfnHostedConfigurationVersion = new aws_appconfig_1.CfnHostedConfigurationVersion(this, 'Resource', {
            applicationId: this.applicationId,
            configurationProfileId: this.configurationProfileId,
            content: this.content,
            contentType: this.contentType,
            description: this.description,
            latestVersionNumber: this.latestVersionNumber,
            versionLabel: this.versionLabel,
        });
        this._cfnHostedConfigurationVersion.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
        this.versionNumber = this._cfnHostedConfigurationVersion.ref;
        this.hostedConfigurationVersionArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}/hostedconfigurationversion/${this.versionNumber}`,
        });
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.content}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
}
exports.HostedConfiguration = HostedConfiguration;
_a = JSII_RTTI_SYMBOL_1;
HostedConfiguration[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.HostedConfiguration", version: "2.98.0-alpha.0" };
class SourcedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_SourcedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SourcedConfiguration);
            }
            throw error;
        }
        this.location = props.location;
        this.locationUri = this.location.locationUri;
        this.versionNumber = props.versionNumber;
        this.sourceKey = this.location.key;
        this.retrievalRole = props.retrievalRole || this.location.type != ConfigurationSourceType.CODE_PIPELINE
            ? new iam.Role(this, 'Role', {
                roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('appconfig.amazonaws.com'),
                inlinePolicies: {
                    ['AllowAppConfigReadFromSourcePolicy']: this.getPolicyForRole(),
                },
            })
            : undefined;
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'Resource', {
            applicationId: this.applicationId,
            locationUri: this.locationUri,
            name: this.name,
            description: this.description,
            retrievalRoleArn: this.retrievalRole?.roleArn,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.versionNumber}
      ${this.location.type}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
    getPolicyForRole() {
        const policy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
        });
        const document = new iam.PolicyDocument({
            statements: [policy],
        });
        if (this.location.type == ConfigurationSourceType.SSM_PARAMETER) {
            policy.addActions('ssm:GetParameter');
            policy.addResources(this.locationUri);
        }
        else if (this.location.type == ConfigurationSourceType.SSM_DOCUMENT) {
            policy.addActions('ssm:GetDocument');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                service: 'ssm',
                resource: 'document',
                resourceName: this.locationUri.split('://')[1],
            }));
        }
        else if (this.location.type == ConfigurationSourceType.S3) {
            const bucketAndObjectKey = this.locationUri.split('://')[1];
            const sep = bucketAndObjectKey.search('/');
            const bucketName = bucketAndObjectKey.substring(0, sep);
            const objectKey = bucketAndObjectKey.substring(sep + 1);
            policy.addActions('s3:GetObject', 's3:GetObjectMetadata', 's3:GetObjectVersion');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                region: '',
                account: '',
                service: 's3',
                arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                resource: `${bucketName}/${objectKey}`,
            }));
            const bucketPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: [
                    's3:GetBucketLocation',
                    's3:GetBucketVersioning',
                    's3:ListBucket',
                    's3:ListBucketVersions',
                ],
                resources: [
                    aws_cdk_lib_1.Stack.of(this).formatArn({
                        region: '',
                        account: '',
                        service: 's3',
                        arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                        resource: bucketName,
                    }),
                ],
            });
            const listBucketsPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:ListAllMyBuckets'],
                resources: ['*'],
            });
            document.addStatements(bucketPolicy, listBucketsPolicy);
        }
        else {
            policy.addActions('secretsmanager:GetSecretValue');
            policy.addResources(this.locationUri);
        }
        if (this.sourceKey) {
            const keyPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['kms:Decrypt'],
                resources: [this.sourceKey.keyArn],
            });
            document.addStatements(keyPolicy);
        }
        return document;
    }
}
exports.SourcedConfiguration = SourcedConfiguration;
_b = JSII_RTTI_SYMBOL_1;
SourcedConfiguration[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.SourcedConfiguration", version: "2.98.0-alpha.0" };
/**
 * The configuration type.
 */
var ConfigurationType;
(function (ConfigurationType) {
    ConfigurationType["FREEFORM"] = "AWS.Freeform";
    ConfigurationType["FEATURE_FLAGS"] = "AWS.AppConfig.FeatureFlags";
})(ConfigurationType || (exports.ConfigurationType = ConfigurationType = {}));
/**
 * The validator type.
 */
var ValidatorType;
(function (ValidatorType) {
    ValidatorType["JSON_SCHEMA"] = "JSON_SCHEMA";
    ValidatorType["LAMBDA"] = "LAMBDA";
})(ValidatorType || (exports.ValidatorType = ValidatorType = {}));
/**
 * The configuration source type.
 */
var ConfigurationSourceType;
(function (ConfigurationSourceType) {
    ConfigurationSourceType["S3"] = "S3";
    ConfigurationSourceType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ConfigurationSourceType["SSM_PARAMETER"] = "SSM_PARAMETER";
    ConfigurationSourceType["SSM_DOCUMENT"] = "SSM_DOCUMENT";
    ConfigurationSourceType["CODE_PIPELINE"] = "CODE_PIPELINE";
})(ConfigurationSourceType || (exports.ConfigurationSourceType = ConfigurationSourceType = {}));
/**
 * Defines a JSON Schema validator.
 */
class JsonSchemaValidator {
    /**
     * Defines a JSON Schema validator from a file.
     *
     * @param path The path to the file that defines the validator
     */
    static fromFile(path) {
        return {
            content: fs.readFileSync(path).toString(),
            type: ValidatorType.JSON_SCHEMA,
        };
    }
    /**
     * Defines a JSON Schema validator from inline code.
     *
     * @param code The inline code that defines the validator
     */
    static fromInline(code) {
        return {
            content: code,
            type: ValidatorType.JSON_SCHEMA,
        };
    }
}
exports.JsonSchemaValidator = JsonSchemaValidator;
_c = JSII_RTTI_SYMBOL_1;
JsonSchemaValidator[_c] = { fqn: "@aws-cdk/aws-appconfig-alpha.JsonSchemaValidator", version: "2.98.0-alpha.0" };
/**
 * Defines an AWS Lambda validator.
 */
class LambdaValidator {
    /**
     *  Defines an AWS Lambda validator from a Lambda function. This will call
     * `addPermission` to your function to grant AWS AppConfig permissions.
     *
     * @param func The function that defines the validator
     */
    static fromFunction(func) {
        if (!func.permissionsNode.tryFindChild('AppConfigPermission')) {
            func.addPermission('AppConfigPermission', {
                principal: new iam.ServicePrincipal('appconfig.amazonaws.com'),
            });
        }
        return {
            content: func.functionArn,
            type: ValidatorType.LAMBDA,
        };
    }
}
exports.LambdaValidator = LambdaValidator;
_d = JSII_RTTI_SYMBOL_1;
LambdaValidator[_d] = { fqn: "@aws-cdk/aws-appconfig-alpha.LambdaValidator", version: "2.98.0-alpha.0" };
/**
 * Defines the hosted configuration content.
 */
class ConfigurationContent {
    /**
     * Defines the hosted configuration content from a file.
     *
     * @param path The path to the file that defines configuration content
     * @param contentType The content type of the configuration
     */
    static fromFile(path, contentType) {
        return {
            content: fs.readFileSync(path).toString(),
            contentType: contentType || mimeTypes.lookup(path) || 'application/json',
        };
    }
    /**
     * Defines the hosted configuration content from inline code.
     *
     * @param content The inline code that defines the configuration content
     * @param contentType The content type of the configuration
     */
    static fromInline(content, contentType) {
        return {
            content,
            contentType: contentType || 'application/octet-stream',
        };
    }
    /**
     * Defines the hosted configuration content as JSON from inline code.
     *
     * @param content The inline code that defines the configuration content
     * @param contentType The content type of the configuration
     */
    static fromInlineJson(content, contentType) {
        return {
            content,
            contentType: contentType || 'application/json',
        };
    }
    /**
     * Defines the hosted configuration content as text from inline code.
     *
     * @param content The inline code that defines the configuration content
     */
    static fromInlineText(content) {
        return {
            content,
            contentType: 'text/plain',
        };
    }
}
exports.ConfigurationContent = ConfigurationContent;
_e = JSII_RTTI_SYMBOL_1;
ConfigurationContent[_e] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationContent", version: "2.98.0-alpha.0" };
/**
 * Defines the integrated configuration sources.
 */
class ConfigurationSource {
    /**
     * Defines configuration content from an Amazon S3 bucket.
     *
     * @param bucket The S3 bucket where the configuration is stored
     * @param objectKey The path to the configuration
     * @param key The KMS Key that the bucket is encrypted with
     */
    static fromBucket(bucket, objectKey, key) {
        return {
            locationUri: bucket.s3UrlForObject(objectKey),
            type: ConfigurationSourceType.S3,
            key,
        };
    }
    /**
     * Defines configuration content from an AWS Secrets Manager secret.
     *
     * @param secret The secret where the configuration is stored
     */
    static fromSecret(secret) {
        return {
            locationUri: secret.secretArn,
            type: ConfigurationSourceType.SECRETS_MANAGER,
            key: secret.encryptionKey,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) Parameter Store parameter.
     *
     * @param parameter The parameter where the configuration is stored
     * @param key The KMS Key that the secure string is encrypted with
     */
    static fromParameter(parameter, key) {
        return {
            locationUri: parameter.parameterArn,
            type: ConfigurationSourceType.SSM_PARAMETER,
            key,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) document.
     *
     * @param document The SSM document where the configuration is stored
     */
    static fromCfnDocument(document) {
        return {
            locationUri: `ssm-document://${document.ref}`,
            type: ConfigurationSourceType.SSM_DOCUMENT,
        };
    }
    /**
     * Defines configuration content from AWS CodePipeline.
     *
     * @param pipeline The pipeline where the configuration is stored
     * @returns
     */
    static fromPipeline(pipeline) {
        return {
            locationUri: `codepipeline://${pipeline.pipelineName}`,
            type: ConfigurationSourceType.CODE_PIPELINE,
        };
    }
}
exports.ConfigurationSource = ConfigurationSource;
_f = JSII_RTTI_SYMBOL_1;
ConfigurationSource[_f] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationSource", version: "2.98.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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