"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RolloutStrategy = exports.PredefinedDeploymentStrategyId = exports.GrowthType = exports.DeploymentStrategy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
/**
 * An AWS AppConfig deployment strategy.
 *
 * @resource AWS::AppConfig::DeploymentStrategy
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html
 */
class DeploymentStrategy extends aws_cdk_lib_1.Resource {
    /**
     * Imports a deployment strategy into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyArn The Amazon Resource Name (ARN) of the deployment strategy
     */
    static fromDeploymentStrategyArn(scope, id, deploymentStrategyArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(deploymentStrategyArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        const deploymentStrategyId = parsedArn.resourceName;
        if (!deploymentStrategyId) {
            throw new Error('Missing required deployment strategy id from deployment strategy ARN');
        }
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.deploymentStrategyId = deploymentStrategyId;
                this.deploymentStrategyArn = deploymentStrategyArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: deploymentStrategyArn,
        });
    }
    /**
     * Imports a deployment strategy into the CDK using its ID.
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyId The ID of the deployment strategy
     */
    static fromDeploymentStrategyId(scope, id, deploymentStrategyId) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const deploymentStrategyArn = stack.formatArn({
            service: 'appconfig',
            resource: 'deploymentstrategy',
            resourceName: deploymentStrategyId,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.deploymentStrategyId = deploymentStrategyId;
                this.deploymentStrategyArn = deploymentStrategyArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: deploymentStrategyArn,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_DeploymentStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DeploymentStrategy);
            }
            throw error;
        }
        this.deploymentDurationInMinutes = props.rolloutStrategy.deploymentDuration.toMinutes();
        this.growthFactor = props.rolloutStrategy.growthFactor;
        this.description = props.description;
        this.finalBakeTimeInMinutes = props.rolloutStrategy.finalBakeTime?.toMinutes();
        this.growthType = props.rolloutStrategy.growthType;
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 64,
            separator: '-',
        });
        const resource = new aws_appconfig_1.CfnDeploymentStrategy(this, 'Resource', {
            name: this.name,
            deploymentDurationInMinutes: this.deploymentDurationInMinutes,
            growthFactor: this.growthFactor,
            replicateTo: 'NONE',
            description: this.description,
            finalBakeTimeInMinutes: this.finalBakeTimeInMinutes,
            growthType: this.growthType,
        });
        this._cfnDeploymentStrategy = resource;
        this.deploymentStrategyId = this._cfnDeploymentStrategy.ref;
        this.deploymentStrategyArn = this.stack.formatArn({
            service: 'appconfig',
            resource: 'deploymentstrategy',
            resourceName: this.deploymentStrategyId,
        });
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_a = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.DeploymentStrategy", version: "2.98.0-alpha.0" };
/**
 * Defines the growth type of the deployment strategy.
 */
var GrowthType;
(function (GrowthType) {
    GrowthType["LINEAR"] = "LINEAR";
    GrowthType["EXPONENTIAL"] = "EXPONENTIAL";
})(GrowthType || (exports.GrowthType = GrowthType = {}));
/**
 * Defines the deployment strategy ID's of AWS AppConfig predefined strategies.
 */
var PredefinedDeploymentStrategyId;
(function (PredefinedDeploymentStrategyId) {
    PredefinedDeploymentStrategyId["CANARY_10_PERCENT_20_MINUTES"] = "AppConfig.Canary10Percent20Minutes";
    PredefinedDeploymentStrategyId["LINEAR_50_PERCENT_EVERY_30_SECONDS"] = "AppConfig.Linear50PercentEvery30Seconds";
    PredefinedDeploymentStrategyId["LINEAR_20_PERCENT_EVERY_6_MINUTES"] = "AppConfig.Linear20PercentEvery6Minutes";
    PredefinedDeploymentStrategyId["ALL_AT_ONCE"] = "AppConfig.AllAtOnce";
})(PredefinedDeploymentStrategyId || (exports.PredefinedDeploymentStrategyId = PredefinedDeploymentStrategyId = {}));
/**
 * Defines the rollout strategy for a deployment strategy and includes the growth factor,
 * deployment duration, growth type, and optionally final bake time.
 */
class RolloutStrategy {
    /**
     * @returns A linear rollout strategy.
     */
    static linear(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_RolloutStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.linear);
            }
            throw error;
        }
        return {
            growthFactor: props.growthFactor,
            deploymentDuration: props.deploymentDuration,
            growthType: GrowthType.LINEAR,
            finalBakeTime: props.finalBakeTime,
        };
    }
    /**
     * @returns An exponential rollout strategy.
     */
    static exponential(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_RolloutStrategyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.exponential);
            }
            throw error;
        }
        return {
            growthFactor: props.growthFactor,
            deploymentDuration: props.deploymentDuration,
            growthType: GrowthType.EXPONENTIAL,
            finalBakeTime: props.finalBakeTime,
        };
    }
}
exports.RolloutStrategy = RolloutStrategy;
_b = JSII_RTTI_SYMBOL_1;
RolloutStrategy[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.RolloutStrategy", version: "2.98.0-alpha.0" };
RolloutStrategy.CANARY_10_PERCENT_20_MINUTES = RolloutStrategy.exponential({
    growthFactor: 10,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(20),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(10),
});
RolloutStrategy.LINEAR_50_PERCENT_EVERY_30_SECONDS = RolloutStrategy.linear({
    growthFactor: 50,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(1),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(1),
});
RolloutStrategy.LINEAR_20_PERCENT_EVERY_6_MINUTES = RolloutStrategy.linear({
    growthFactor: 20,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(30),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(30),
});
RolloutStrategy.ALL_AT_ONCE = RolloutStrategy.linear({
    growthFactor: 100,
    deploymentDuration: aws_cdk_lib_1.Duration.minutes(0),
    finalBakeTime: aws_cdk_lib_1.Duration.minutes(10),
});
//# sourceMappingURL=data:application/json;base64,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