"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Environment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const iam = require("aws-cdk-lib/aws-iam");
const extension_1 = require("./extension");
class EnvironmentBase extends aws_cdk_lib_1.Resource {
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
}
/**
 * An AWS AppConfig environment.
 *
 * @resource AWS::AppConfig::Environment
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-environment.html
 */
class Environment extends EnvironmentBase {
    /**
     * Imports an environment into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the environment construct
     * @param environmentArn The Amazon Resource Name (ARN) of the environment
     */
    static fromEnvironmentArn(scope, id, environmentArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(environmentArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Missing required /$/{applicationId}/environment//$/{environmentId} from environment ARN: ${parsedArn.resourceName}`);
        }
        const resourceName = parsedArn.resourceName.split('/');
        if (resourceName.length != 3 || !resourceName[0] || !resourceName[2]) {
            throw new Error('Missing required parameters for environment ARN: format should be /$/{applicationId}/environment//$/{environmentId}');
        }
        const applicationId = resourceName[0];
        const environmentId = resourceName[2];
        class Import extends EnvironmentBase {
            constructor() {
                super(...arguments);
                this.applicationId = applicationId;
                this.environmentId = environmentId;
                this.environmentArn = environmentArn;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: environmentArn,
        });
    }
    /**
     * Imports an environment into the CDK from its attributes.
     *
     * @param scope The parent construct
     * @param id The name of the environment construct
     * @param attr The attributes of the environment
     */
    static fromEnvironmentAttributes(scope, id, attr) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_EnvironmentAttributes(attr);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEnvironmentAttributes);
            }
            throw error;
        }
        const applicationId = attr.application.applicationId;
        const environmentId = attr.environmentId;
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const environmentArn = stack.formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${applicationId}/environment/${environmentId}`,
        });
        class Import extends EnvironmentBase {
            constructor() {
                super(...arguments);
                this.application = attr.application;
                this.applicationId = attr.application.applicationId;
                this.name = attr.name;
                this.environmentId = environmentId;
                this.environmentArn = environmentArn;
                this.description = attr.description;
                this.monitors = attr.monitors;
            }
        }
        return new Import(scope, id, {
            environmentFromArn: environmentArn,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_EnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Environment);
            }
            throw error;
        }
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 64,
            separator: '-',
        });
        this.application = props.application;
        this.applicationId = this.application.applicationId;
        this.description = props.description;
        this.monitors = props.monitors;
        const resource = new aws_appconfig_1.CfnEnvironment(this, 'Resource', {
            applicationId: this.applicationId,
            name: this.name,
            description: this.description,
            monitors: this.monitors?.map((monitor) => {
                return {
                    alarmArn: monitor.alarm.alarmArn,
                    alarmRoleArn: monitor.alarmRole?.roleArn || this.createAlarmRole(monitor.alarm.alarmArn).roleArn,
                };
            }),
        });
        this._cfnEnvironment = resource;
        this.environmentId = this._cfnEnvironment.ref;
        this.environmentArn = this.stack.formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/environment/${this.environmentId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.environmentArn, this.name);
        this.application.addExistingEnvironment(this);
    }
    createAlarmRole(alarmArn) {
        const policy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['cloudwatch:DescribeAlarms'],
            resources: [alarmArn],
        });
        const document = new iam.PolicyDocument({
            statements: [policy],
        });
        const role = new iam.Role(this, 'Role', {
            roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('appconfig.amazonaws.com'),
            inlinePolicies: {
                ['AllowAppConfigMonitorAlarmPolicy']: document,
            },
        });
        return role;
    }
}
exports.Environment = Environment;
_a = JSII_RTTI_SYMBOL_1;
Environment[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.Environment", version: "2.98.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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