import { CfnResource } from "aws-cdk-lib";
import { IVpc } from "aws-cdk-lib/aws-ec2";
import { Construct } from "constructs";
/**
 * Properties for creating a new {@link CidrContext}.
 */
export interface CidrContextProps {
    /**
     * The VPC whose subnets will be configured.
     */
    readonly vpc: IVpc;
    /**
     * Split the CIDRs into this many groups (by default one for each subnet).
     */
    readonly cidrCount?: number;
    /**
     * The ID of a BYOIP IPv6 address pool from which to allocate the CIDR block.
     *
     * If this parameter is not specified or is undefined, the CIDR block will be
     * provided by AWS.
     */
    readonly addressPool?: string;
    /**
     * An IPv6 CIDR block from the IPv6 address pool to use for this VPC.
     *
     * The {@link EnableIpv6Props#addressPool} attribute is required if this
     * parameter is specified.
     */
    readonly cidrBlock?: string;
    /**
     * Whether this VPC should auto-assign an IPv6 address to launched ENIs.
     *
     * True by default.
     */
    readonly assignAddressOnLaunch?: boolean;
}
/**
 * Interface for the CidrContext class.
 */
export interface ICidrContext {
    /**
     * The IPv6-enabled VPC.
     */
    readonly vpc: IVpc;
}
/**
 * Allocates IPv6 CIDRs and routes for subnets in a VPC.
 *
 * @see {@link https://github.com/aws/aws-cdk/issues/894}
 * @see {@link https://github.com/aws/aws-cdk/issues/5927}
 */
export declare class CidrContext extends Construct implements ICidrContext {
    readonly vpc: IVpc;
    private readonly assignAddressOnLaunch;
    /**
     * Creates a new BetterVpc.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param options - The constructor options.
     */
    constructor(scope: Construct, id: string, options: CidrContextProps);
    /**
     * Figure out the minimun required CIDR subnets and the number desired.
     *
     * @param vpc - The VPC.
     * @param cidrCount - Optional. Divide the VPC CIDR into this many subsets.
     */
    protected validateCidrCount(vpc: IVpc, cidrCount?: number): number;
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    protected assignPublicSubnetCidrs(vpc: IVpc, cidrs: string[], cidrBlock: CfnResource): void;
    /**
     * Override the template; set the IPv6 CIDR for private subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    protected assignPrivateSubnetCidrs(vpc: IVpc, cidrs: string[], cidrBlock: CfnResource): void;
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    private assignIsolatedSubnetCidrs;
}
