"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const assert_1 = require("./vendored/assert");
/**
 * (experimental) Partial and special matching during template assertions.
 *
 * @experimental
 */
class Match {
    /**
     * (experimental) Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @experimental
     */
    static absentProperty() {
        return assert_1.ABSENT;
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * (experimental) Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * (experimental) Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * (experimental) Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @experimental
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.119.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        this.name = 'exact';
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult();
        if (typeof this.pattern !== typeof actual) {
            result.push(this, [], `Expected type ${typeof this.pattern} but received ${getType(actual)}`);
            return result;
        }
        if (this.pattern === assert_1.ABSENT) {
            throw new Error('absentProperty() can only be used in an object matcher');
        }
        if (actual !== this.pattern) {
            result.push(this, [], `Expected ${this.pattern} but received ${actual}`);
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
        if (this.partial) {
            this.name = 'arrayWith';
        }
        else {
            this.name = 'arrayEquals';
        }
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult().push(this, [], `Expected type array but received ${getType(actual)}`);
        }
        if (!this.partial && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult().push(this, [], `Expected array of length ${this.pattern.length} but received ${actual.length}`);
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult();
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement) ? patternElement : new LiteralMatch(this.name, patternElement);
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.partial || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.push(this, [], `Missing element${element}at pattern index ${patternIdx}`);
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
        if (this.partial) {
            this.name = 'objectLike';
        }
        else {
            this.name = 'objectEquals';
        }
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult().push(this, [], `Expected type object but received ${getType(actual)}`);
        }
        const result = new matcher_1.MatchResult();
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.push(this, [`/${a}`], 'Unexpected key');
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (patternVal === assert_1.ABSENT) {
                if (patternKey in actual) {
                    result.push(this, [`/${patternKey}`], 'Key should be absent');
                }
                continue;
            }
            if (!(patternKey in actual)) {
                result.push(this, [`/${patternKey}`], 'Missing key');
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
function getType(obj) {
    return Array.isArray(obj) ? 'array' : typeof obj;
}
//# sourceMappingURL=data:application/json;base64,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