"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends core_1.Resource {
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
        }
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantEncrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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