# -*- coding: utf-8 -*-
"""
=================================
11- Ultra-deep embedded Julia set
=================================

This example shows the kind of structure that occur very deep in the Mandelbrot
set [#f2]_. The width of this image is only *2.e-2608*.
This is not only below the separation power of double, but the delta are also
way below the minimal magnitude that can be stored in a double
(around  1e-323). A ad-hoc dataype is used internally, with and extra integer
to hold the exponent.

The period of the central minibrot is 2703248 ; the use of chained bivariate
approximations allows to skip several millions of iterations.

As the running time for this script is close to 15 minutes (10 minutes for the
reference, 5 minutes for the iterations) it has been pre-computed.

Reference:
`fractalshades.models.Perturbation_mandelbrot`

.. [#f2] **Credit:** Coordinates courtesy of Microfractal :
        <https://www.deviantart.com/microfractal/art/Mandelbrot-Deep-Julia-Morphing-22-Golden-Sphere-886123409>
"""

import os
import numpy as np

import fractalshades as fs
import fractalshades.settings as fssettings

import fractalshades.models as fsm
import fractalshades.colors as fscolors
from fractalshades.postproc import (
    Postproc_batch,
    Continuous_iter_pp,
    Raw_pp,
    DEM_normal_pp,
)
from fractalshades.colors.layers import (
    Color_layer,
    Bool_layer,
    Blinn_lighting,
    Normal_map_layer,
)


def plot(directory):
    """
    Example plot of a very deep location
    Credit for the coordinates :
https://www.deviantart.com/microfractal/art/Mandelbrot-Deep-Julia-Morphing-22-Golden-Sphere-886123409
    """
    
    precision = 3520
    nx = 3800
    x = '-1.94052953281236478154447638234806627298703651795056228166276001522388828787174615559555659145810476091403059107477748597361179339357923583452149175909248822158521966880825328563504616800005043876253885657910324383295486905365790654915286217066358321439531605581507973542605271965559834352969837838159160386221150298390773055655766499285642134450743788520483566036000282935033752222676363268556174766472963463638240784316879892528825281047518140026011262096284392385130459639426478278945663841008193663659464360709989871264373023066219075252573726455668598505325142929053648656989458873227942056214622018460216125720975194712243177288705693887541754700768806996137077960614664932737722662473546748292632251649320109361051015366337050961635256601525807385693078493567732879801805462798492266204279867337199560610662590062916768723802959183385985745188331226584578885076765633283818653832610607932074950453296699466201047915471948260448520908376970160079816447999771858696561802461181117195573762694872952871430691889519239747733406967335944700501086772801290020721295364750975541599060274126072517686909725946459865231912031337818097309361956411472189613326789042288653202826489288929471138854054654337355858317153339604816876453212679294171463403939572087640416758012923471485232404377041974075416231733702986359960062062444112757287193297223355553938433281496473248473166407667363951700022535044783748423986651150687289327507391789283482692136344502401737157803900057141556553405902985789742710114915530349507800764921593827778207181350128892999532905347037732146882060072639815741610613194761316822080422175147720742983496070300399024886098586617245411812189745450358557091722480124244641070244316584341439781920206501006800488493796809344712068412518921829465125329546627787272937922869773815986975040833238698817380323577978549907214529090504612900731752681481359926621501787643693706736240766114883049630634430258267371936935622606613913563647932040589790124739485671907123306457808796786962561473116961045973297273341463173007343268304549115465854679711383744068171592534841819280240257555254497627548270079224197368344576628688722257287190554514466115312975515542209083611046931476564562110563490864288961129185054834067164334622601176771989693139457748624107435906961394702198068200894787380298642058911205035110590504806392369429602155592520178866738636532615504086035981165987295439029771851151979777289939374876892780149689945892321827192378313587083324962167900184950258058095566225574577578744781335612306348510002491343750865169362015932957568215019291179011774575307174619866903969731769851246436206093455126958784548462165154211229544172457867584030914313747288264760567886601353760967276683152915474796516776897385061240728718865307359009330948859983064800083444826997726808344107281220213557454354031986628086607841136606535574716638647513635028300483275108993451235636260582707256428344723505336072416961937915806451348291777665788401670345521459233728538711741347019417265699766404816694894603345490450677566601332500564160017632686779812736639820788398833017834942943387404382054918617241072167670903416765873548573132150958032289145571020612957306816985458332230119750047983783296043511795816884425287267958302139254515223820459522552353949196718863045120887593285921693907821640442009525879813946890482903266694575358202419039447589300264208369550365294143228040145985347409309571258254323651025435292027911535882705241080719181320450313420959342496729676809471053637993014397133396934150772475980793983946720242298113345984693758699713687668131437134133885181907924403350846656901478812435937598277440974716865403124419849751459272405719806359356478194634929907010485152050610065650325612245553608230324857203903453287610553481342934191981763926897952590168739288795309325566680946888092968245591659028573402281192639026443886395729537699391228218189991037410758516066184603190927760772845217196363116120547806836206914664594063609143282885885654270829378546213681775920716752339485085231373898493553574593626974093318850634594423515391951205780987096153343203853196466203771999950'
    y = '0.00000203743075935398908839907854039267548738259616806537040956135073824743784187483527763415392279154542634108432806562614041761550498024996549503758554145754466893338004323349288268483898066703307737165697688116501664966304995858541710943457973298028767071478480111819646778983570005939548558336891745453468306515715129257182629182983277412113990887396612223838398948174178925856298017211685703306880435683612513444576367858007455172858218092564320724736594663399977335871447256639216459725434850653837562453847104866558381324796239086758750360291502683654860929564628071864113367494887376334691853733874744148745701056424229790574047619679294933160311934533664689349285363710284136538853317582419833258078109794382301101898986502300385964980434410770641396264552882587413475932157675465475889706912663671391223860152495368045246957456253618448323059652430809512338096428676487461565731639442831754735265786488584522557218453606027715412787230853354616787424345437827018537920764533393468478477399242586195009279825131018069967537433625867223015878731226158831389918637420751667597695992167284243742556541988596156682712851925823737563893532345082116736886719349968718966013360836483979240250420250726689411427753567651538322631035425056715469423684583280030936582540843641711011484059163837871580177506506936454976303548468001783160061053708927667666944472498162393105940216675706723132010534157978175849794870915998834162092487147003099064592587224823976079746164747379155287230360736646908803972798507912057870166813357203588476914190674108268289505426515685671720324117491565854385774377968354321670891996180169720606878026632572480450116396780390108970211968168077978913669076142984632355916837483314071476701114702575197679771335295626227310684230712039017962825551646743824593712371663277530213542263515320993335301306319771752654349840238456648887042107607888429345759636632508150641834192794686234751486771230976697927931119226684420135650456499839307320532855269021422460771599225157416697166690140543384558196531596970236811116835612500765492835724592760232795089625451767005420976359394266427663933542273921258421217532117861745297234434761566739760433409608895826900258964496017483806431667086058660674098565519066333196126755381809605307394820540239010029567622581247919349928308630247663799533865319381989186940818213849526203446306952811963340567480828844442589094279918391349175862779255358217546229034869282729070656779077587035289933261112172748158142800145507425142137441408370995998924508186938890179818251340953812565051516976598004720158282481186832582347370360840746941461773161865311989205513808076247886815761936091195183840294766761560698973967668275248323198157523336071691001024093837749756423055858268639502188396224646841483410465472703228330589117929892431070565647047764192527399277582207756709161943360273399189762258576352589248129527063646197040412015799137596563282979957250531340826922128364238961849831197450635976832599609904818294818371470866992382219399272664326440307233108628876970920718936280693983169009813159368834217562845132462842021950306759932861359961897530514316784849173254830239212607993399552171602626631298233333234689307657916891846099207384725957930800353770799343877161255891577162714737200272354048120308606702961093640806325779053489145707098987467378303110579844623288884924385917006264803581462367408618440509204291849583420110553137295249634661523579291463214824204087662457518833556010250664209586141104316785391108819061144444029733465391545475887204732699781056465731109638470309997717961483264446745512128326539201576905867995967739057360502428099061031394436176041036766619047420450571954815137996667306239843083538894370437380517982187757216995376161601870311623114443308773852452142463659788080861616721120180024325462836445272763321424591395071696724813859028324249116262685588808077821581675524636288138882476462485576626755322199510117478047514333491558690823196916388632250464029395714511038872597225434730185089563116760651303240832397249109394834846104439619664198399067697318837783761237863921584641212001368666502699000000'
    dx = '1.2749563896634140e-3056'

    fs.settings.log_directory = directory
    fs.set_log_handlers(verbosity="debug @ console + log")
    fs.settings.inspect_calc = True
    fs.settings.postproc_dtype = "float64"

    # Set to True if you only want to rerun the post-processing part
    # even with partially computed tiles
#    fssettings.skip_calc = False
    # Set to True to enable multi-processing
    fssettings.enable_multithreading = True
    # Set to True to skip Newton iterations
#    fssettings.no_newton = False
    # Set to True to enable additionnal text output
#    fs.settings.inspect_calc = True

    calc_name="deep"
    colormap = fscolors.cmap_register["classic"]

    f = fsm.Perturbation_mandelbrot(directory)
    f.zoom(
        precision=precision,
        x=x,
        y=y,
        dx=dx,
        nx=nx,
        xy_ratio=16./9.,
        theta_deg=0., 
        projection="cartesian",
    )

    f.calc_std_div(
        calc_name=calc_name,
        subset=None,
        max_iter=10100100,
        M_divergence=1.e3,
        epsilon_stationnary=1.e-3,
        BLA_eps = 1e-6,
        interior_detect=False
    )

    # Plot the image
    pp = Postproc_batch(f, calc_name)
    pp.add_postproc("cont_iter", Continuous_iter_pp())
    pp.add_postproc("interior", Raw_pp("stop_reason", func="x != 1."))
    pp.add_postproc("DEM_map", DEM_normal_pp(kind="potential"))

    plotter = fs.Fractal_plotter(
            pp,
            final_render=False,
            supersampling="5x5",
            jitter=False,
#            reload=True
    )
    plotter.add_layer(Bool_layer("interior", output=False))
#    plotter.add_layer(Normal_map_layer("DEM_map", max_slope=40, output=True))
    plotter.add_layer(Color_layer(
            "cont_iter",
            func="x",
            colormap=colormap,
            probes_z= [3375121.0, 3375131.0],
            output=True
    ))

    plotter["cont_iter"].set_mask(
            plotter["interior"],
            mask_color=(0., 0., 0.)
    )
#    plotter["DEM_map"].set_mask(
#            plotter["interior"],
#            mask_color=(0., 0., 0.)
#    )

    # This is where we define the lighting (here 2 ccolored light sources)
    # and apply the shading
#    light = Blinn_lighting(0.5, np.array([1., 1., 1.]))
#    light.add_light_source(
#        k_diffuse=0.2,
#        k_specular=400.,
#        shininess=400.,        
#        polar_angle=60.,
#        azimuth_angle=20.,
#        color=np.array([0.5, 0.5, .4]))
#    light.add_light_source(
#        k_diffuse=2.5,
#        k_specular=0.,
#        shininess=0.,
#        polar_angle=65.,
#        azimuth_angle=30.,
#        color=np.array([1.0, 1.0, 1.0]))
#    plotter["cont_iter"].shade(plotter["DEM_map"], light)

    plotter.plot()

def _plot_from_data(plot_dir):
    # Private function only used when building fractalshades documentation
    # This example takes too long too run to autogenerate the image for the
    # gallery each - so just grabbing the file from the html doc static path
    import PIL

    data_path = fs.settings.output_context["doc_data_dir"]
    im = PIL.Image.open(os.path.join(data_path, "gaia.jpg"))
    rgb_im = im.convert('RGB')
    tag_dict = {"Software": "fractalshades " + fs.__version__,
                "example_plot": "gaia"}
    pnginfo = PIL.PngImagePlugin.PngInfo()
    for k, v in tag_dict.items():
        pnginfo.add_text(k, str(v))
    if fs.settings.output_context["doc"]:
        fs.settings.add_figure(fs._Pillow_figure(rgb_im, pnginfo))
    else:
        # Should not happen
        raise RuntimeError()


if __name__ == "__main__":
    # Some magic to get the directory for plotting: with a name that matches
    # the file or a temporary dir if we are building the documentation
    try:
        realpath = os.path.realpath(__file__)
        plot_dir = os.path.splitext(realpath)[0]
        plot(plot_dir)
    except NameError:
        import tempfile
        with tempfile.TemporaryDirectory() as plot_dir:
            fs.utils.exec_no_output(_plot_from_data, plot_dir)
