# -*- coding: utf-8 -*-
import os
import unittest
import shutil

import numpy as np
import numba

import mpmath

import fractalshades as fs
import fractalshades.utils as fsutils
import fractalshades.colors as fscolors
import fractalshades.models as fsm
import fractalshades.numpy_utils.xrange as fsx
import fractalshades.numpy_utils.numba_xr as fsxn
# import fractalshades.bivar_series

from fractalshades.postproc import (
    Postproc_batch,
    Raw_pp,
    Attr_normal_pp,
    Attr_pp,
    Fractal_array
)
import test_config
from fractalshades.colors.layers import (
    Color_layer,
    Bool_layer,
    Normal_map_layer,
    Blinn_lighting
)

import fractalshades.mpmath_utils.FP_loop as fsFP

Xr_template = fsx.Xrange_array.zeros([1], dtype=np.complex128)

@numba.njit
def numba_path_loop(
        ref_path, has_xr, ref_index_xr, ref_xr, ref_div_iter, drift_xr, dx_xr
):
    npts = ref_path.size
    
    out_is_xr = np.zeros((1,), dtype=numba.bool_)
    out_xr = Xr_template.repeat(1)
    refpath_ptr = np.zeros((2,), dtype=np.int32)
    
    
    print("npts", npts, has_xr, "\n ref index:\n", ref_index_xr)
    count = 0
    for j in range(npts):
        for i in (j , j, npts - j - 1):
            val = fs.perturbation.ref_path_get(
                ref_path, i, has_xr, ref_index_xr, ref_xr, refpath_ptr,
                out_is_xr, out_xr, 0
            )
            
            if out_is_xr[0]:
                print("is_xr", i, val, fsxn.to_Xrange_scalar(out_xr[0]))
                count += 1

    print("count xr", count)
    if count != 14 * 3:
        print("count", count)
        raise ValueError("Unexpected count")

    count = 0
    for i in (7433792, 7433795, 8785472, 8785473, 9461312, 675720, 675728):
        #         Y       N        Y        N        Y       N       Y  
        val = fs.perturbation.ref_path_get(
            ref_path, i, has_xr, ref_index_xr, ref_xr, refpath_ptr,
            out_is_xr, out_xr, 0
        )
        
        if out_is_xr[0]:
            print("is_xr", i, val, fsxn.to_Xrange_scalar(out_xr[0]))
            count += 1
        else:
            print("** NOT xr", i, val)

    if count != 4:
        print("count", count)
        raise ValueError("Unexpected count")



@numba.njit
def numba_c_from_pix(path, pix):
    return path.c_from_pix(pix)
    
    
class Test_ref_path(unittest.TestCase):
    
    @classmethod
    # @test_config.no_stdout
    def setUpClass(cls):
        fs.settings.enable_multiprocessing = True
        fs.settings.no_newton = True
        fs.settings.inspect_calc = True
        
        precision = 3520
        nx = 800
        x = '-1.9409989391128007782656638595713128206620929316331395903205283705275932149841553750079140508152501109445961064000387852149507811657094626324996392008081820445955741490587617909708619603737265548027769325647808985287741667276189821676033432683374240723052323372896622554689290278821522432095519048328761094875168059910075072612524746195696519482376711787954155676296696827707057348137590781477540653443160271404114741216279924299516050033371623738987930710049260335938454436747992050897445704854917586460267198917634232454874517524790905068408711299098852857223323363509317448492707948571935557902448516804312250656708860690680767226144394692148838449346680921087412029850014210409147937112323614271639154365986968749816836442985665512979922489943829925482859841402388822224364772960765860128299173467963835512792813373451933644130190266047607001031626499249499592567711348988794983423352102489653363614657987130851011066068082416311059571884201802812522326939248656260215898332770887339844184688424916821959905805787211079924762420560654209080231130357236288188593275206143270109163936044056855567309338390204460230556526667618113052517191169646813610992208066490740332700166077086244561644939752386971282938070707062898838928187674154565542324706485606883204149973662143729325062503353762046809254607154103878222668282005954040495000651634097511941293052468376780564225465557438420172736278899353415715205080501056910932380856513690069593717239355697113322999606963893343303065997244593517188694362601778555657829079220370979486386183376634551544169026880446433151630826730127399985709844229666877539084763034446297595098204169627029966553348731711298433915468877133916519870332995252770006087468201433091412692008675169426600509762262849033820684824479730400854046509072164630272105114166613615665383021053646289448207336461725630828678598527683609575006544933462912457658924436663804582292428129309162915840098216747977268766925226272677267826315722555021136934491464926926641085339160830952887601459585519624489323898936587933143756193630971066578717659019875144049965572880866540996031144922280813352065159362962936897218127976473669535727210317367178865163942427120257230318803642220091013441782124465936161868040076934432584798273802125003893761405910549636791922164569969871504895180875775512279622397659490539731258965222183682582044022842758452337516752189727551206382556078493830490372988205049395299138260871313038171904760429268109644267193074206275040851482988811238053209498575928806745490180665861235757156293268030156174736154214485511919238045324816790747039434094153238651378208655247035749519428374239948111490578363711926298127059816373882058600875440218265729937727712935557101248183859985480838214443248343204994169001603385068217409551664275124868238925925271002064990910751541295196946319404974130124223074815816387748372081603618046256402766723419509314015491326315372861880224396707850752490829513864536227468094212074909783507683557390914984737208904927522859784984066452431380596052384391155762961147112917902257288838205513568126100751182438074841839964967562205987620459771593676482435160564881907643374624834394770129519338651384779340621276744712596399177749754956987947612707663018919330037816063293842647052555147743226921275393227281792532802856285703297338604821969492356674112869979073125870095512233460880231177088317720580337642382172126187069216048936896730950168087435988621276438670059341103609929304930466412268150569753470717829497601938341623581803667066999928999945000062'
        y = '-0.0006521165369165588520106289441620153907907521525225557951700039268755659160275378414816331241993503713942651869474366440330624054932785747734116130598457275168672169867853790149073948820621927863546898987531675745541556010963860271946131945706089440068213570737152573434606181998626256475661137064241766615685133034114571184540746713081041577482152866404680905298142203271097108866125320734562827910017740404764291477614758081664091324083106696109319507742512146578699926177581123430550120851818916049981949393089874937840577370413575565615246397463453690404270526656455145637869566754373564864548747775061651693403960187403612827482714675143082173905414385810506804378880397100996175280822311114495867725750471436402145707242763362689139153766093202506743259707579782531683072699910204376229255257696447791057044885184061849070063540925613028401048182129422816270970456315092465855569329878796473503666036123284601909076758201573065328180211040459230345709044071756847669905912521106047214804555579992552727318466143562534207465701332898411609149336015158023746864705973770293526683875460324480616782478489019514943512702395590818455582259983339029054638765126731537575594335734368117123722683120375030995584809981966023016675121788001130361752945926045051983789243281329028107416493849599211739205918880442308088915329310667744587253842928202077978689211781621700292204988439971992046135099101850443216579189710924423016693808479474589682525790322932538431715348758724089186172736870724706725359784401019519888555644853285575115223472590818823322033130852641478536530503881747200363162574382337579455223211205019832848615171631087121056343365803496414693646695845027511119821045191586941544022389773784151557473277272394880876628653639136977979073123486169650096416150642999247909147333278062324113459547152270378118487801961875006181455991513879900323624590458328414797373565255061007383050772917420374420930369627261609756033085579925058681478773760867701230719359928389502388023578804808713069253869301107296738982313988108484002367456921622985540672687977893371677916030176767500564905285025226973308704535270965189005321129735333599100313629076978281635241128387571784303118677495016595486491171040002394480779899042204488631259847989603182340726213078367178896618081990169319498713349339065257474424401748553283927933449943175175157120972516636257833849555669271463331231601029167028638597915809746995436188809835668111701784052366810307436108276491541042658178481843136392746657892940367221519240125914939061964441432380740020708127640600546604568699045234845708728863090863984386209155813013615576381026653379878624402126265227089167061378994809588030662831377110537145242600584959148498586439529663105983709419546957848439948376427305067215182145348517650481959560955434577158090652441197554228656503253796471623707876797570793456353888545895776536724341010890647565137237971578364800606022054805371016117249815862385204930532791360055457643453800167233033393824944921504096748637258867979270585206447548364249344195079436376739232814985700753366335710763351616828921383429188346008648781525793755795069682228036514982477038907976343304196109685257025904974333612600761354191140826329760186432247441069680365217200145218033541210372615053282512008534408785235009976598833958899392833195540809260984815364215770028371283427130718815533338521166040923413722562752702386025562655776477893889452984598715385588865771230862335806477085969230662862126372402082027768431991530300520064005268033000000000000000000'
        dx = '2.e-2608'
#        complex_type = np.complex128

        subset_dir = os.path.join(
            test_config.temporary_data_dir,
            "_numba_ref_path_dir"
        )
        fsutils.mkdir_p(subset_dir)
        cls.subset_dir = subset_dir
        cls.calc_name = "test"
        # cls.dir_ref = os.path.join(test_config.ref_data_dir, "subset_REF")
        cls.f = f = fsm.Perturbation_mandelbrot(subset_dir)

        f.zoom(precision=precision, x=x, y=y, dx=dx, nx=nx, xy_ratio=1.0,
               theta_deg=0., projection="cartesian")
        f.calc_std_div(
#                datatype=complex_type,
                calc_name=cls.calc_name,
                subset=None,
                max_iter=10000000,
                M_divergence=1.e3,
                epsilon_stationnary=1.e-3,
                interior_detect=False,
                BLA_eps=1.e-6,
                calc_dzndc=False
        )

        print("################# before get_FP_orbit")
        f.get_FP_orbit()
        print("################# after get_FP_orbit")
        cls.FP_params = f.FP_params
        cls.ref_path = f.Zn_path


    def test_numba_path(self):
        """
        14 xr indices
        [ 675728 1351568 2027296 2703248 3378976 4054816 4730544 5406496 6082224
        6758064 7433792 8109744 8785472 9461312]"""
        
        # ref_path = self.f.get_Ref_path()
        (ref_path, has_xr, ref_index_xr, ref_xr, ref_div_iter, ref_order, drift_xr, dx_xr
         ) = self.f.get_path_data()

        print("ref_path", ref_path)
        
        numba_path_loop(
            ref_path, has_xr, ref_index_xr, ref_xr, ref_div_iter, drift_xr, dx_xr
        )
        
    def test_print(self):
        fs.perturbation.PerturbationFractal.print_FP(
            self.FP_params, self.ref_path
        )

    @classmethod
    def tearDownClass(cls):
        fs.settings.no_newton = False

class Test_newton(unittest.TestCase):

    def test_ball_method(self):
        """
        Test Cython ball method
        """
        ball_order = {
            "x": "-1.38489865821023436791757551552306535886843948840412919252407990736888673896107",
            "y": "0.0227499123767246576841168164274146683393042605219762097416661295912819114324773",
            "maxiter": 100000,
            "radius_pixels": 3,
            "radius": "1.253026641642179625e-70",
            "M_divergence": 1000.0,
            "order": 4252
        } 
        x = ball_order["x"]
        y = ball_order["y"]
        r0 = ball_order["radius"]
        precision = int(78 * 3.33) # precision in bits

        order = fsFP.perturbation_mandelbrot_ball_method(
            str(x).encode('utf8'),
            str(y).encode('utf8'),
            precision,
            str(r0).encode('utf8'),
            ball_order["maxiter"],
            ball_order["M_divergence"]
        )
        print("order", order)
        self.assertEqual(order, ball_order["order"])


    def test_str_out(self):
        x = "-1.50000000000000000000000000000000000000000000000000000000000001"
        precision = int(len(x) * 3.33)
        mpmath.mp.prec = precision
        ret = fsFP._test_mpfr_to_python(
            str(x).encode('utf8'),
            precision
        )

        is_equal = mpmath.almosteq(mpmath.mpf(x) * 2., ret)
        print("ret", ret, type(ret))
        print("ret * 2", ret * 2)
        print(float(ret))
        self.assertTrue(is_equal)

        y = "-1.50000000000000000000000000000000000000000000000000000000000001"
        precision = int(len(x) * 3.33)
        ret = fsFP._test_mpc_to_python(
            str(x).encode('utf8'),
            str(y).encode('utf8'),
            precision
        )
        print(ret)
        print(dir(ret))
        print(ret.imag, type(ret.imag))
        print(float(ret.imag))
#        a = mpmath.mpc(
#            mpmath.mpf(ret.real.as_mantissa_exp()),
#            mpmath.mpf(ret.imag.as_mantissa_exp())
#        )
        is_equal = mpmath.almosteq(mpmath.mpf(x) * (1. + 1j), ret)
        print(is_equal)
        self.assertTrue(is_equal)


    def test_newton(self):
        """
        Test Cython newton method
        """
        newton_search = {
            "x_start": "-1.38489865821023436791757551552306535886843948840412919252407990736888673946771734",
                      # -1.384898658210234367917575515523065358868439488404129192524079907368886731071071
            "y_start": "0.0227499123767246576841168164274146683393042605219762097416661295912819134778267851",
            "order": 4252,
                        # -1.384898658210234367917575515523065358868439488404129192524079907368886731071071"
            "x_nucleus": "-1.384898658210234367917575515523065358868439488404129192524079907368886731071070290597737327432208840677453517173286448408866226044469621428477672357862724621488179028111732996542591281209148762118889234892037810796912513242881089802259331283320590242459763937836417161921527945857754559159319285383388065105320462521737268935854537254334081892499119184659981831545828333620648594267400219876222339466444419570130912837345844427308581414807326890483942795318275203584063292725367898743833997728784162826193628251523069755931013368411822190869321433789564544441585133097038489340740401348693780401645998731914380562063495469379430034192833658874660407862857944557262450884949376176485242263883332366742107603556473032701375817056416109169174003851015583671029376503919964379185805386973459627223242717548185397791661030283940357630242077950828221222761345551013939547504363904891452248134959442967751256635420953269361209693142594751827252276311622211161303338713979387498146022555169458560196959431871292976784477566279809987430287392674863384462176841792217105111645585061149478516056187429670357540451584091730367671802224669203260354259074441057400338852271137370597619479063946211724981190636722057148412523719407928857594000998351741891319181825479539120938921559113929429837241097240999763366672555498654838371701089925378492586384604575774588053016794521112694182187419806040838810594490140975831449433315007137466504477415629759911137969660764255628152461579806142566466449124890768241003530265410721759548010502473087774944305621076745345290581374121843593772783992094679266476532974848358115409883409689238746613390059112346135143734842796331105408303889560627944284297643019691733262427261916197896130042069856228805916942520541952967086438748232278951326182431433894528986942084948117111513036920627128726094889130088748930129403425796896801983706618026928855677717907574861885240438957301376359009785124937415946628171498544918239692049650684317929521661550684735160900072378209293274343835975392116213583081848096758294069277767163769152819596241698026068104100443690368498641842969647166021531796621557668344692626249974606083378018842624731144995170567814145619634650463990257153200014705543982893973757842416417820304612581777566216837658169909814169440092746576335835098963395705271146613845201996459979302162346673064003186518770906488677117370116939332474769708384422290588262596274070150009629962935250095760660371483134194932278960224049065837640766189638531338144873811957391221091189356404060868126759178113102134453760794308221409826015639123643227761785779462100892435413298488572582041971557284512888540575326096827707762308681757124108440265214018015454332578708469255029949216987491131310454344950705159583116745438937285217159110821755079801011043089055951101274140132218431575020160520464933779172769220322899291896495798852628445100151776277336891147664053100229545550893340495566461430485847441553687227850322369201543764030791477427743602367226508284650107332830870941688118261829055293684480892916096509513257172569369773483638202147216251906820294188440494826073983239414485919996068532801114187089293959470994399531875598400639481899592705652612995793891273687001796161288375352871412483742568114114333811883789177126487510484415032705474559228643152165499519515065933939609956399298992509076827007723345792984020456899748496005006747798330097835898418452660482943473941789529554687641528914237042847875897579678796304046953546419858883298767417601641900931960342240020493789122651064",
            "y_nucleus": "0.02274991237672465768411681642741466833930426052197620974166612959128193946599148663186825181400444801617902649746806986512836765355177903929141487291683110650978461712462524687126133894875596060308681361104833937528120348545366959122931144861989474428824041235640513711128271395006129931760926892841430434944360116868670528335319769241973236989333953830700644975630936980081530182606798363910233428393712610721672702307367478384840762969935947898265290497971257548401454342622724819054947703466325891518996898444847755836312159070427425530522626305221612707628912420367401353915076925646997332810781726997188872553836780195943610730461291000447968626131265356928903193789956336871991855398664589981778911852346698606110912746334808586202348059280620487697328192294599478384339499955964505585016647296172942826891221285671615887805243446787277399610483064104718775312209863851691540079419951662873752956844077804483606220177402951076090919048548882834980184087105282050028584606602280161455941010223184600649246559311064438481399357643787014775659058517659844777792673179277016030869941750791665874059099872067480342929268437753109606495110846803946033879691682368471370782381873518785492205170508743931724987766853562421777707611931574109403499438780979579145822038475927284789183803331626623203423328865605605714022837667637217986098934905700572834627663035072349710098142630056586758504392030909497301367626010390038849757543029936188135957260231103066541685149275531948779898210446009843622801340293733204034917653186732997729872409542920840578952875280173036440726688634383226906795479812112850408574421645735398524467540399629397951897284902011848690229677440280928964895140060438241750659537420047862387843078311544680447481063179572104824044132364190550559402772107342368169119639746361790000212704083306665845759989019932885593184934678464280631470852709516522194744801017054738958179981093671269759564160133507409509794667835185910813779922915851775931689297034580502259010325584871138943616102271820201450109434558108214463968881695570510623250163633895206410719849984215787324622253700200123257094829690065737926302030563039471262994128923247934489520076295937940715125500589390271463847802808716916054940726125865174867319944467953622748681213861887922154541729800764344421696151229219579308114931121163670891657190289916657590109284793324632455907281806538817932094507269636801746632242063636456132003769269606371214308660473980268455976527592186575377374400877440857097563032203825886708721992762579522190196819458675618729177099950191480595760045869448677154498032710518356116184402530957106267612384132671925147775943068806550989509319014324066642386918837372440528786995664105806947281219784909324164428213864372668955128776320531547273003793924255816758008153517203649953490669231338984964677497707773706731377133321320709438739797078857519267931064685463095366122293841160283549330426185711473729449237131851762471331548604120200999897025646626746384166206359292874381760858754652216027816952480190791698144851155013112459358463168100209942581466058418656370690811294322094354350871386054984191414163005245993506775113207165765364938298440142673518637655755085818994789733903416061759698648770332901147863373593785422162004965272876477501550673032814150382610403569366848966477835395924891081465661604955885574229809273601533307483102106387977778081572012569967160272697015243920849565452956090988298448875399315697690578185407249503667923686545839968981499842959967983934762594249424971212143006331771",
        }
        x = newton_search["x_start"]
        y = newton_search["y_start"]
        precision = int(3522 * 3.3) # precision in bits
        pix = int(84 * 3.3) # precision in bits
        mpmath.mp.prec = precision
        print("prec", precision)
        # eps_cv = mpmath.mpf(2.)**(-mpmath.mp.prec) is a good first estimate
        eps = mpmath.mpf(val=(2, -precision))
        eps_pix = mpmath.mpf(val=(2, -pix))
        print("eps", eps)

        is_ok, val = fsFP.perturbation_mandelbrot_find_nucleus(
            x.encode('utf8'),
            y.encode('utf8'),
            precision,
            newton_search["order"],
            80,
            str(eps).encode('utf8'),
            str(eps_pix).encode('utf8'),
        )

        with mpmath.workprec(precision - 10):
            print(val)
            expected = mpmath.mpf(newton_search["x_nucleus"])
            print("**expected", expected.real)
            print("**got", is_ok, val.real)
            is_equal = mpmath.almosteq(expected.real, val.real)

        self.assertTrue(is_equal)

BURNING_SHIP = 1

class Test_BS(unittest.TestCase):

    def test_path(self):
        x = 0.1
        y = 0.1
        c = x + 1j * y
        npts = 10

        orbit_complex = np.empty((npts,), dtype=np.complex128)
        orbit_float = orbit_complex.view(np.float64)
        
        (i, partial_dict, xr_dict
         ) = fsFP.perturbation_nonholomorphic_FP_loop(
            orbit_float,
            False,
            npts - 1,
            1000.,
            str(x).encode('utf8'),
            str(y).encode('utf8'),
            53,
            BURNING_SHIP
            
        )
        print("orbit_complex", orbit_complex)
        print("i", i)
        z = complex(0.)
        for i in range(10):
            assert orbit_complex[i] == z
            x = z.real
            y = z.imag
            new_x = x ** 2 - y ** 2 + c.real
            new_y = 2 * np.abs(x * y) - c.imag
            z = new_x + 1j * new_y
#            print("new_z", i + 1, z)
    
    def test_order(self):
        x_str = "-1.7545128115395"
        y_str = "0.0015894811966473"
        seed_px = "0.01"

        order = fsFP.perturbation_nonholomorphic_ball_method(
            x_str.encode('utf8'),
            y_str.encode('utf8'),
            53,
            seed_px.encode('utf8'),
            100,
            1000.,
            BURNING_SHIP
        )
        print("order", order)
        assert order == 3

        x_str = "0.88410156557344"
        y_str = "1.5218981991448"
        seed_px = "0.01"

        order = fsFP.perturbation_nonholomorphic_ball_method(
            x_str.encode('utf8'),
            y_str.encode('utf8'),
            53,
            seed_px.encode('utf8'),
            100,
            1000.,
            BURNING_SHIP
        )
        assert order == 3

    def test_newton(self):
        x_str = "-1.7545128115395"
        y_str = "0.0015894811966473"
        seed_px = "0.01"
        seed_prec = 53
        order = 3
        eps_cv = mpmath.mpf(val=(2, -seed_prec))
         
        is_ok, val = fsFP.perturbation_nonholomorphic_find_any_nucleus(
            x_str.encode('utf8'),
            y_str.encode('utf8'),
            seed_prec,
            order,
            40,
            str(eps_cv).encode('utf8'),
            seed_px.encode('utf8'),
            BURNING_SHIP
        )
        assert order == 3
        print(is_ok, val)

        x_str = "0.88410156557344"
        y_str = "1.5218981991448"
        seed_px = "0.012"
        seed_prec = 53
        order = 3
        eps_cv = mpmath.mpf(val=(2, -seed_prec))
         
        is_ok, val = fsFP.perturbation_nonholomorphic_find_any_nucleus(
            x_str.encode('utf8'),
            y_str.encode('utf8'),
            seed_prec,
            order,
            40,
            str(eps_cv).encode('utf8'),
            seed_px.encode('utf8'),
            BURNING_SHIP
        )
        assert order == 3
        print(is_ok, val)

#        x_str = "-1.7545128115395"
#        y_str = "0.0015894811966473"
#        seed_px = "0.001"
#
#        order = fsFP.perturbation_BS_ball_method(
#            x_str.encode('utf8'),
#            y_str.encode('utf8'),
#            53,
#            seed_px.encode('utf8'),
#            100,
#            1000. 
#        )
#        assert order == 3

if __name__ == "__main__":
    full_test = True
    runner = unittest.TextTestRunner(verbosity=2)
    if full_test:
        runner.run(test_config.suite([Test_ref_path,
                                      Test_BS,
                                      Test_newton]))
    else:
        suite = unittest.TestSuite()
#        suite.addTest(Test_bivar_SA("test_basic"))
##        suite.addTest(Test_bivar_SA("test_SA"))
#        suite.addTest(Test_bivar_SA("test_bivar_SA"))
#        suite.addTest(Test_newton("test_ball_method"))
#        suite.addTest(Test_newton("test_str_out"))
        # suite.addTest(Test_newton("test_newton"))
#        suite.addTest(Test_ref_path("test_print"))
        # suite.addTest(Test_bivar_SA("test_bivar_SA"))Test_BS
        suite.addTest(Test_BS("test_path"))
        suite.addTest(Test_BS("test_order"))
        suite.addTest(Test_BS("test_newton"))
        runner.run(suite)
