"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
            redrivePolicy: this.buildDeadLetterConfig(this.deadLetterQueue),
        });
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue;
        deadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [deadLetterQueue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new aws_iam_1.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': props.topic.topicArn },
            },
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                deadLetterTargetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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