"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Define a CloudWatch EventBus
 *
 * @resource AWS::Events::EventBus
 */
class EventBus extends core_1.Resource {
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.stringValue({ produce: () => this.node.uniqueId }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param eventBusArn ARN of imported event bus
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = eventBusArn;
                this.eventBusName = parts.resourceName || '';
                this.eventBusPolicy = '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported event bus properties
     */
    static fromEventBusAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = attrs.eventBusArn;
                this.eventBusName = attrs.eventBusName;
                this.eventBusPolicy = attrs.eventBusPolicy;
                this.eventSourceName = attrs.eventSourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge
     * so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined)
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/permissions-reference-cwe.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
//# sourceMappingURL=data:application/json;base64,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