"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
module.exports = {
    'CodeBuild action': {
        'that is cross-account and has outputs': {
            'causes an error'(test) {
                const app = new core_1.App();
                const projectStack = new core_1.Stack(app, 'ProjectStack', {
                    env: {
                        region: 'us-west-2',
                        account: '012345678912',
                    },
                });
                const project = new codebuild.PipelineProject(projectStack, 'Project');
                const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                    env: {
                        region: 'us-west-2',
                        account: '012345678913',
                    },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new cpactions.CodeCommitSourceAction({
                                    actionName: 'CodeCommit',
                                    repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'repo-name'),
                                    output: sourceOutput,
                                })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({
                    stageName: 'Build',
                });
                // this works fine - no outputs!
                buildStage.addAction(new cpactions.CodeBuildAction({
                    actionName: 'Build1',
                    input: sourceOutput,
                    project,
                }));
                const buildAction2 = new cpactions.CodeBuildAction({
                    actionName: 'Build2',
                    input: sourceOutput,
                    project,
                    outputs: [new codepipeline.Artifact()],
                });
                test.throws(() => {
                    buildStage.addAction(buildAction2);
                }, /https:\/\/github\.com\/aws\/aws-cdk\/issues\/4169/);
                test.done();
            },
        },
        'can be backed by an imported project'(test) {
            const stack = new core_1.Stack();
            const codeBuildProject = codebuild.PipelineProject.fromProjectName(stack, 'CodeBuild', 'codeBuildProjectNameInAnotherAccount');
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: new s3.Bucket(stack, 'Bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                input: sourceOutput,
                                project: codeBuildProject,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': 'codeBuildProjectNameInAnotherAccount',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const codeBuildAction = new cpactions.CodeBuildAction({
                actionName: 'CodeBuild',
                input: sourceOutput,
                project: new codebuild.PipelineProject(stack, 'CodeBuild', {
                    buildSpec: codebuild.BuildSpec.fromObject({
                        version: '0.2',
                        env: {
                            'exported-variables': [
                                'SomeVar',
                            ],
                        },
                        phases: {
                            build: {
                                commands: [
                                    'export SomeVar="Some Value"',
                                ],
                            },
                        },
                    }),
                }),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: s3.Bucket.fromBucketName(stack, 'Bucket', 'bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            codeBuildAction,
                            new cpactions.ManualApprovalAction({
                                actionName: 'Approve',
                                additionalInformation: codeBuildAction.variable('SomeVar'),
                                notificationTopic: sns.Topic.fromTopicArn(stack, 'Topic', 'arn:aws:sns:us-east-1:123456789012:mytopic'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Namespace': 'Build_CodeBuild_NS',
                            },
                            {
                                'Name': 'Approve',
                                'Configuration': {
                                    'CustomData': '#{Build_CodeBuild_NS.SomeVar}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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