"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../lib");
module.exports = {
    'basic pipeline'(test) {
        const stack = new core_1.Stack();
        const repository = new codecommit.Repository(stack, 'MyRepo', {
            repositoryName: 'my-repo',
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const sourceOutput = new codepipeline.Artifact('SourceArtifact');
        const source = new cpactions.CodeCommitSourceAction({
            actionName: 'source',
            output: sourceOutput,
            repository,
        });
        pipeline.addStage({
            stageName: 'source',
            actions: [source],
        });
        const project = new codebuild.PipelineProject(stack, 'MyBuildProject');
        pipeline.addStage({
            stageName: 'build',
            actions: [
                new cpactions.CodeBuildAction({
                    actionName: 'build',
                    input: sourceOutput,
                    project,
                }),
            ],
        });
        test.notDeepEqual(assert_1.SynthUtils.toCloudFormation(stack), {});
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'Tokens can be used as physical names of the Pipeline'(test) {
        const stack = new core_1.Stack(undefined, 'StackName');
        new codepipeline.Pipeline(stack, 'Pipeline', {
            pipelineName: core_1.Aws.STACK_NAME,
        });
        assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Name': {
                'Ref': 'AWS::StackName',
            },
        }));
        test.done();
    },
    'pipeline with GitHub source with poll trigger'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.POLL,
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Actions': [
                        {
                            'Configuration': {
                                'PollForSourceChanges': true,
                            },
                            'Name': 'GH',
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'Name': 'Boo',
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.done();
    },
    'pipeline with GitHub source without triggers'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                    trigger: cpactions.GitHubTrigger.NONE,
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::CodePipeline::Webhook')));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Actions': [
                        {
                            'Configuration': {
                                'PollForSourceChanges': false,
                            },
                            'Name': 'GH',
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'Name': 'Boo',
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.done();
    },
    'github action uses ThirdParty owner'(test) {
        const stack = new core_1.Stack();
        const secret = new core_1.CfnParameter(stack, 'GitHubToken', { type: 'String', default: 'my-token' });
        const p = new codepipeline.Pipeline(stack, 'P');
        p.addStage({
            stageName: 'Source',
            actions: [
                new cpactions.GitHubSourceAction({
                    actionName: 'GH',
                    runOrder: 8,
                    output: new codepipeline.Artifact('A'),
                    branch: 'branch',
                    oauthToken: core_1.SecretValue.plainText(secret.valueAsString),
                    owner: 'foo',
                    repo: 'bar',
                }),
            ],
        });
        p.addStage({
            stageName: 'Two',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'Boo' }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Webhook'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'PArtifactsBucket5E711C12',
                },
                'Type': 'S3',
            },
            'RoleArn': {
                'Fn::GetAtt': [
                    'PRole07BDC907',
                    'Arn',
                ],
            },
            'Stages': [
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Source',
                                'Owner': 'ThirdParty',
                                'Provider': 'GitHub',
                                'Version': '1',
                            },
                            'Configuration': {
                                'Owner': 'foo',
                                'Repo': 'bar',
                                'Branch': 'branch',
                                'OAuthToken': {
                                    'Ref': 'GitHubToken',
                                },
                                'PollForSourceChanges': false,
                            },
                            'Name': 'GH',
                            'OutputArtifacts': [
                                {
                                    'Name': 'A',
                                },
                            ],
                            'RunOrder': 8,
                        },
                    ],
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Approval',
                                'Owner': 'AWS',
                                'Provider': 'Manual',
                                'Version': '1',
                            },
                            'Name': 'Boo',
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'Two',
                },
            ],
        }));
        test.deepEqual([], core_1.ConstructNode.validate(p.node));
        test.done();
    },
    'onStateChange'(test) {
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const pipeline = new codepipeline.Pipeline(stack, 'PL');
        pipeline.addStage({
            stageName: 'S1',
            actions: [
                new cpactions.S3SourceAction({
                    actionName: 'A1',
                    output: new codepipeline.Artifact('Artifact'),
                    bucket: new s3.Bucket(stack, 'Bucket'),
                    bucketKey: 'Key',
                }),
            ],
        });
        pipeline.addStage({
            stageName: 'S2',
            actions: [
                new cpactions.ManualApprovalAction({ actionName: 'A2' }),
            ],
        });
        pipeline.onStateChange('OnStateChange', {
            target: new targets.SnsTopic(topic),
            description: 'desc',
            eventPattern: {
                detail: {
                    state: ['FAILED'],
                },
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            'Description': 'desc',
            'EventPattern': {
                'detail': {
                    'state': [
                        'FAILED',
                    ],
                },
                'detail-type': [
                    'CodePipeline Pipeline Execution State Change',
                ],
                'source': [
                    'aws.codepipeline',
                ],
                'resources': [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    'Ref': 'AWS::Partition',
                                },
                                ':codepipeline:',
                                {
                                    'Ref': 'AWS::Region',
                                },
                                ':',
                                {
                                    'Ref': 'AWS::AccountId',
                                },
                                ':',
                                {
                                    'Ref': 'PLD5425AEA',
                                },
                            ],
                        ],
                    },
                ],
            },
            'State': 'ENABLED',
            'Targets': [
                {
                    'Arn': {
                        'Ref': 'TopicBFC7AF6E',
                    },
                    'Id': 'Target0',
                },
            ],
        }));
        test.deepEqual([], core_1.ConstructNode.validate(pipeline.node));
        test.done();
    },
    'PipelineProject': {
        'with a custom Project Name': {
            'sets the source and artifacts to CodePipeline'(test) {
                const stack = new core_1.Stack();
                new codebuild.PipelineProject(stack, 'MyProject', {
                    projectName: 'MyProject',
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeBuild::Project', {
                    'Name': 'MyProject',
                    'Source': {
                        'Type': 'CODEPIPELINE',
                    },
                    'Artifacts': {
                        'Type': 'CODEPIPELINE',
                    },
                    'ServiceRole': {
                        'Fn::GetAtt': [
                            'MyProjectRole9BBE5233',
                            'Arn',
                        ],
                    },
                    'Environment': {
                        'Type': 'LINUX_CONTAINER',
                        'PrivilegedMode': false,
                        'Image': 'aws/codebuild/standard:1.0',
                        'ComputeType': 'BUILD_GENERAL1_SMALL',
                    },
                }));
                test.done();
            },
        },
    },
    'Lambda PipelineInvokeAction can be used to invoke Lambda functions from a CodePipeline'(test) {
        const stack = new core_1.Stack();
        const lambdaFun = new lambda.Function(stack, 'Function', {
            code: new lambda.InlineCode('bla'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const source1Output = new codepipeline.Artifact('sourceArtifact1');
        const source1 = new cpactions.S3SourceAction({
            actionName: 'SourceAction1',
            bucketKey: 'some/key',
            output: source1Output,
            bucket,
        });
        const source2Output = new codepipeline.Artifact('sourceArtifact2');
        const source2 = new cpactions.S3SourceAction({
            actionName: 'SourceAction2',
            bucketKey: 'another/key',
            output: source2Output,
            bucket,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [
                source1,
                source2,
            ],
        });
        const lambdaAction = new cpactions.LambdaInvokeAction({
            actionName: 'InvokeAction',
            lambda: lambdaFun,
            inputs: [
                source2Output,
                source1Output,
            ],
            outputs: [
                new codepipeline.Artifact('lambdaOutput1'),
                new codepipeline.Artifact('lambdaOutput2'),
                new codepipeline.Artifact('lambdaOutput3'),
            ],
        });
        pipeline.addStage({
            stageName: 'Stage',
            actions: [lambdaAction],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'ArtifactStore': {
                'Location': {
                    'Ref': 'PipelineArtifactsBucket22248F97',
                },
                'Type': 'S3',
            },
            'RoleArn': {
                'Fn::GetAtt': [
                    'PipelineRoleD68726F7',
                    'Arn',
                ],
            },
            'Stages': [
                {
                    'Name': 'Source',
                },
                {
                    'Actions': [
                        {
                            'ActionTypeId': {
                                'Category': 'Invoke',
                                'Owner': 'AWS',
                                'Provider': 'Lambda',
                                'Version': '1',
                            },
                            'Configuration': {
                                'FunctionName': {
                                    'Ref': 'Function76856677',
                                },
                            },
                            'InputArtifacts': [
                                { 'Name': 'sourceArtifact2' },
                                { 'Name': 'sourceArtifact1' },
                            ],
                            'Name': 'InvokeAction',
                            'OutputArtifacts': [
                                { 'Name': 'lambdaOutput1' },
                                { 'Name': 'lambdaOutput2' },
                                { 'Name': 'lambdaOutput3' },
                            ],
                            'RunOrder': 1,
                        },
                    ],
                    'Name': 'Stage',
                },
            ],
        }));
        test.equal((lambdaAction.actionProperties.outputs || []).length, 3);
        assert_1.expect(stack, /* skip validation */ true).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'codepipeline:PutJobSuccessResult',
                            'codepipeline:PutJobFailureResult',
                        ],
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'FunctionServiceRoleDefaultPolicy2F49994A',
            'Roles': [
                {
                    'Ref': 'FunctionServiceRole675BB04A',
                },
            ],
        }));
        test.done();
    },
    'cross-region Pipeline': {
        'generates the required Action & ArtifactStores properties in the template'(test) {
            const pipelineRegion = 'us-west-2';
            const pipelineAccount = '123';
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'TestStack', {
                env: {
                    region: pipelineRegion,
                    account: pipelineAccount,
                },
            });
            const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
                crossRegionReplicationBuckets: {
                    'us-west-1': s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'sfo-replication-bucket'),
                },
            });
            const sourceBucket = new s3.Bucket(stack, 'MyBucket');
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            const sourceAction = new cpactions.S3SourceAction({
                actionName: 'BucketSource',
                bucketKey: '/some/key',
                output: sourceOutput,
                bucket: sourceBucket,
            });
            pipeline.addStage({
                stageName: 'Stage1',
                actions: [sourceAction],
            });
            pipeline.addStage({
                stageName: 'Stage2',
                actions: [
                    new cpactions.CloudFormationCreateReplaceChangeSetAction({
                        actionName: 'Action1',
                        changeSetName: 'ChangeSet',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'SomeStack',
                        region: pipelineRegion,
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationCreateUpdateStackAction({
                        actionName: 'Action2',
                        templatePath: sourceOutput.atPath('template.yaml'),
                        stackName: 'OtherStack',
                        region: 'us-east-1',
                        adminPermissions: false,
                    }),
                    new cpactions.CloudFormationExecuteChangeSetAction({
                        actionName: 'Action3',
                        changeSetName: 'ChangeSet',
                        stackName: 'SomeStack',
                        region: 'us-west-1',
                    }),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': 'us-west-1',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'sfo-replication-bucket',
                        },
                    },
                    {
                        'Region': 'us-east-1',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'teststack-support-us-easteplicationbucket1a8063b3cdac6e7e0e73',
                        },
                    },
                    {
                        'Region': 'us-west-2',
                        'ArtifactStore': {
                            'Type': 'S3',
                            'EncryptionKey': {
                                'Type': 'KMS',
                                'Id': {},
                            },
                        },
                    },
                ],
                'Stages': [
                    {
                        'Name': 'Stage1',
                    },
                    {
                        'Name': 'Stage2',
                        'Actions': [
                            {
                                'Name': 'Action1',
                                'Region': 'us-west-2',
                            },
                            {
                                'Name': 'Action2',
                                'Region': 'us-east-1',
                            },
                            {
                                'Name': 'Action3',
                                'Region': 'us-west-1',
                            },
                        ],
                    },
                ],
            }));
            test.notEqual(pipeline.crossRegionSupport[pipelineRegion], undefined);
            test.notEqual(pipeline.crossRegionSupport['us-west-1'], undefined);
            const usEast1Support = pipeline.crossRegionSupport['us-east-1'];
            test.notEqual(usEast1Support, undefined);
            test.equal(usEast1Support.stack.region, 'us-east-1');
            test.equal(usEast1Support.stack.account, pipelineAccount);
            test.ok(usEast1Support.stack.node.id.indexOf('us-east-1') !== -1, `expected '${usEast1Support.stack.node.id}' to contain 'us-east-1'`);
            test.done();
        },
        'allows specifying only one of artifactBucket and crossRegionReplicationBuckets'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new codepipeline.Pipeline(stack, 'Pipeline', {
                    artifactBucket: new s3.Bucket(stack, 'Bucket'),
                    crossRegionReplicationBuckets: {
                    // even an empty map should trigger this validation...
                    },
                });
            }, /Only one of artifactBucket and crossRegionReplicationBuckets can be specified!/);
            test.done();
        },
        'does not create a new artifact Bucket if one was provided in the cross-region Buckets for the Pipeline region'(test) {
            const pipelineRegion = 'us-west-2';
            const stack = new core_1.Stack(undefined, undefined, {
                env: {
                    region: pipelineRegion,
                },
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                crossRegionReplicationBuckets: {
                    [pipelineRegion]: new s3.Bucket(stack, 'Bucket', {
                        bucketName: 'my-pipeline-bucket',
                    }),
                },
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.CodeCommitSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                repository: new codecommit.Repository(stack, 'Repo', { repositoryName: 'Repo' }),
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                project: new codebuild.PipelineProject(stack, 'Project'),
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::S3::Bucket', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': pipelineRegion,
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': {
                                'Ref': 'Bucket83908E77',
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
        'allows providing a resource-backed action from a different region directly'(test) {
            const account = '123456789012';
            const app = new core_1.App();
            const replicationRegion = 'us-west-1';
            const replicationStack = new core_1.Stack(app, 'ReplicationStack', { env: { region: replicationRegion, account } });
            const project = new codebuild.PipelineProject(replicationStack, 'CodeBuildProject', {
                projectName: 'MyCodeBuildProject',
            });
            const pipelineRegion = 'us-west-2';
            const pipelineStack = new core_1.Stack(app, 'TestStack', { env: { region: pipelineRegion, account } });
            const sourceOutput = new codepipeline.Artifact('SourceOutput');
            new codepipeline.Pipeline(pipelineStack, 'MyPipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [new cpactions.CodeCommitSourceAction({
                                actionName: 'CodeCommitAction',
                                output: sourceOutput,
                                repository: codecommit.Repository.fromRepositoryName(pipelineStack, 'Repo', 'my-codecommit-repo'),
                            })],
                    },
                    {
                        stageName: 'Build',
                        actions: [new cpactions.CodeBuildAction({
                                actionName: 'CodeBuildAction',
                                input: sourceOutput,
                                project,
                            })],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'ArtifactStores': [
                    {
                        'Region': replicationRegion,
                        'ArtifactStore': {
                            'Type': 'S3',
                            'Location': 'replicationstackeplicationbucket2464cd5c33b386483b66',
                            'EncryptionKey': {
                                'Id': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':kms:us-west-1:123456789012:alias/ionstacktencryptionalias043cb2f8ceac9da9c07c',
                                        ],
                                    ],
                                },
                                'Type': 'KMS',
                            },
                        },
                    },
                    {
                        'Region': pipelineRegion,
                    },
                ],
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuildAction',
                                'Region': replicationRegion,
                                'Configuration': {
                                    'ProjectName': 'MyCodeBuildProject',
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::S3::Bucket', {
                'BucketName': 'replicationstackeplicationbucket2464cd5c33b386483b66',
            }));
            test.done();
        },
    },
    'cross-account Pipeline': {
        'with a CodeBuild Project in a different account works correctly'(test) {
            const app = new core_1.App();
            const buildAccount = '901234567890';
            const buildRegion = 'bermuda-triangle-1';
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region: buildRegion },
            });
            const rolePhysicalName = 'ProjectRolePhysicalName';
            const projectRole = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
                roleName: rolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
                role: projectRole,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: '123456789012', region: buildRegion },
            });
            const sourceBucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket: sourceBucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            `:iam::${buildAccount}:role/buildstackebuildactionrole166c75d1d8be701b1ad8`,
                                        ],
                                    ],
                                },
                            },
                        ],
                    },
                ],
            }));
            assert_1.expect(buildStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                        // log permissions from the CodeBuild Project Construct...
                        },
                        {
                        // report group permissions from the CodeBuild Project construct...
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                            'Resource': [
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58',
                                        ],
                                    ],
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                'Ref': 'AWS::Partition',
                                            },
                                            ':s3:::pipelinestackeartifactsbucket5409dc84bb108027cb58/*',
                                        ],
                                    ],
                                },
                            ],
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                            'Resource': '*',
                        },
                    ],
                },
            }));
            test.done();
        },
        'adds a dependency on the Stack containing a new action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = new iam.Role(buildStack, 'ProjectRole', {
                assumedBy: new iam.AccountPrincipal(pipelineAccount),
                roleName: actionRolePhysicalName,
            });
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = codebuild.Project.fromProjectName(buildStack, 'Project', projectPhysicalName);
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { 'Ref': 'AWS::Partition' },
                                            `:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                                        ]],
                                },
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 1);
            test.done();
        },
        'does not add a dependency on the Stack containing an imported action Role'(test) {
            const region = 'us-west-2';
            const pipelineAccount = '123456789012';
            const buildAccount = '901234567890';
            const app = new core_1.App();
            const buildStack = new core_1.Stack(app, 'BuildStack', {
                env: { account: buildAccount, region },
            });
            const actionRolePhysicalName = 'ProjectRolePhysicalName';
            const actionRoleInOtherAccount = iam.Role.fromRoleArn(buildStack, 'ProjectRole', `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`);
            const projectPhysicalName = 'ProjectPhysicalName';
            const project = new codebuild.PipelineProject(buildStack, 'Project', {
                projectName: projectPhysicalName,
            });
            const pipelineStack = new core_1.Stack(app, 'PipelineStack', {
                env: { account: pipelineAccount, region },
            });
            const bucket = new s3.Bucket(pipelineStack, 'ArtifactBucket', {
                bucketName: 'source-bucket',
                encryption: s3.BucketEncryption.KMS,
            });
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                artifactBucket: bucket,
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3',
                                bucket,
                                bucketKey: 'path/to/file.zip',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'CodeBuild',
                                project,
                                input: sourceOutput,
                                role: actionRoleInOtherAccount,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'CodeBuild',
                                'Configuration': {
                                    'ProjectName': projectPhysicalName,
                                },
                                'RoleArn': `arn:aws:iam::${buildAccount}:role/${actionRolePhysicalName}`,
                            },
                        ],
                    },
                ],
            }));
            test.equal(pipelineStack.dependencies.length, 0);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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