#!/usr/bin/env python
#
# Correctness test for LAMMPS, comparison with result produced by yaff;
# calculates single-point energy with both LAMMPS and yaff, and compares the results.
# The energy difference should not exceed 1 kJ/mol, otherwise an error is raised.
# It should run to completion in a matter of seconds.
#
# If this test passes, there are no guarantees that the installation works fully correctly.
# But if it fails, it's a strong signal that there is something wrong with the (LAMMPS) installation.
#
# author: Veronique Van Speybroeck (Center for Molecular Modeling, Ghent University), April 2020

import numpy as np
import os
from mpi4py import MPI
from yaff import log, System, angstrom, ForceField, swap_noncovalent_lammps
from molmod import kjmol
import tempfile


def main():

    # create input files in temporary directory
    tmpdir = tempfile.mkdtemp()

    init_chk = os.path.join(tmpdir, 'init.chk')
    with open(init_chk, 'w') as fp:
        fp.write(INIT_CHK)

    pars_txt = os.path.join(tmpdir, 'pars.txt')
    with open(pars_txt, 'w') as fp:
        fp.write(PARS_TXT)

    # Setup MPI
    comm = MPI.COMM_WORLD
    rank = comm.Get_rank()

    # Set random seed, important to get the same velocities for all processes
    np.random.seed(5)
    # Turn off logging for all processes, it can be turned on for one selected process later on
    log.set_level(log.silent)
    if rank == 0:
        log.set_level(log.medium)

    # Load in the structure and the force field
    system = System.from_file(init_chk)

    # Initialize the Yaff and LAMMPS force fields
    rcut = 12 * angstrom
    ff_yaff = ForceField.generate(system, pars_txt, rcut=rcut, smooth_ei=False,
                                  gcut_scale=1.5, alpha_scale=3.2)  # , tailcorrections=True)
    ff = swap_noncovalent_lammps(ff_yaff, fn_system='lammps.dat', fn_log="log.lammps",
                                 suffix='', fn_table='lammps_smoothei2.table', comm=comm)

    # Print out the Yaff single-point energy
    print('Yaff energy')
    energy_yaff = ff_yaff.compute()
    print(energy_yaff / kjmol)
    for part in ff_yaff.parts:
        print('%s: %.3f kJ/mol' % (part.name, part.energy / kjmol))

    # Print out the LAMMPS single-point energy
    print('LAMMPS energy')
    energy_lammps = ff.compute()
    print(energy_lammps / kjmol)
    for part in ff.parts:
        print('%s: %.3f kJ/mol' % (part.name, part.energy / kjmol))

    assert np.abs(energy_yaff - energy_lammps) < 1 * kjmol, "The two energies are not the same"


INIT_CHK = """bonds                                     kind=intar 592,2
                    52                     14                     74                     14
                   130                     14                      4                     52
                     4                     92                    112                      4
                   152                     71                    152                    112
                   176                     74                    176                    127
                    92                    180                    130                    180
                   200                    176                    224                    180
                   152                    228                     52                    348
                    11                     52                     11                     71
                    11                    127                     53                     15
                    75                     15                    131                     15
                    53                      5                     93                      5
                   113                      5                    153                     70
                   153                    113                    177                     75
                   177                    126                    181                     93
                   131                    181                    201                    177
                   225                    181                    153                    229
                   349                     53                     10                     53
                    10                     70                     10                    126
                    12                     54                     72                     12
                   128                     12                      6                     54
                     6                     94                    114                      6
                   154                     69                    154                    114
                    72                    178                    178                    125
                    94                    182                    128                    182
                   202                    178                    226                    182
                   154                    230                     54                    350
                     9                     54                      9                     69
                     9                    125                      8                     41
                    56                      8                      8                    148
                    56                    165                    107                    165
                   173                    103                    145                    173
                   185                     97                    185                    148
                   185                    213                    221                    165
                   233                    173                    353                     41
                    41                      3                     41                     21
                    97                     21                      3                    103
                     3                    107                    145                     21
                     0                     36                     19                     36
                    36                     22                     48                      3
                    48                     16                     64                     16
                     3                     79                      3                    119
                    16                    140                     48                     13
                    81                     13                    121                     13
                    64                    156                    156                    119
                   168                     81                    168                    140
                   196                     79                    121                    196
                   168                    204                    216                    196
                   156                    244                    344                     48
                   240                    277                    236                    301
                   273                    204                    216                    313
                   249                    301                    313                    277
                   273                    341                    361                    249
                   385                    273                    389                    277
                   413                    301                    425                    313
                   453                    341                    249                    220
                   212                    341                      3                     39
                    16                     39                     21                     39
                    42                     11                     11                     59
                    11                    151                     59                    166
                   104                    166                    100                    174
                   146                    174                    186                     98
                   186                    151                    186                    214
                   166                    222                    234                    174
                   354                     42                      0                     42
                    42                     22                     98                     22
                     0                    100                    104                      0
                   146                     22                    241                    276
                   300                    237                    272                    205
                   312                    217                    248                    300
                   312                    276                    272                    340
                   360                    248                    384                    272
                   276                    388                    300                    412
                   424                    312                    340                    452
                   248                    221                    340                    213
                    17                     47                     20                     47
                    17                     61                     88                     20
                    17                    133                    136                     20
                     6                     47                      6                     86
                     6                    110                    163                     61
                   163                    110                     86                    191
                   136                    191                     88                    195
                   195                    133                    211                    195
                   163                    239                    243                    191
                    47                    359                    209                    282
                   241                    330                    262                    310
                   282                    318                    330                    270
                   262                    374                    270                    382
                   394                    282                    310                    422
                   318                    430                    442                    330
                   229                    262                    201                    318
                   217                    270                    245                    310
                     2                     31                      9                     31
                    12                     31                    225                    285
                   305                    229                    257                    245
                   333                    205                    289                    213
                   337                    233                    305                    257
                   289                    333                    337                    285
                   369                    257                    397                    285
                   401                    289                    417                    305
                   445                    333                    449                    337
                   252                    238                    210                    324
                   264                    202                    320                    226
                   296                    252                    320                    292
                   264                    324                    252                    364
                   376                    264                    292                    404
                   408                    296                    432                    320
                   324                    436                    234                    292
                   296                    222                     19                     45
                    45                     22                     19                     63
                    90                     22                     19                    135
                   138                     22                      4                     45
                     4                     84                      4                    108
                   161                     63                    161                    108
                    84                    189                    138                    189
                   193                     90                    193                    135
                   209                    193                    161                    237
                   241                    189                    357                     45
                    27                     15                     27                      5
                    18                     27                     25                     13
                    25                      7                     16                     25
                     1                     37                     18                     37
                    37                     23                    242                    279
                   238                    303                    275                    206
                   218                    315                    251                    303
                   315                    279                    275                    343
                   363                    251                    275                    387
                   279                    391                    303                    415
                   315                    427                    343                    455
                   251                    222                    214                    343
                   227                    287                    307                    231
                   259                    247                    207                    335
                   291                    215                    339                    235
                   259                    307                    291                    335
                   339                    287                    259                    371
                   287                    399                    291                    403
                   307                    419                    335                    447
                   339                    451                    280                    211
                   328                    243                    260                    308
                   280                    316                    328                    268
                   260                    372                    268                    380
                   392                    280                    308                    420
                   316                    428                    440                    328
                   260                    231                    203                    316
                   219                    268                    308                    247
                   226                    286                    306                    230
                   258                    246                    206                    334
                   290                    214                    338                    234
                   306                    258                    290                    334
                   338                    286                    370                    258
                   286                    398                    402                    290
                   418                    306                    334                    446
                   450                    338                     49                      2
                    49                     17                     65                     17
                     2                     78                      2                    118
                    17                    141                     49                     12
                    80                     12                    120                     12
                    65                    157                    157                    118
                    80                    169                    169                    141
                   197                     78                    120                    197
                   169                    205                    217                    197
                   245                    157                    345                     49
                    16                     46                     21                     46
                    16                     60                     89                     21
                    16                    132                    137                     21
                    46                      7                      7                     87
                     7                    111                    162                     60
                   162                    111                    190                     87
                   137                    190                     89                    194
                   194                    132                    210                    194
                   162                    238                    242                    190
                    46                    358                    253                    239
                   211                    325                    265                    203
                   321                    227                    297                    253
                   321                    293                    265                    325
                   365                    253                    377                    265
                   405                    293                    409                    297
                   433                    321                    437                    325
                   235                    293                    297                    223
                     1                     50                     50                     18
                    66                     18                      1                     77
                     1                    117                     18                    142
                    50                     15                     83                     15
                   123                     15                     66                    158
                   117                    158                    170                     83
                   170                    142                     77                    198
                   123                    198                    170                    206
                   218                    198                    158                    246
                   346                     50                    208                    283
                   240                    331                    263                    311
                   283                    319                    331                    271
                   263                    375                    271                    383
                   283                    395                    311                    423
                   319                    431                    331                    443
                   228                    263                    200                    319
                   216                    271                    244                    311
                    26                     14                     26                      4
                    26                     19                     18                     44
                    44                     23                     18                     62
                    91                     23                     18                    134
                   139                     23                     44                      5
                    85                      5                    109                      5
                   160                     62                    160                    109
                   188                     85                    139                    188
                   192                     91                    192                    134
                   208                    192                    160                    236
                   240                    188                     44                    356
                    34                     10                     34                     23
                    34                      5                    281                    210
                   329                    242                    309                    261
                   281                    317                    329                    269
                   373                    261                    381                    269
                   393                    281                    421                    309
                   429                    317                    441                    329
                   261                    230                    202                    317
                   218                    269                    309                    246
                   243                    278                    302                    239
                   274                    207                    314                    219
                   250                    302                    314                    278
                   274                    342                    362                    250
                   386                    274                    278                    390
                   302                    414                    426                    314
                   342                    454                    250                    223
                   342                    215                    224                    284
                   304                    228                    256                    244
                   204                    332                    288                    212
                   336                    232                    304                    256
                   288                    332                    336                    284
                   368                    256                    284                    396
                   400                    288                    416                    304
                   332                    444                    448                    336
                   237                    255                    209                    327
                   201                    267                    225                    323
                   299                    255                    323                    295
                   267                    327                    255                    367
                   267                    379                    295                    407
                   299                    411                    323                    435
                   327                    439                    233                    295
                   299                    221                     33                      9
                    33                     20                     33                      6
                    40                      9                     57                      9
                     9                    149                     57                    164
                   106                    164                    172                    102
                   144                    172                    184                     96
                   184                    149                    184                    212
                   164                    220                    232                    172
                   352                     40                     40                      2
                    40                     20                     96                     20
                     2                    102                    106                      2
                   144                     20                      0                     29
                    11                     29                     29                     14
                     0                     51                     19                     51
                    19                     67                      0                     76
                     0                    116                     19                    143
                    51                     14                     82                     14
                   122                     14                     67                    159
                   116                    159                     82                    171
                   171                    143                     76                    199
                   122                    199                    171                    207
                   219                    199                    159                    247
                    51                    347                      2                     38
                    17                     38                     20                     38
                    10                     43                     58                     10
                    10                    150                     58                    167
                   105                    167                    101                    175
                   147                    175                     99                    187
                   187                    150                    187                    215
                   167                    223                    235                    175
                    43                    355                      1                     43
                    43                     23                     99                     23
                     1                    101                    105                      1
                   147                     23                     32                      8
                    32                     21                     32                      7
                    13                     55                     73                     13
                   129                     13                      7                     55
                     7                     95                    115                      7
                   155                     68                    115                    155
                    73                    179                    179                    124
                    95                    183                    129                    183
                   203                    179                    227                    183
                   155                    231                     55                    351
                     8                     55                      8                     68
                     8                    124                      3                     30
                     8                     30                     13                     30
                   236                    254                    208                    326
                   200                    266                    224                    322
                   298                    254                    322                    294
                   266                    326                    254                    366
                   378                    266                    294                    406
                   410                    298                    434                    322
                   326                    438                    232                    294
                   298                    220                     11                     35
                    35                     22                     35                      4
                    24                     12                     24                      6
                    24                     17                      1                     28
                    10                     28                     28                     15
charges                                   kind=fltar 456
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
 2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00  2.361566205200000e+00
-1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00
-1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00
-1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00
-1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00 -1.093870316700000e+00
-1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00
-1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00
-1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00
-1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00 -1.055042603900000e+00
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
-6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01 -6.831393458000000e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
 7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01  7.815150188000001e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01 -1.544640324000000e-01
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
-8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02 -8.298140200000001e-02
 4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01
 4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01
 4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01
 4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01  4.906092530000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
 1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01  1.385876479000000e-01
dipoles                                   kind=none   None
ffatype_ids                               kind=intar 456
                     0                      0                      0                      0
                     0                      0                      0                      0
                     0                      0                      0                      0
                     0                      0                      0                      0
                     0                      0                      0                      0
                     0                      0                      0                      0
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     2                      2                      2                      2
                     2                      2                      2                      2
                     2                      2                      2                      2
                     2                      2                      2                      2
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     3                      3                      3                      3
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     4                      4                      4                      4
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     5                      5                      5                      5
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     7                      7                      7                      7
                     7                      7                      7                      7
                     7                      7                      7                      7
                     7                      7                      7                      7
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
ffatypes                                  kind=strar 9
                    ZR                   O_OX                   O_OH                   O_CA
                  C_CA                   C_PC                   C_PH                   H_OH
                  H_PH
masses                                    kind=none   None
numbers                                   kind=intar 456
                    40                     40                     40                     40
                    40                     40                     40                     40
                    40                     40                     40                     40
                    40                     40                     40                     40
                    40                     40                     40                     40
                    40                     40                     40                     40
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     8                      8                      8                      8
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     6                      6                      6                      6
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
                     1                      1                      1                      1
pos                                       kind=fltar 456,3
 3.490725830842442e+01  1.989736115878592e+01  1.989640487320012e+01  3.490808649393055e+01
 3.979513994951059e+01 -1.300625055309490e-03  1.501055756938832e+01  1.989535638139978e+01
 1.246179555604761e-03  1.501137588505900e+01  3.979320094458823e+01  1.989720602732817e+01
 4.874308149304886e+00  1.989583090118305e+01  1.989787247092695e+01  4.874894653776321e+00
 3.979257955564209e+01  5.968951244624455e-04  2.477156000121577e+01  1.989794132834076e+01
-6.649172855657153e-04  2.477213204653341e+01  3.979471652734251e+01  1.989575923404384e+01
 1.989201946003435e+01  3.491390226935287e+01  1.989723959501853e+01  1.989176072681321e+01
 1.501673299775960e+01 -1.129256244541729e-03  3.978764801128609e+01  3.491304735065996e+01
 1.057641614954819e-03  3.978733906375515e+01  1.501590858399707e+01  1.989643443478721e+01
 1.989021170080069e+01  1.989797529016069e+01  3.491309701053401e+01  1.989131281056547e+01
 3.979348778678292e+01  1.501588819750306e+01  3.978803993526803e+01  1.989696868977292e+01
 1.501699610198728e+01  3.978913215770860e+01  3.979262010168185e+01  3.491343905174566e+01
 1.989046538478880e+01  4.880841585670589e+00  1.989494772283481e+01  1.989057374713003e+01
 2.477729070125715e+01  4.719741553780736e-04  3.978878868209046e+01  4.880309341815733e+00
-4.301581744627571e-04  3.978892228577907e+01  2.477678835910098e+01  1.989859264118343e+01
 1.989148884038907e+01  1.989658031878211e+01  4.880302130079328e+00  1.989302042109499e+01
 3.979559229700724e+01  2.477684310638939e+01  3.978633008797122e+01  1.989499819501557e+01
 2.477750655215571e+01  3.978786594705887e+01  3.979386468015720e+01  4.880389304875714e+00
 2.202461643802651e+01  2.203189112311701e+01  3.765944899136999e+01  2.202531036798885e+01
 2.134740604106842e+00  1.776133298387446e+01  2.129765527659656e+00  2.203148419271726e+01
 1.776286517993534e+01  2.130436098341955e+00  2.134407460468690e+00  3.765846622785212e+01
 3.765364101627898e+01  3.765888318746777e+01  3.765879402579836e+01  3.765331908779535e+01
 1.776255570111377e+01  1.776226963765306e+01  1.775666476589178e+01  3.765859689708251e+01
 1.776200250148918e+01  1.775636570738458e+01  1.776230133401189e+01  3.765909028320628e+01
 2.202672939433552e+01  3.766037150215158e+01  2.203097986183339e+01  2.202631608352776e+01
 1.776234599247950e+01  2.134213937499734e+00  2.128786365030469e+00  3.765879446242553e+01
 2.135086538364289e+00  2.128334805551631e+00  1.776082738113674e+01  2.203203177401806e+01
 3.765329158533921e+01  2.203073913533792e+01  2.203181153117562e+01  3.765410410304322e+01
 2.134932100735816e+00  2.133688919571228e+00  1.775586776786592e+01  2.203127523481187e+01
 2.135670833274500e+00  1.775671455087363e+01  2.135559199454761e+00  2.203111303100527e+01
 1.762577734966588e+01  1.763000570487011e+01  2.266330822679377e+00  1.762650624498306e+01
 3.752875276030682e+01  2.216275038100298e+01  3.752050766793352e+01  1.762948080272583e+01
 2.216322617063571e+01  3.752126841974270e+01  3.752811262536633e+01  2.265957839948938e+00
 2.261537912485077e+00  2.265729289858605e+00  2.265998740387887e+00  2.260329120181068e+00
 2.216117704414238e+01  2.216406386780314e+01  2.215772730325949e+01  2.268087772864094e+00
 2.216185216065963e+01  2.215650720606797e+01  2.216346869842136e+01  2.266326751780221e+00
 1.762407035216446e+01  2.265842029938293e+00  1.763000274242215e+01  1.762402526074742e+01
 2.216292137234051e+01  3.752887572170109e+01  3.752297244443796e+01  2.266345156290114e+00
 3.752607068387407e+01  3.752292973172899e+01  2.216340736083256e+01  1.763120267852185e+01
 2.260410474901676e+00  1.763088489134654e+01  1.763105821224282e+01  2.261022444448104e+00
 3.752716039127341e+01  3.752806178485923e+01  2.215703687890538e+01  1.763106712433468e+01
 3.752683455326825e+01  2.215763882359795e+01  3.752726086333593e+01  1.763026158731759e+01
 1.612738088836549e+01  3.302533778908620e+01  2.003984794891178e+01  1.612731613417120e+01
 1.312766242903430e+01  1.393784446245362e-01  3.602237942245713e+01  3.302688557668290e+01
 1.336174039388602e-01  3.602227098524808e+01  1.312927418068616e+01  2.002680785221010e+01
 2.365498542535781e+01  6.769074757939987e+00  2.002635382191530e+01  2.365500685717470e+01
 2.666617394718741e+01  1.412707297290573e-01  3.760418659459016e+00  6.766650584719018e+00
 1.318037843702247e-01  3.760457177349364e+00  2.666375182178951e+01  2.004022690334620e+01
 1.612528143643354e+01  6.767097824203487e+00  1.976273718790822e+01  1.612523916539509e+01
 2.666352562681059e+01  3.966262386478711e+01  3.602441305318901e+01  6.769318338518314e+00
 3.965201139308716e+01  3.602440379697210e+01  2.666578382452883e+01  1.975787140941610e+01
 2.365671567575621e+01  3.302657742274238e+01  1.976444893991181e+01  2.365670486219636e+01
 1.312989855683247e+01  3.966000342369477e+01  3.758748749953809e+00  3.302470531337752e+01
 3.965437653497323e+01  3.758723999249336e+00  1.312801586164986e+01  1.975637385138161e+01
 2.002184863407880e+01  1.613346505188802e+01  3.302473883007246e+01  2.002197289124604e+01
 3.602810468126447e+01  1.312926947232245e+01  1.367082338189690e-01  1.613261305267060e+01
 1.312846150972057e+01  1.368852196166233e-01  3.602742198898699e+01  3.302655767808238e+01
 3.302040837247949e+01  2.004004219509145e+01  1.613126321395754e+01  3.302171468849843e+01
 1.339550457157747e-01  3.602721125685146e+01  1.312168471554626e+01  2.003596460621149e+01
 3.603032797522945e+01  1.312300101336320e+01  1.299388364031086e-01  1.613262606149094e+01
 1.974570901927019e+01  2.366325071637388e+01  3.302633042641153e+01  1.975025798565546e+01
 3.764227587050050e+00  1.312719456401017e+01  3.965627467935057e+01  2.366228684834167e+01
 1.313040914493622e+01  3.966058902913779e+01  3.763443724451319e+00  3.302506543319744e+01
 6.761061605966100e+00  2.002909549751064e+01  2.366299714003316e+01  6.762419925076353e+00
 1.306631168260614e-01  3.765354530141899e+00  2.665923351994980e+01  2.003919387995704e+01
 3.764283077950346e+00  2.666057056535371e+01  1.409459540052401e-01  2.366032672794897e+01
 2.002276797307826e+01  2.366128056142727e+01  6.768489650517730e+00  2.002335749192145e+01
 3.767118915544936e+00  2.666318227829811e+01  1.353561089701583e-01  2.365954347220269e+01
 2.666633018845635e+01  1.359144051724433e-01  3.765186960268605e+00  6.767508539746105e+00
 6.762372935307679e+00  1.976218024753378e+01  1.613337749116571e+01  6.763322579988857e+00
 3.965965385602520e+01  3.602973569058194e+01  2.665831602525251e+01  1.975794207783412e+01
 3.602778715627186e+01  2.665929230927920e+01  3.965505961599392e+01  1.613053989336842e+01
 1.975816760699984e+01  1.613154259519901e+01  6.766902427345431e+00  1.974859444413245e+01
 3.603079508399954e+01  2.666528581190256e+01  3.965774147924260e+01  1.613010897504268e+01
 2.666432519015519e+01  3.964836314388970e+01  3.602918489878213e+01  6.768965429210017e+00
 3.301905972503401e+01  1.976430008312319e+01  2.366060689044069e+01  3.302006072637284e+01
 3.966224338434107e+01  3.763234929262723e+00  1.312334670151796e+01  1.975504589289441e+01
 3.766416647857637e+00  1.312434492210792e+01  3.965324384079992e+01  2.366270507752447e+01
 3.301950714223298e+01  1.613276804758171e+01  2.002645820182657e+01  3.301986897404555e+01
 3.603051998232966e+01  1.314988135041166e-01  1.312358457612722e+01  1.613025759663716e+01
 1.415055264284224e-01  1.312390868917699e+01  3.602808549953281e+01  2.004013952859266e+01
 6.763290683819170e+00  2.366025326910892e+01  2.003956523794497e+01  6.763158961257040e+00
 3.763205542795933e+00  1.318388397269749e-01  2.665850003634340e+01  2.366337918885094e+01
 1.411759155409970e-01  2.665839516949183e+01  3.766344007689240e+00  2.002699494901033e+01
 6.762101013310686e+00  1.613097528116441e+01  1.975831801028227e+01  6.762315863480681e+00
 3.602748860718660e+01  3.965405243376151e+01  2.665936672231643e+01  1.613336088963971e+01
 3.965979670007713e+01  2.665956327231921e+01  3.602986646059846e+01  1.976269607944013e+01
 3.302082841722822e+01  2.366306534385745e+01  1.975646236830511e+01  3.302090302941561e+01
 3.766520306973559e+00  3.965207390224676e+01  1.312251280487792e+01  2.366008615429296e+01
 3.966192258716966e+01  1.312260994416990e+01  3.763585958017277e+00  1.976457660862210e+01
 1.612441575700581e+01  2.003737628478283e+01  3.302667362056058e+01  1.612579002578034e+01
 1.300943615919166e-01  1.312929183554348e+01  3.602385461209880e+01  2.003980285143647e+01
 1.312835912727251e+01  3.602521570582152e+01  1.326126129270848e-01  3.302462604224628e+01
 2.002237388759288e+01  3.302542651164779e+01  1.613272634740917e+01  2.002283097144105e+01
 1.312989650757725e+01  3.602729144147526e+01  1.359486117124403e-01  3.302472864920428e+01
 3.603021827855345e+01  1.363774176440271e-01  1.312908901372393e+01  1.613127922476698e+01
 2.365474508978707e+01  1.975805959178482e+01  3.302489375434092e+01  2.365582793051143e+01
 3.965333108119757e+01  1.312759813226865e+01  3.759593281615865e+00  1.976353965391668e+01
 1.313009378981553e+01  3.760648541044594e+00  3.965951782488528e+01  3.302640175483977e+01
 2.002108500568356e+01  6.769639743094411e+00  2.365960668883464e+01  2.002163482344500e+01
 2.666450545568627e+01  3.765702798044551e+00  1.371212288279759e-01  6.768226947003763e+00
 3.764085113435486e+00  1.376581682393422e-01  2.666323425010762e+01  2.366364349018840e+01
 2.365627916904770e+01  2.003828360473675e+01  6.767686206404353e+00  2.365782489480705e+01
 1.419852084602276e-01  2.666398547283920e+01  3.757574288722359e+00  2.002804095734190e+01
 2.666557499549343e+01  3.759137131165368e+00  1.315951885129683e-01  6.768189733765365e+00
 1.975012398629565e+01  6.767812764910553e+00  1.613003602182881e+01  1.974648029687565e+01
 2.666604139189277e+01  3.602990672505781e+01  3.965980196493943e+01  6.766864834421964e+00
 3.602775511780144e+01  3.965656552988282e+01  2.666497095133764e+01  1.613377602848543e+01
 1.612688033848156e+01  1.975583736606142e+01  6.769177718413183e+00  1.612788536211239e+01
 3.965478529834676e+01  2.666538980316798e+01  3.602174832606473e+01  1.976282697384082e+01
 2.666416652842498e+01  3.602275648993071e+01  3.966118561074479e+01  6.766715134787656e+00
 1.974852660756867e+01  3.302723074932664e+01  2.366260586375733e+01  1.975887704013076e+01
 1.312889263768425e+01  3.763466875326005e+00  3.964795683860110e+01  3.302561618320703e+01
 3.766146355335531e+00  3.965779704774696e+01  1.312740553030542e+01  2.366079595410458e+01
 6.070393232326075e+00  1.381933706730796e+01  1.979313958501530e+01  6.070471408146224e+00
 3.371591174479433e+01  3.969036761895347e+01  2.596783518999552e+01  1.382208783733355e+01
 3.971929472589517e+01  2.596788992018665e+01  3.371866820841480e+01  1.982353971784280e+01
 1.381424310577510e+01  6.074865117949615e+00  1.982335856793499e+01  1.381416819778865e+01
 2.597129560299399e+01  3.972213953575969e+01  3.371263185842697e+01  6.078150646602359e+00
 3.968771830132448e+01  3.371260764874877e+01  2.597463522592599e+01  1.979325482015008e+01
 6.071483900165442e+00  6.074449799909492e+00  7.236782374018715e-02  6.072049100844775e+00
 2.597200804076883e+01  2.000532892129358e+01  2.596622477296280e+01  6.077413377537295e+00
 1.996568623469416e+01  2.596679116572648e+01  2.597498695728957e+01  1.051286467109458e-01
 1.381552098889720e+01  1.381872516087529e+01  1.043858578372248e-01  1.381562643317187e+01
 3.371647841617332e+01  2.000523014592194e+01  3.371119906842123e+01  1.382154433182282e+01
 1.996582988029359e+01  3.371133037630351e+01  3.371919486592358e+01  7.307237473549275e-02
 1.978604205769553e+01  6.075965011519122e+00  1.381845043505683e+01  1.978062943042749e+01
 2.597484531270559e+01  3.371811247719431e+01  3.972121224358062e+01  6.074671765569149e+00
 3.371672196307043e+01  3.971632947131886e+01  2.597339444562062e+01  1.382252554427384e+01
 1.381511813506187e+01  1.979081527893156e+01  6.078005149428230e+00  1.381614027082065e+01
 3.968998167641033e+01  2.597427225406033e+01  3.371065250110522e+01  1.982316375095708e+01
 2.597189458322347e+01  3.371167642967917e+01  3.972166730162139e+01  6.074502304910115e+00
 1.017823148820772e-01  1.382083681627463e+01  1.381966838343492e+01  1.016723266520908e-01
 3.371584994645583e+01  3.371844212121823e+01  1.996152380957845e+01  1.382219812315616e+01
 3.371624729825850e+01  1.996138578455740e+01  3.371705844856085e+01  1.382150204851188e+01
 6.070733681004529e+00  1.982221614461082e+01  1.382214115058187e+01  6.071733248281731e+00
 3.971946359517061e+01  3.371850189123408e+01  2.596650715229699e+01  1.979367757564411e+01
 3.371617677123812e+01  2.596755002315471e+01  3.969016960899948e+01  1.381892306913047e+01
 1.981833251995918e+01  1.382045279181060e+01  6.074732363763417e+00  1.978290291603092e+01
 3.371899708180666e+01  2.597422355619830e+01  3.971898447817124e+01  1.381907949100311e+01
 2.597199366490836e+01  3.968411603194392e+01  3.371744029580114e+01  6.077732087982101e+00
 6.070304077918620e+00  7.013824118642557e-02  6.076477281148843e+00  6.068828833688231e+00
 1.996866676987030e+01  2.597406230101006e+01  2.596942141712212e+01  1.058151271052866e-01
 2.597210393971820e+01  2.596793498137526e+01  2.000417416696340e+01  6.075992250926570e+00
 1.019418554937276e-01  6.076832173324530e+00  6.075799373709719e+00  1.016526624744318e-01
 2.597132779031803e+01  2.597520360598918e+01  1.996152308122143e+01  6.078150149712796e+00
 2.597089180181899e+01  1.996123581569601e+01  2.597246444240781e+01  6.076813840776871e+00
 1.381471798005188e+01  6.925573882686667e-02  1.382140747150762e+01  1.381289132850737e+01
 2.000057639307594e+01  3.371856733658931e+01  3.371388404817216e+01  7.384027953358849e-02
 3.371612406308586e+01  3.371209663413830e+01  2.000512599147392e+01  1.381965040346558e+01
 8.672218495339466e-02  1.182113723457815e+01  1.181976662645888e+01  8.647357086259645e-02
 3.171574447001204e+01  3.171896211118905e+01  1.995581763268942e+01  1.182249444690000e+01
 3.171589190994230e+01  1.995555548181683e+01  3.171691935369907e+01  1.182159907104908e+01
 1.980518242918080e+01  8.075723288088817e+00  1.181855888400319e+01  1.979485639753668e+01
 2.797486502931759e+01  3.171845572585710e+01  3.972788797135372e+01  8.074480802549155e+00
 3.171655894678791e+01  3.971863022968562e+01  2.797343752985148e+01  1.182260057382002e+01
 8.681805591030312e-02  8.076748381181810e+00  8.075517086361035e+00  8.649642558696696e-02
 2.797105971007662e+01  2.797506979613716e+01  1.995578366624222e+01  8.078250622361793e+00
 2.797083786278593e+01  1.995546899629196e+01  2.797236273169486e+01  8.076925168362079e+00
 1.982004072849675e+01  1.182032349308245e+01  8.074416886154317e+00  1.979617177880267e+01
 3.171930899339469e+01  2.797417314519791e+01  3.972660524613276e+01  1.181895861617269e+01
 2.797188572127837e+01  3.970390833615899e+01  3.171775381447850e+01  8.077844263412164e+00
 1.181464594528382e+01  6.364404154195957e-02  1.182144072667786e+01  1.181289182593741e+01
 1.998131907233046e+01  3.171883045703952e+01  3.171407599961245e+01  7.222992768481579e-02
 3.171605315609878e+01  3.171234753579476e+01  1.998982744517894e+01  1.181979456383545e+01
 1.181590487102266e+01  1.181868769157720e+01  8.513476984165391e-02  1.181602475200967e+01
 3.171643406794604e+01  1.999018376435125e+01  3.171098277688720e+01  1.182176428651959e+01
 1.995995491826899e+01  3.171112767781727e+01  3.171943303282861e+01  7.145057903744415e-02
 8.070820696763668e+00  1.982298145635792e+01  1.182232316837899e+01  8.071862883416708e+00
 3.972183845723185e+01  3.171869179758548e+01  2.796618792385944e+01  1.981221058707145e+01
 3.171617637041888e+01  2.796725926203570e+01  3.970967635530060e+01  1.181892138516928e+01
 8.070058893997910e+00  1.181934841040094e+01  1.981198824433493e+01  8.070079138860185e+00
 3.171585776933837e+01  3.970963272277147e+01  2.796803236167278e+01  1.182233286373358e+01
 3.972086493743659e+01  2.796804321000422e+01  3.171885314215038e+01  1.982611246226420e+01
 1.181536689045192e+01  1.980868572401337e+01  8.077966288373313e+00  1.181640674181759e+01
 3.970865358128022e+01  2.797427141448886e+01  3.171056688977725e+01  1.982540184660307e+01
 2.797169546342603e+01  3.171161096692349e+01  3.972412161428723e+01  8.074354740892542e+00
 8.071557650782674e+00  8.074247555177745e+00  7.050062415355079e-02  8.071937158277732e+00
 2.797179511031450e+01  1.999022016786892e+01  2.796615149304444e+01  8.077453993368954e+00
 1.995989620307934e+01  2.796653645085193e+01  2.797498989324110e+01  8.610397791641479e-02
 8.070192812496019e+00  6.419249672714625e-02  8.076634488707834e+00  8.068576960257134e+00
 1.996676883827650e+01  2.797423034732491e+01  2.796948134292814e+01  8.681848309330077e-02
 2.797174613304083e+01  2.796786900495130e+01  1.998923487678351e+01  8.075657974312703e+00
 1.181421957081106e+01  8.074719134443027e+00  1.982560272853746e+01  1.181418439204753e+01
 2.797118276917622e+01  3.972415117171580e+01  3.171281335341664e+01  8.078079782490921e+00
 3.970664144464227e+01  3.171281828807571e+01  2.797458778876385e+01  1.981255358789447e+01
 1.250582840368720e+01  2.914666489505505e+01  1.998581534089329e+01  1.250555392452712e+01
 9.248897282293649e+00  7.835262295367966e-02  3.240056679838487e+01  2.914956750159762e+01
 7.216198845370839e-02  3.240029035817515e+01  9.251877777028364e+00  1.995832468698563e+01
 2.727682685020807e+01  1.064733679832643e+01  1.995805679017008e+01  2.727706445068976e+01
 3.054483721808866e+01  7.934678771389009e-02  7.381941531267555e+00  1.064405630013955e+01
 7.118433105206592e-02  7.382150352822855e+00  3.054156769814290e+01  1.998590571660755e+01
 1.250394035343708e+01  1.064451216149219e+01  1.982367589490861e+01  1.250394150469548e+01
 3.054096543236675e+01  3.972523888817460e+01  3.240217001267826e+01  1.064795009276426e+01
 3.971225738628854e+01  3.240219080493355e+01  3.054444885832707e+01  1.981950221232768e+01
 2.727870998158595e+01  2.914897470662509e+01  1.982820340944857e+01  2.727870467008315e+01
 9.252447607585127e+00  3.972124509535219e+01  7.380286044239628e+00  2.914610026781122e+01
 3.971829193564196e+01  7.380313442766089e+00  9.249563638323313e+00  1.981633403544856e+01
 1.995358184209804e+01  1.251219148170519e+01  2.914608716823374e+01  1.995370776589150e+01
 3.240640230252035e+01  9.251744125897076e+00  8.244779772052184e-02  1.251074774892251e+01
 9.249961791504605e+00  8.262260277302413e-02  3.240515295940634e+01  2.914910926924576e+01
 2.914257287367412e+01  1.998549079972283e+01  1.250955281191710e+01  2.914426401275458e+01
 7.350297028980025e-02  3.240568646705869e+01  9.243054471566870e+00  1.997402188651311e+01
 3.240830698776938e+01  9.244761484098495e+00  6.189148939201462e-02  1.251077883534666e+01
 1.979825586453108e+01  2.728530134522020e+01  2.914863567309117e+01  1.981218172973174e+01
 7.386247757270069e+00  9.248727792595064e+00  3.971809964485507e+01  2.728381138884808e+01
 9.252779292377982e+00  3.973057336130524e+01  7.384935655012308e+00  2.914672192417138e+01
 1.063843546879799e+01  1.996787866602766e+01  2.728478595396909e+01  1.064007827185785e+01
 6.248503884654118e-02  7.387304932994692e+00  3.053764264773101e+01  1.998488054863796e+01
 7.385891618028505e+00  3.053927242401543e+01  7.958304302831659e-02  2.728208107966552e+01
 1.995389339736400e+01  2.728286897274880e+01  1.064677747535507e+01  1.995465237533794e+01
 7.388576086402149e+00  3.054065125537959e+01  8.156017581817152e-02  2.728163032850796e+01
 3.054491321143102e+01  8.237271191933480e-02  7.387147515844949e+00  1.064532433867220e+01
 1.064064144911362e+01  1.982117919575010e+01  1.251193295058655e+01  1.064169543905434e+01
 3.972171227461700e+01  3.240826441885544e+01  3.053600713271579e+01  1.981846056839645e+01
 3.240571637965510e+01  3.053708318279578e+01  3.971870503406586e+01  1.250841471409844e+01
 1.981917140243515e+01  1.250976775384085e+01  1.064427029807278e+01  1.979937368957093e+01
 3.240888823917486e+01  3.054399175844921e+01  3.972946762609432e+01  1.250848906475585e+01
 3.054172390240114e+01  3.970975878197266e+01  3.240740815167378e+01  1.064764652507601e+01
 2.914074075046528e+01  1.982519073371558e+01  2.728215597735856e+01  2.914177144439387e+01
 3.972410873815303e+01  7.384869490924587e+00  9.245549147409207e+00  1.981431908474011e+01
 7.388356027877606e+00  9.246581232590794e+00  3.971488949564753e+01  2.728469747473889e+01
 2.914117494752227e+01  1.251141673142221e+01  1.995809668274219e+01  2.914130571311315e+01
 3.240900361129270e+01  7.183463838087101e-02  9.246077592444484e+00  1.250823884053288e+01
 7.868624859847995e-02  9.246172523504816e+00  3.240587342867793e+01  1.998587772349170e+01
 1.064178459359191e+01  2.728234418637113e+01  1.998584040987460e+01  1.064138989865301e+01
 7.384695265489253e+00  7.093542359955422e-02  3.053635310448776e+01  2.728542024374890e+01
 7.958732361744429e-02  3.053596410725073e+01  7.387813094763260e+00  1.995828969742503e+01
 1.063989242559662e+01  1.250880220686596e+01  1.982033324739158e+01  1.063993698584806e+01
 3.240526724604702e+01  3.971423354355418e+01  3.053783607612447e+01  1.251200900396604e+01
 3.972176911631418e+01  3.053785409584417e+01  3.240851832576362e+01  1.982573833443115e+01
 2.914301260858894e+01  2.728492033812162e+01  1.981829124652232e+01  2.914301309702387e+01
 7.388401395221556e+00  3.971173552946184e+01  9.244326650155632e+00  2.728176840956382e+01
 3.972209528333754e+01  9.244371446135403e+00  7.385277513573646e+00  1.982694757587705e+01
 1.250253644651036e+01  1.997532718677503e+01  2.914902816664720e+01  1.250420067224088e+01
 6.225125755118907e-02  9.251603872392659e+00  3.240189573532817e+01  1.998513356865633e+01
 9.249973964243997e+00  3.240353477187163e+01  7.211970693129093e-02  2.914620118676577e+01
 1.995393787550432e+01  2.914706199942743e+01  1.251102892710483e+01  1.995465351697644e+01
 9.252576110802481e+00  3.240506344656923e+01  8.156487948945704e-02  2.914600353430419e+01
 3.240888336278172e+01  8.232440982266527e-02  9.251338817625795e+00  1.250944608926462e+01
 2.727657950440097e+01  1.981839115286423e+01  2.914636593689280e+01  2.727763665353707e+01
 3.971602095410041e+01  9.249111685914333e+00  7.381327631726466e+00  1.982165962042658e+01
 9.252482255200558e+00  7.382371631377547e+00  3.972168759197840e+01  2.914884752711017e+01
 1.995356941647684e+01  1.064816574879485e+01  2.728164498581899e+01  1.995365955642835e+01
 3.054222718582404e+01  7.387503871390357e+00  8.250265477260228e-02  1.064654019438886e+01
 7.385830461303763e+00  8.263054034687108e-02  3.054080418095189e+01  2.728516779888928e+01
 2.727833052046908e+01  1.998519974506595e+01  1.064548531502190e+01  2.728002383833265e+01
 8.094385793664415e-02  3.054159192237094e+01  7.378931128107247e+00  1.996660099670826e+01
 3.054403909115141e+01  7.380628325579808e+00  6.215919869969651e-02  1.064646509416267e+01
 1.981159637597344e+01  1.064549576767454e+01  1.250830977419344e+01  1.979873853028092e+01
 3.054466173689181e+01  3.240812916897033e+01  3.973008654417936e+01  1.064434535235626e+01
 3.240597389691740e+01  3.972019207480745e+01  3.054331817249836e+01  1.251195501748814e+01
 1.250488334235093e+01  1.981389887899715e+01  1.064779539996564e+01  1.250593009569637e+01
 3.971496496937775e+01  3.054410045150013e+01  3.240014293036852e+01  1.982521256729928e+01
 3.054152070995434e+01  3.240120559029220e+01  3.972313179566277e+01  1.064417987929424e+01
 1.979887820564971e+01  2.914950054984696e+01  2.728451372266434e+01  1.981975232771712e+01
 9.250539573816612e+00  7.384667021286449e+00  3.971126979682839e+01  2.914788966709210e+01
 7.388480626223573e+00  3.972996575126965e+01  9.249105286255935e+00  2.728245773618650e+01
 1.656140352983275e+01  3.323475259180652e+00  1.657133431343251e+01  1.656228486168100e+01
 2.322349657026098e+01  3.647247979787997e+01  3.646546169865879e+01  3.325015483781570e+00
 3.646325433420166e+01  3.646639285916446e+01  2.322505375654974e+01  1.657064509644183e+01
 3.321383769169834e+00  1.657394415365840e+01  1.657024029142375e+01  3.322075242770350e+00
 3.646992948505324e+01  3.646762546863646e+01  2.321523144537045e+01  1.656909322726821e+01
 3.646800862694339e+01  2.321590504485378e+01  3.646489593582192e+01  1.657191337305349e+01
 1.656956411173918e+01  1.656842604137276e+01  3.327274985978196e+00  1.656729775746637e+01
 3.646973268922548e+01  2.322301421175649e+01  3.646045320739267e+01  1.656928168460218e+01
 2.322219907283236e+01  3.645832042111462e+01  3.647038251516410e+01  3.324250667262286e+00
 3.324106003631590e+00  3.323659830263010e+00  3.324461885732755e+00  3.321163002993453e+00
 2.321719133232705e+01  2.322591938551336e+01  2.321612567307243e+01  3.331304292322689e+00
 2.321923913577702e+01  2.321318355210028e+01  2.322475152249319e+01  3.327115786599175e+00
 1.448944722131224e+01  2.861313146337814e+01  1.998477602906750e+01  1.448914595640550e+01
 8.715294775229181e+00  7.347930002723484e-02  3.438410506576655e+01  2.861576865638980e+01
 6.835783933128936e-02  3.438380019536345e+01  8.718022713931383e+00  1.995076562189102e+01
 2.529331214772399e+01  1.118116209974402e+01  1.995028455263344e+01  2.529350580934633e+01
 3.107854588758818e+01  7.431730396495140e-02  5.398369723788479e+00  1.117774779924335e+01
 6.754427777628247e-02  5.398536456241224e+00  3.107513017847854e+01  1.998492936122963e+01
 1.448752526350660e+01  1.117810789195383e+01  1.983223375976018e+01  1.448754052791589e+01
 3.107454260602082e+01  3.973070753958378e+01  3.438570500463899e+01  1.118175374742984e+01
 3.971629871794456e+01  3.438572084205047e+01  3.107827581579216e+01  1.981814105818846e+01
 2.529519522967835e+01  2.861511043460138e+01  1.983375422213745e+01  2.529519432716770e+01
 8.718561551334329e+00  3.972578965580740e+01  5.396661366581465e+00  2.861258996078253e+01
 3.971860137850852e+01  5.396699257838535e+00  8.716022068188586e+00  1.981927884398634e+01
 1.994557227591399e+01  1.449580384176039e+01  2.861260018181787e+01  1.994603400832985e+01
 3.438996210394035e+01  8.718040855201783e+00  8.140600259372471e-02  1.449431250987591e+01
 8.716262550446167e+00  8.195334433150248e-02  3.438866709098891e+01  2.861520523568056e+01
 2.860897736649211e+01  1.998473304627838e+01  1.449315335058396e+01  2.861058284317422e+01
 7.030640141129860e-02  3.438925690276886e+01  8.709353203848824e+00  1.996857236937689e+01
 3.439187149971026e+01  8.710955403386045e+00  5.397955675313217e-02  1.449430252135673e+01
 1.979889453654178e+01  2.530173130694460e+01  2.861492197952455e+01  1.981730763602466e+01
 5.402690622395350e+00  8.715004614110915e+00  3.972146634985789e+01  2.530023522028510e+01
 8.719112665180688e+00  3.973860242769609e+01  5.401378818002596e+00  2.861304640443996e+01
 1.117224999094351e+01  1.996445895034417e+01  2.530124906059843e+01  1.117389193699622e+01
 5.506063887368379e-02  5.403779633604668e+00  3.107122842464413e+01  1.998362619065168e+01
 5.402290683161406e+00  3.107284097543882e+01  7.429214717454556e-02  2.529848300738154e+01
 1.994626036922956e+01  2.529931081918375e+01  1.118047396761855e+01  1.994744260682303e+01
 5.404985504065141e+00  3.107422728283204e+01  8.009945833070133e-02  2.529809253594331e+01
 3.107873149054796e+01  8.138996301420534e-02  5.403580781310575e+00  1.117902882087334e+01
 1.117430183085210e+01  1.983127498759274e+01  1.449549065276872e+01  1.117537287197525e+01
 3.972529112469727e+01  3.439184251765676e+01  3.106971199112278e+01  1.982358958190448e+01
 3.438927339756147e+01  3.107083668155762e+01  3.971882266901002e+01  1.449196789548167e+01
 1.982289427475211e+01  1.449328911631026e+01  1.117812731481820e+01  1.980757029975721e+01
 3.439244183073311e+01  3.107768990432324e+01  3.972994805186935e+01  1.449206274981191e+01
 3.107541068049627e+01  3.971322023016205e+01  3.439102177946479e+01  1.118118583417689e+01
 2.860704662505399e+01  1.982863249291954e+01  2.529859432522522e+01  2.860804032355126e+01
 3.972805580124251e+01  5.401315354072053e+00  8.711884134688830e+00  1.981918386718058e+01
 5.404778464414528e+00  8.712894517999343e+00  3.972007939532279e+01  2.530112347079487e+01
 2.860762657046938e+01  1.449500932353360e+01  1.995037807371602e+01  2.860765200227069e+01
 3.439255741510998e+01  6.117377466542123e-02  8.712331150749863e+00  1.449179550555223e+01
 8.066921826109379e-02  8.712316304769622e+00  3.438940463304745e+01  1.998486149867214e+01
 1.117557555703258e+01  2.529879722374251e+01  1.998487359367005e+01  1.117510202059823e+01
 5.401156931422038e+00  6.046255371340203e-02  3.107004125837593e+01  2.530184687063480e+01
 8.138735662856671e-02  3.106957659572577e+01  5.404235763710312e+00  1.995065584102628e+01
 1.117371768190502e+01  1.449233662233708e+01  1.982058890873824e+01  1.117376249240272e+01
 3.438880267116292e+01  3.971707009601584e+01  3.107141545264543e+01  1.449560439057040e+01
 3.972681490687355e+01  3.107142551856906e+01  3.439212175814145e+01  1.982890862870563e+01
 2.860941476294594e+01  2.530132334939649e+01  1.982231444211803e+01  2.860943175299674e+01
 5.404802634144197e+00  3.971530375689093e+01  8.710506146768992e+00  2.529824904529780e+01
 3.973096383937591e+01  8.710588049014270e+00  5.401739802105893e+00  1.983220109754306e+01
 1.448612795718051e+01  1.997760194380987e+01  2.861539319297428e+01  1.448777140403815e+01
 5.468794276042826e-02  8.717947822965622e+00  3.438545169889156e+01  1.998403416131029e+01
 8.716233837084291e+00  3.438704759885889e+01  6.112822590070287e-02  2.861242175624176e+01
 1.994639747038860e+01  2.861334390527322e+01  1.449458222796956e+01  1.994745440452426e+01
 8.718638184610272e+00  3.438855636067932e+01  8.009428117476187e-02  2.861253683590428e+01
 3.439251573952119e+01  8.123890205283871e-02  8.717668900520691e+00  1.449302189603736e+01
 2.529300666519455e+01  1.981668167179750e+01  2.861271002750497e+01  2.529405600684820e+01
 3.971358111022431e+01  8.715474507432207e+00  5.397799631182405e+00  1.983209962291707e+01
 8.718708903917502e+00  5.398836011260602e+00  3.973202722728961e+01  2.861508879625010e+01
 1.994564297311250e+01  1.118206545198795e+01  2.529814577709530e+01  1.994594638320838e+01
 3.107600590862849e+01  5.403966678364231e+00  8.150620387871944e-02  1.118015379270882e+01
 5.402240269580677e+00  8.187321997526248e-02  3.107430332794916e+01  2.530154235992309e+01
 2.529478888771636e+01  1.998429020532284e+01  1.117928023340339e+01  2.529648723728984e+01
 8.340923829511278e-02  3.107539346562041e+01  5.395363897422516e+00  1.995550159858810e+01
 3.107765062999068e+01  5.397043036597902e+00  5.437882981761424e-02  1.118006350374238e+01
 1.980951094406628e+01  1.117907606281405e+01  1.449191065228317e+01  1.980642453543212e+01
 3.107824402343124e+01  3.439171603939867e+01  3.973106480495004e+01  1.117816253786496e+01
 3.438951215306260e+01  3.972568211781174e+01  3.107715274144377e+01  1.449548053904397e+01
 1.448844913065645e+01  1.981837869800219e+01  1.118150206992924e+01  1.448950294886975e+01
 3.971340045275402e+01  3.107781423344316e+01  3.438367885356273e+01  1.983548859363656e+01
 3.107522232019939e+01  3.438475872468907e+01  3.973265346741821e+01  1.117785437229317e+01
 1.979993063384021e+01  2.861579935031427e+01  2.530094509441142e+01  1.983061637004195e+01
 8.716948815700588e+00  5.401098791563760e+00  3.970971124743446e+01  2.861407731715894e+01
 5.404939826039998e+00  3.973759256575831e+01  8.715403587862335e+00  2.529890195423824e+01
radii                                     kind=fltar 456
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00  4.472981758991604e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00  2.112713817723960e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00  2.197751493750416e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
 1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00  1.368161720958986e+00
radii2                                    kind=none   None
rvecs                                     kind=fltar 3,3
 3.979364985026868e+01  7.614451550399652e-06 -6.349718219381480e-06  7.617082337342254e-06
 3.979363720174106e+01  1.466093923475890e-05 -6.348208809181665e-06  1.466769197263813e-05
 3.979365172754711e+01
scope_ids                                 kind=none   None
scopes                                    kind=none   None
valence_charges                           kind=none   None
"""

PARS_TXT = """# BONDHARM
#---------
BONDHARM:UNIT  K kjmol/A**2
BONDHARM:UNIT  R0 A

BONDHARM:PARS      O_OH        ZR  3.7856874859e+02  2.2345796544e+00
BONDHARM:PARS      O_CA        ZR  8.4085734618e+02  2.1879092596e+00
BONDHARM:PARS      O_OX        ZR  7.5654881429e+02  2.1161540051e+00
BONDHARM:PARS      C_CA      O_CA  4.4101463252e+03  1.2792436607e+00
BONDHARM:PARS      C_CA      C_PC  2.1038566297e+03  1.4832348712e+00
BONDHARM:PARS      C_PC      C_PH  3.2247609625e+03  1.4015678878e+00
BONDHARM:PARS      C_PH      C_PH  3.3880065753e+03  1.3929483296e+00
BONDHARM:PARS      C_PH      H_PH  3.3762726364e+03  1.0876572905e+00
BONDHARM:PARS      H_OH      O_OH  5.0216717525e+03  9.9986798534e-01


# BENDAHARM
#----------
BENDAHARM:UNIT  K kjmol/rad**2
BENDAHARM:UNIT  THETA0 deg

BENDAHARM:PARS        ZR      O_OH        ZR  4.1362556428e+02  1.0775418131e+02
BENDAHARM:PARS      C_CA      O_CA        ZR  2.6430192082e+02  1.3689725294e+02
BENDAHARM:PARS      C_PH      C_PH      H_PH  3.0835003755e+02  1.2017556632e+02
BENDAHARM:PARS      C_PC      C_CA      O_CA  1.6684448679e+02  1.1975013868e+02
BENDAHARM:PARS      C_CA      C_PC      C_PH  6.0256074066e+02  1.1990380480e+02
BENDAHARM:PARS      C_PC      C_PH      C_PH  4.7103023496e+02  1.2003765295e+02
BENDAHARM:PARS      O_CA      C_CA      O_CA  6.8388715389e+02  1.2311437675e+02
BENDAHARM:PARS      O_OH        ZR      O_OX  5.3052365716e+02  6.4534430940e+01
BENDAHARM:PARS      C_PH      C_PC      C_PH  4.2496973028e+02  1.1972403242e+02
BENDAHARM:PARS      C_PC      C_PH      H_PH  2.9895866247e+02  1.1956474865e+02
BENDAHARM:PARS        ZR      O_OX        ZR  5.0533812788e+02  1.2046678892e+02
BENDAHARM:PARS      H_OH      O_OH        ZR  1.7902855792e+02  1.1039262816e+02
BENDAHARM:PARS      O_OX        ZR      O_OX  1.0039193248e+02  6.3800382263e+01


# TORSION
#--------
TORSION:UNIT  A kjmol
TORSION:UNIT  PHI0 deg

TORSION:PARS      C_PH      C_PC      C_PH      C_PH  2 3.4910522540e+01  0.0000000000e+00
TORSION:PARS      O_CA      C_CA      O_CA        ZR  2 7.0218267368e+00  0.0000000000e+00
TORSION:PARS      C_CA      C_PC      C_PH      C_PH  2 4.6013908971e+01  0.0000000000e+00
TORSION:PARS      C_PH      C_PC      C_PH      H_PH  2 2.8550678889e+01  0.0000000000e+00
TORSION:PARS      C_CA      C_PC      C_PH      H_PH  2 1.8865768343e+01  0.0000000000e+00
TORSION:PARS      C_PC      C_CA      O_CA        ZR  2 3.2834550916e+01  0.0000000000e+00
TORSION:PARS      C_PC      C_PH      C_PH      H_PH  2 3.3870440544e+01  0.0000000000e+00
TORSION:PARS      H_PH      C_PH      C_PH      H_PH  2 1.7461761843e+01  0.0000000000e+00
TORSION:PARS      C_PH      C_PC      C_CA      O_CA  2 1.3072558996e+01  0.0000000000e+00


# OOPDIST
#--------
OOPDIST:UNIT  K kjmol/A**2
OOPDIST:UNIT  D0 A

OOPDIST:PARS      C_PC      O_CA      O_CA      C_CA  1.3685283732e+03  0.0000000000e+00
OOPDIST:PARS      C_PC      C_PH      H_PH      C_PH  2.0695840516e+02  0.0000000000e+00


#Fixed charges
#---------------

FIXQ:UNIT Q0 e
FIXQ:UNIT P e
FIXQ:UNIT R angstrom
FIXQ:SCALE 1 1.0
FIXQ:SCALE 2 1.0
FIXQ:SCALE 3 1.0
FIXQ:DIELECTRIC 1.0

# Atomic parameters
# ----------------------------------------------------
# KEY        label  Q_0A              R_A
# ----------------------------------------------------
FIXQ:ATOM       ZR  0.0000000000000   2.3670000000
FIXQ:ATOM       CE  0.0000000000000   2.3670000000
FIXQ:ATOM     O_OH  0.0000000000000   1.1180000000
FIXQ:ATOM     O_OX  0.0000000000000   1.1180000000
FIXQ:ATOM     O_CA  0.0000000000000   1.1180000000
FIXQ:ATOM     C_PH  0.0000000000000   1.1630000000
FIXQ:ATOM     C_PC  0.0000000000000   1.1630000000
FIXQ:ATOM     C_CA  0.0000000000000   1.1630000000
FIXQ:ATOM     H_PH  0.0000000000000   0.7240000000
FIXQ:ATOM     H_OH  0.0000000000000   0.7240000000

# Bond parameters
# ----------------------------------------------------
# KEY         label0   label1           P_AB
# ----------------------------------------------------
FIXQ:BOND      C_PH      C_PH   0.0000928607
FIXQ:BOND      C_CA      C_PC   0.0432515406
FIXQ:BOND      O_CA        ZR  -0.3140076067
FIXQ:BOND      O_OX        ZR  -0.3646234389
FIXQ:BOND      C_PH      H_PH  -0.1385876479
FIXQ:BOND      H_OH      O_OH   0.4906092530
FIXQ:BOND      C_CA      O_CA   0.3691317391
FIXQ:BOND      C_PC      C_PH  -0.0556062459
FIXQ:BOND      O_OH        ZR  -0.1881444503

# van der Waals
#==============

# The following mathemetical form is supported:
#  - MM3:   EPSILON*(1.84e5*exp(-12*r/SIGMA)-2.25*(SIGMA/r)^6)
#  - LJ:    4.0*EPSILON*((SIGMA/r)^12 - (SIGMA/r)^6)
#
# Remark:
#  In MM3, if ONLYPAULI=1 then only the pauli term will be used.
#  If ONLYPAULI=0, the full MM3 expression is used with 12.

MM3:UNIT SIGMA angstrom
MM3:UNIT EPSILON kcalmol
MM3:SCALE 1 0.0
MM3:SCALE 2 0.0
MM3:SCALE 3 1.0

# ---------------------------------------------
# KEY      ffatype  SIGMA  EPSILON  ONLYPAULI
# ---------------------------------------------
MM3:PARS      ZR     2.540   0.300      0
MM3:PARS      CE     2.740   0.340      0
MM3:PARS      HF     2.530   0.516      0
MM3:PARS    O_CA     1.820   0.059      0
MM3:PARS    O_OX     1.820   0.059      0
MM3:PARS    O_OH     1.820   0.059      0
MM3:PARS    C_CA     1.940   0.056      0
MM3:PARS    C_PC     1.960   0.056      0
MM3:PARS    C_PH     1.960   0.056      0
MM3:PARS    H_OH     1.600   0.016      0
MM3:PARS    H_PH     1.620   0.020      0
MM3:PARS    C_FO     1.940   0.056      0
MM3:PARS    H_FO     1.620   0.020      0
"""


main()
