"""

Source: https://modis-land.gsfc.nasa.gov/pdf/sn_bound_10deg.txt

"""

from typing import Tuple

from rio_tiler.errors import RioTilerError


class InvalidModlandGridID(RioTilerError):
    """Invalid MODLAND grid id."""


# Only non-fill tiles (460)
# format:
#   horizontal_grid, vertical_grid, bbox(xmin, ymin, xmax, ymax)
MODLAND_GRID = [
    ("14", "00", (-180.0, 80.0, -172.7151, 80.4083)),
    ("15", "00", (-180.0, 80.0, -115.1274, 83.625)),
    ("16", "00", (-180.0, 80.0, -57.5397, 86.8167)),
    ("17", "00", (-180.0, 80.0, 57.2957, 90.0)),
    ("18", "00", (-0.004, 80.0, 180.0, 90.0)),
    ("19", "00", (57.5877, 80.0, 180.0, 86.8167)),
    ("20", "00", (115.1754, 80.0, 180.0, 83.625)),
    ("21", "00", (172.7631, 80.0, 180.0, 80.4083)),
    ("11", "01", (-180.0, 70.0, -175.4039, 70.5333)),
    ("12", "01", (-180.0, 70.0, -146.1659, 73.875)),
    ("13", "01", (-180.0, 70.0, -116.9278, 77.1667)),
    ("14", "01", (-180.0, 70.0, -87.6898, 80.0)),
    ("15", "01", (-172.7631, 70.0, -58.4517, 80.0)),
    ("16", "01", (-115.1754, 70.0, -29.2137, 80.0)),
    ("17", "01", (-57.5877, 70.0, 0.048, 80.0)),
    ("18", "01", (0.0, 70.0, 57.6357, 80.0)),
    ("19", "01", (29.238, 70.0, 115.2234, 80.0)),
    ("20", "01", (58.4761, 70.0, 172.8111, 80.0)),
    ("21", "01", (87.7141, 70.0, 180.0, 80.0)),
    ("22", "01", (116.9522, 70.0, 180.0, 77.1583)),
    ("23", "01", (146.1902, 70.0, 180.0, 73.875)),
    ("24", "01", (175.4283, 70.0, 180.0, 70.5333)),
    ("09", "02", (-180.0, 60.0, -159.9833, 63.6167)),
    ("10", "02", (-180.0, 60.0, -139.9833, 67.1167)),
    ("11", "02", (-180.0, 60.0, -119.9833, 70.0)),
    ("12", "02", (-175.4283, 60.0, -99.9833, 70.0)),
    ("13", "02", (-146.1902, 60.0, -79.9833, 70.0)),
    ("14", "02", (-116.9522, 60.0, -59.9833, 70.0)),
    ("15", "02", (-87.7141, 60.0, -39.9833, 70.0)),
    ("16", "02", (-58.4761, 60.0, -19.9833, 70.0)),
    ("17", "02", (-29.238, 60.0, 0.0244, 70.0)),
    ("18", "02", (0.0, 60.0, 29.2624, 70.0)),
    ("19", "02", (20.0, 60.0, 58.5005, 70.0)),
    ("20", "02", (40.0, 60.0, 87.7385, 70.0)),
    ("21", "02", (60.0, 60.0, 116.9765, 70.0)),
    ("22", "02", (80.0, 60.0, 146.2146, 70.0)),
    ("23", "02", (100.0, 60.0, 175.4526, 70.0)),
    ("24", "02", (120.0, 60.0, 180.0, 70.0)),
    ("25", "02", (140.0, 60.0, 180.0, 67.1167)),
    ("26", "02", (160.0, 60.0, 180.0, 63.6167)),
    ("06", "03", (-180.0, 50.0, -171.1167, 52.3333)),
    ("07", "03", (-180.0, 50.0, -155.5594, 56.2583)),
    ("08", "03", (-180.0, 50.0, -140.0022, 60.0)),
    ("09", "03", (-180.0, 50.0, -124.4449, 60.0)),
    ("10", "03", (-160.0, 50.0, -108.8877, 60.0)),
    ("11", "03", (-140.0, 50.0, -93.3305, 60.0)),
    ("12", "03", (-120.0, 50.0, -77.7732, 60.0)),
    ("13", "03", (-100.0, 50.0, -62.216, 60.0)),
    ("14", "03", (-80.0, 50.0, -46.6588, 60.0)),
    ("15", "03", (-60.0, 50.0, -31.1015, 60.0)),
    ("16", "03", (-40.0, 50.0, -15.5443, 60.0)),
    ("17", "03", (-20.0, 50.0, 0.0167, 60.0)),
    ("18", "03", (0.0, 50.0, 20.0167, 60.0)),
    ("19", "03", (15.5572, 50.0, 40.0167, 60.0)),
    ("20", "03", (31.1145, 50.0, 60.0167, 60.0)),
    ("21", "03", (46.6717, 50.0, 80.0167, 60.0)),
    ("22", "03", (62.229, 50.0, 100.0167, 60.0)),
    ("23", "03", (77.7862, 50.0, 120.0167, 60.0)),
    ("24", "03", (93.3434, 50.0, 140.0167, 60.0)),
    ("25", "03", (108.9007, 50.0, 160.0167, 60.0)),
    ("26", "03", (124.4579, 50.0, 180.0, 60.0)),
    ("27", "03", (140.0151, 50.0, 180.0, 60.0)),
    ("28", "03", (155.5724, 50.0, 180.0, 56.25)),
    ("29", "03", (171.1296, 50.0, 180.0, 52.3333)),
    ("04", "04", (-180.0, 40.0, -169.6921, 43.7667)),
    ("05", "04", (-180.0, 40.0, -156.638, 48.1917)),
    ("06", "04", (-180.0, 40.0, -143.5839, 50.0)),
    ("07", "04", (-171.1296, 40.0, -130.5299, 50.0)),
    ("08", "04", (-155.5724, 40.0, -117.4758, 50.0)),
    ("09", "04", (-140.0151, 40.0, -104.4217, 50.0)),
    ("10", "04", (-124.4579, 40.0, -91.3676, 50.0)),
    ("11", "04", (-108.9007, 40.0, -78.3136, 50.0)),
    ("12", "04", (-93.3434, 40.0, -65.2595, 50.0)),
    ("13", "04", (-77.7862, 40.0, -52.2054, 50.0)),
    ("14", "04", (-62.229, 40.0, -39.1513, 50.0)),
    ("15", "04", (-46.6717, 40.0, -26.0973, 50.0)),
    ("16", "04", (-31.1145, 40.0, -13.0432, 50.0)),
    ("17", "04", (-15.5572, 40.0, 0.013, 50.0)),
    ("18", "04", (0.0, 40.0, 15.5702, 50.0)),
    ("19", "04", (13.0541, 40.0, 31.1274, 50.0)),
    ("20", "04", (26.1081, 40.0, 46.6847, 50.0)),
    ("21", "04", (39.1622, 40.0, 62.2419, 50.0)),
    ("22", "04", (52.2163, 40.0, 77.7992, 50.0)),
    ("23", "04", (65.2704, 40.0, 93.3564, 50.0)),
    ("24", "04", (78.3244, 40.0, 108.9136, 50.0)),
    ("25", "04", (91.3785, 40.0, 124.4709, 50.0)),
    ("26", "04", (104.4326, 40.0, 140.0281, 50.0)),
    ("27", "04", (117.4867, 40.0, 155.5853, 50.0)),
    ("28", "04", (130.5407, 40.0, 171.1426, 50.0)),
    ("29", "04", (143.5948, 40.0, 180.0, 50.0)),
    ("30", "04", (156.6489, 40.0, 180.0, 48.1917)),
    ("31", "04", (169.7029, 40.0, 180.0, 43.7583)),
    ("02", "05", (-180.0, 30.0, -173.1955, 33.5583)),
    ("03", "05", (-180.0, 30.0, -161.6485, 38.95)),
    ("04", "05", (-180.0, 30.0, -150.1014, 40.0)),
    ("05", "05", (-169.7029, 30.0, -138.5544, 40.0)),
    ("06", "05", (-156.6489, 30.0, -127.0074, 40.0)),
    ("07", "05", (-143.5948, 30.0, -115.4604, 40.0)),
    ("08", "05", (-130.5407, 30.0, -103.9134, 40.0)),
    ("09", "05", (-117.4867, 30.0, -92.3664, 40.0)),
    ("10", "05", (-104.4326, 30.0, -80.8194, 40.0)),
    ("11", "05", (-91.3785, 30.0, -69.2724, 40.0)),
    ("12", "05", (-78.3244, 30.0, -57.7254, 40.0)),
    ("13", "05", (-65.2704, 30.0, -46.1784, 40.0)),
    ("14", "05", (-52.2163, 30.0, -34.6314, 40.0)),
    ("15", "05", (-39.1622, 30.0, -23.0844, 40.0)),
    ("16", "05", (-26.1081, 30.0, -11.5374, 40.0)),
    ("17", "05", (-13.0541, 30.0, 0.0109, 40.0)),
    ("18", "05", (0.0, 30.0, 13.065, 40.0)),
    ("19", "05", (11.547, 30.0, 26.119, 40.0)),
    ("20", "05", (23.094, 30.0, 39.1731, 40.0)),
    ("21", "05", (34.641, 30.0, 52.2272, 40.0)),
    ("22", "05", (46.188, 30.0, 65.2812, 40.0)),
    ("23", "05", (57.735, 30.0, 78.3353, 40.0)),
    ("24", "05", (69.282, 30.0, 91.3894, 40.0)),
    ("25", "05", (80.829, 30.0, 104.4435, 40.0)),
    ("26", "05", (92.376, 30.0, 117.4975, 40.0)),
    ("27", "05", (103.923, 30.0, 130.5516, 40.0)),
    ("28", "05", (115.4701, 30.0, 143.6057, 40.0)),
    ("29", "05", (127.0171, 30.0, 156.6598, 40.0)),
    ("30", "05", (138.5641, 30.0, 169.7138, 40.0)),
    ("31", "05", (150.1111, 30.0, 180.0, 40.0)),
    ("32", "05", (161.6581, 30.0, 180.0, 38.9417)),
    ("33", "05", (173.2051, 30.0, 180.0, 33.5583)),
    ("01", "06", (-180.0, 20.0, -170.2596, 27.2667)),
    ("02", "06", (-180.0, 20.0, -159.6178, 30.0)),
    ("03", "06", (-173.2051, 20.0, -148.976, 30.0)),
    ("04", "06", (-161.6581, 20.0, -138.3342, 30.0)),
    ("05", "06", (-150.1111, 20.0, -127.6925, 30.0)),
    ("06", "06", (-138.5641, 20.0, -117.0507, 30.0)),
    ("07", "06", (-127.0171, 20.0, -106.4089, 30.0)),
    ("08", "06", (-115.4701, 20.0, -95.7671, 30.0)),
    ("09", "06", (-103.923, 20.0, -85.1254, 30.0)),
    ("10", "06", (-92.376, 20.0, -74.4836, 30.0)),
    ("11", "06", (-80.829, 20.0, -63.8418, 30.0)),
    ("12", "06", (-69.282, 20.0, -53.2, 30.0)),
    ("13", "06", (-57.735, 20.0, -42.5582, 30.0)),
    ("14", "06", (-46.188, 20.0, -31.9165, 30.0)),
    ("15", "06", (-34.641, 20.0, -21.2747, 30.0)),
    ("16", "06", (-23.094, 20.0, -10.6329, 30.0)),
    ("17", "06", (-11.547, 20.0, 0.0096, 30.0)),
    ("18", "06", (0.0, 20.0, 11.5566, 30.0)),
    ("19", "06", (10.6418, 20.0, 23.1036, 30.0)),
    ("20", "06", (21.2836, 20.0, 34.6506, 30.0)),
    ("21", "06", (31.9253, 20.0, 46.1976, 30.0)),
    ("22", "06", (42.5671, 20.0, 57.7446, 30.0)),
    ("23", "06", (53.2089, 20.0, 69.2917, 30.0)),
    ("24", "06", (63.8507, 20.0, 80.8387, 30.0)),
    ("25", "06", (74.4924, 20.0, 92.3857, 30.0)),
    ("26", "06", (85.1342, 20.0, 103.9327, 30.0)),
    ("27", "06", (95.776, 20.0, 115.4797, 30.0)),
    ("28", "06", (106.4178, 20.0, 127.0267, 30.0)),
    ("29", "06", (117.0596, 20.0, 138.5737, 30.0)),
    ("30", "06", (127.7013, 20.0, 150.1207, 30.0)),
    ("31", "06", (138.3431, 20.0, 161.6677, 30.0)),
    ("32", "06", (148.9849, 20.0, 173.2147, 30.0)),
    ("33", "06", (159.6267, 20.0, 180.0, 30.0)),
    ("34", "06", (170.2684, 20.0, 180.0, 27.2667)),
    ("00", "07", (-180.0, 10.0, -172.6141, 19.1917)),
    ("01", "07", (-180.0, 10.0, -162.4598, 20.0)),
    ("02", "07", (-170.2684, 10.0, -152.3055, 20.0)),
    ("03", "07", (-159.6267, 10.0, -142.1513, 20.0)),
    ("04", "07", (-148.9849, 10.0, -131.997, 20.0)),
    ("05", "07", (-138.3431, 10.0, -121.8427, 20.0)),
    ("06", "07", (-127.7013, 10.0, -111.6885, 20.0)),
    ("07", "07", (-117.0596, 10.0, -101.5342, 20.0)),
    ("08", "07", (-106.4178, 10.0, -91.3799, 20.0)),
    ("09", "07", (-95.776, 10.0, -81.2257, 20.0)),
    ("10", "07", (-85.1342, 10.0, -71.0714, 20.0)),
    ("11", "07", (-74.4924, 10.0, -60.9171, 20.0)),
    ("12", "07", (-63.8507, 10.0, -50.7629, 20.0)),
    ("13", "07", (-53.2089, 10.0, -40.6086, 20.0)),
    ("14", "07", (-42.5671, 10.0, -30.4543, 20.0)),
    ("15", "07", (-31.9253, 10.0, -20.3001, 20.0)),
    ("16", "07", (-21.2836, 10.0, -10.1458, 20.0)),
    ("17", "07", (-10.6418, 10.0, 0.0089, 20.0)),
    ("18", "07", (0.0, 10.0, 10.6506, 20.0)),
    ("19", "07", (10.1543, 10.0, 21.2924, 20.0)),
    ("20", "07", (20.3085, 10.0, 31.9342, 20.0)),
    ("21", "07", (30.4628, 10.0, 42.576, 20.0)),
    ("22", "07", (40.6171, 10.0, 53.2178, 20.0)),
    ("23", "07", (50.7713, 10.0, 63.8595, 20.0)),
    ("24", "07", (60.9256, 10.0, 74.5013, 20.0)),
    ("25", "07", (71.0799, 10.0, 85.1431, 20.0)),
    ("26", "07", (81.2341, 10.0, 95.7849, 20.0)),
    ("27", "07", (91.3884, 10.0, 106.4266, 20.0)),
    ("28", "07", (101.5427, 10.0, 117.0684, 20.0)),
    ("29", "07", (111.6969, 10.0, 127.7102, 20.0)),
    ("30", "07", (121.8512, 10.0, 138.352, 20.0)),
    ("31", "07", (132.0055, 10.0, 148.9938, 20.0)),
    ("32", "07", (142.1597, 10.0, 159.6355, 20.0)),
    ("33", "07", (152.314, 10.0, 170.2773, 20.0)),
    ("34", "07", (162.4683, 10.0, 180.0, 20.0)),
    ("35", "07", (172.6225, 10.0, 180.0, 19.1833)),
    ("00", "08", (-180.0, -0.0, -169.9917, 10.0)),
    ("01", "08", (-172.6225, -0.0, -159.9917, 10.0)),
    ("02", "08", (-162.4683, -0.0, -149.9917, 10.0)),
    ("03", "08", (-152.314, -0.0, -139.9917, 10.0)),
    ("04", "08", (-142.1597, -0.0, -129.9917, 10.0)),
    ("05", "08", (-132.0055, -0.0, -119.9917, 10.0)),
    ("06", "08", (-121.8512, -0.0, -109.9917, 10.0)),
    ("07", "08", (-111.6969, -0.0, -99.9917, 10.0)),
    ("08", "08", (-101.5427, -0.0, -89.9917, 10.0)),
    ("09", "08", (-91.3884, -0.0, -79.9917, 10.0)),
    ("10", "08", (-81.2341, -0.0, -69.9917, 10.0)),
    ("11", "08", (-71.0799, -0.0, -59.9917, 10.0)),
    ("12", "08", (-60.9256, -0.0, -49.9917, 10.0)),
    ("13", "08", (-50.7713, -0.0, -39.9917, 10.0)),
    ("14", "08", (-40.6171, -0.0, -29.9917, 10.0)),
    ("15", "08", (-30.4628, -0.0, -19.9917, 10.0)),
    ("16", "08", (-20.3085, -0.0, -9.9917, 10.0)),
    ("17", "08", (-10.1543, -0.0, 0.0085, 10.0)),
    ("18", "08", (0.0, -0.0, 10.1627, 10.0)),
    ("19", "08", (10.0, -0.0, 20.317, 10.0)),
    ("20", "08", (20.0, -0.0, 30.4713, 10.0)),
    ("21", "08", (30.0, -0.0, 40.6255, 10.0)),
    ("22", "08", (40.0, -0.0, 50.7798, 10.0)),
    ("23", "08", (50.0, -0.0, 60.9341, 10.0)),
    ("24", "08", (60.0, -0.0, 71.0883, 10.0)),
    ("25", "08", (70.0, -0.0, 81.2426, 10.0)),
    ("26", "08", (80.0, -0.0, 91.3969, 10.0)),
    ("27", "08", (90.0, -0.0, 101.5511, 10.0)),
    ("28", "08", (100.0, -0.0, 111.7054, 10.0)),
    ("29", "08", (110.0, -0.0, 121.8597, 10.0)),
    ("30", "08", (120.0, -0.0, 132.0139, 10.0)),
    ("31", "08", (130.0, -0.0, 142.1682, 10.0)),
    ("32", "08", (140.0, -0.0, 152.3225, 10.0)),
    ("33", "08", (150.0, -0.0, 162.4767, 10.0)),
    ("34", "08", (160.0, -0.0, 172.631, 10.0)),
    ("35", "08", (170.0, -0.0, 180.0, 10.0)),
    ("00", "09", (-180.0, -10.0, -169.9917, -0.0)),
    ("01", "09", (-172.6225, -10.0, -159.9917, -0.0)),
    ("02", "09", (-162.4683, -10.0, -149.9917, -0.0)),
    ("03", "09", (-152.314, -10.0, -139.9917, -0.0)),
    ("04", "09", (-142.1597, -10.0, -129.9917, -0.0)),
    ("05", "09", (-132.0055, -10.0, -119.9917, -0.0)),
    ("06", "09", (-121.8512, -10.0, -109.9917, -0.0)),
    ("07", "09", (-111.6969, -10.0, -99.9917, -0.0)),
    ("08", "09", (-101.5427, -10.0, -89.9917, -0.0)),
    ("09", "09", (-91.3884, -10.0, -79.9917, -0.0)),
    ("10", "09", (-81.2341, -10.0, -69.9917, -0.0)),
    ("11", "09", (-71.0799, -10.0, -59.9917, -0.0)),
    ("12", "09", (-60.9256, -10.0, -49.9917, -0.0)),
    ("13", "09", (-50.7713, -10.0, -39.9917, -0.0)),
    ("14", "09", (-40.6171, -10.0, -29.9917, -0.0)),
    ("15", "09", (-30.4628, -10.0, -19.9917, -0.0)),
    ("16", "09", (-20.3085, -10.0, -9.9917, -0.0)),
    ("17", "09", (-10.1543, -10.0, 0.0085, -0.0)),
    ("18", "09", (0.0, -10.0, 10.1627, -0.0)),
    ("19", "09", (10.0, -10.0, 20.317, -0.0)),
    ("20", "09", (20.0, -10.0, 30.4713, -0.0)),
    ("21", "09", (30.0, -10.0, 40.6255, -0.0)),
    ("22", "09", (40.0, -10.0, 50.7798, -0.0)),
    ("23", "09", (50.0, -10.0, 60.9341, -0.0)),
    ("24", "09", (60.0, -10.0, 71.0883, -0.0)),
    ("25", "09", (70.0, -10.0, 81.2426, -0.0)),
    ("26", "09", (80.0, -10.0, 91.3969, -0.0)),
    ("27", "09", (90.0, -10.0, 101.5511, -0.0)),
    ("28", "09", (100.0, -10.0, 111.7054, -0.0)),
    ("29", "09", (110.0, -10.0, 121.8597, -0.0)),
    ("30", "09", (120.0, -10.0, 132.0139, -0.0)),
    ("31", "09", (130.0, -10.0, 142.1682, -0.0)),
    ("32", "09", (140.0, -10.0, 152.3225, -0.0)),
    ("33", "09", (150.0, -10.0, 162.4767, -0.0)),
    ("34", "09", (160.0, -10.0, 172.631, -0.0)),
    ("35", "09", (170.0, -10.0, 180.0, -0.0)),
    ("00", "10", (-180.0, -19.1917, -172.6141, -10.0)),
    ("01", "10", (-180.0, -20.0, -162.4598, -10.0)),
    ("02", "10", (-170.2684, -20.0, -152.3055, -10.0)),
    ("03", "10", (-159.6267, -20.0, -142.1513, -10.0)),
    ("04", "10", (-148.9849, -20.0, -131.997, -10.0)),
    ("05", "10", (-138.3431, -20.0, -121.8427, -10.0)),
    ("06", "10", (-127.7013, -20.0, -111.6885, -10.0)),
    ("07", "10", (-117.0596, -20.0, -101.5342, -10.0)),
    ("08", "10", (-106.4178, -20.0, -91.3799, -10.0)),
    ("09", "10", (-95.776, -20.0, -81.2257, -10.0)),
    ("10", "10", (-85.1342, -20.0, -71.0714, -10.0)),
    ("11", "10", (-74.4924, -20.0, -60.9171, -10.0)),
    ("12", "10", (-63.8507, -20.0, -50.7629, -10.0)),
    ("13", "10", (-53.2089, -20.0, -40.6086, -10.0)),
    ("14", "10", (-42.5671, -20.0, -30.4543, -10.0)),
    ("15", "10", (-31.9253, -20.0, -20.3001, -10.0)),
    ("16", "10", (-21.2836, -20.0, -10.1458, -10.0)),
    ("17", "10", (-10.6418, -20.0, 0.0089, -10.0)),
    ("18", "10", (0.0, -20.0, 10.6506, -10.0)),
    ("19", "10", (10.1543, -20.0, 21.2924, -10.0)),
    ("20", "10", (20.3085, -20.0, 31.9342, -10.0)),
    ("21", "10", (30.4628, -20.0, 42.576, -10.0)),
    ("22", "10", (40.6171, -20.0, 53.2178, -10.0)),
    ("23", "10", (50.7713, -20.0, 63.8595, -10.0)),
    ("24", "10", (60.9256, -20.0, 74.5013, -10.0)),
    ("25", "10", (71.0799, -20.0, 85.1431, -10.0)),
    ("26", "10", (81.2341, -20.0, 95.7849, -10.0)),
    ("27", "10", (91.3884, -20.0, 106.4266, -10.0)),
    ("28", "10", (101.5427, -20.0, 117.0684, -10.0)),
    ("29", "10", (111.6969, -20.0, 127.7102, -10.0)),
    ("30", "10", (121.8512, -20.0, 138.352, -10.0)),
    ("31", "10", (132.0055, -20.0, 148.9938, -10.0)),
    ("32", "10", (142.1597, -20.0, 159.6355, -10.0)),
    ("33", "10", (152.314, -20.0, 170.2773, -10.0)),
    ("34", "10", (162.4683, -20.0, 180.0, -10.0)),
    ("35", "10", (172.6225, -19.1833, 180.0, -10.0)),
    ("01", "11", (-180.0, -27.2667, -170.2596, -20.0)),
    ("02", "11", (-180.0, -30.0, -159.6178, -20.0)),
    ("03", "11", (-173.2051, -30.0, -148.976, -20.0)),
    ("04", "11", (-161.6581, -30.0, -138.3342, -20.0)),
    ("05", "11", (-150.1111, -30.0, -127.6925, -20.0)),
    ("06", "11", (-138.5641, -30.0, -117.0507, -20.0)),
    ("07", "11", (-127.0171, -30.0, -106.4089, -20.0)),
    ("08", "11", (-115.4701, -30.0, -95.7671, -20.0)),
    ("09", "11", (-103.923, -30.0, -85.1254, -20.0)),
    ("10", "11", (-92.376, -30.0, -74.4836, -20.0)),
    ("11", "11", (-80.829, -30.0, -63.8418, -20.0)),
    ("12", "11", (-69.282, -30.0, -53.2, -20.0)),
    ("13", "11", (-57.735, -30.0, -42.5582, -20.0)),
    ("14", "11", (-46.188, -30.0, -31.9165, -20.0)),
    ("15", "11", (-34.641, -30.0, -21.2747, -20.0)),
    ("16", "11", (-23.094, -30.0, -10.6329, -20.0)),
    ("17", "11", (-11.547, -30.0, 0.0096, -20.0)),
    ("18", "11", (0.0, -30.0, 11.5566, -20.0)),
    ("19", "11", (10.6418, -30.0, 23.1036, -20.0)),
    ("20", "11", (21.2836, -30.0, 34.6506, -20.0)),
    ("21", "11", (31.9253, -30.0, 46.1976, -20.0)),
    ("22", "11", (42.5671, -30.0, 57.7446, -20.0)),
    ("23", "11", (53.2089, -30.0, 69.2917, -20.0)),
    ("24", "11", (63.8507, -30.0, 80.8387, -20.0)),
    ("25", "11", (74.4924, -30.0, 92.3857, -20.0)),
    ("26", "11", (85.1342, -30.0, 103.9327, -20.0)),
    ("27", "11", (95.776, -30.0, 115.4797, -20.0)),
    ("28", "11", (106.4178, -30.0, 127.0267, -20.0)),
    ("29", "11", (117.0596, -30.0, 138.5737, -20.0)),
    ("30", "11", (127.7013, -30.0, 150.1207, -20.0)),
    ("31", "11", (138.3431, -30.0, 161.6677, -20.0)),
    ("32", "11", (148.9849, -30.0, 173.2147, -20.0)),
    ("33", "11", (159.6267, -30.0, 180.0, -20.0)),
    ("34", "11", (170.2684, -27.2667, 180.0, -20.0)),
    ("02", "12", (-180.0, -33.5583, -173.1955, -30.0)),
    ("03", "12", (-180.0, -38.95, -161.6485, -30.0)),
    ("04", "12", (-180.0, -40.0, -150.1014, -30.0)),
    ("05", "12", (-169.7029, -40.0, -138.5544, -30.0)),
    ("06", "12", (-156.6489, -40.0, -127.0074, -30.0)),
    ("07", "12", (-143.5948, -40.0, -115.4604, -30.0)),
    ("08", "12", (-130.5407, -40.0, -103.9134, -30.0)),
    ("09", "12", (-117.4867, -40.0, -92.3664, -30.0)),
    ("10", "12", (-104.4326, -40.0, -80.8194, -30.0)),
    ("11", "12", (-91.3785, -40.0, -69.2724, -30.0)),
    ("12", "12", (-78.3244, -40.0, -57.7254, -30.0)),
    ("13", "12", (-65.2704, -40.0, -46.1784, -30.0)),
    ("14", "12", (-52.2163, -40.0, -34.6314, -30.0)),
    ("15", "12", (-39.1622, -40.0, -23.0844, -30.0)),
    ("16", "12", (-26.1081, -40.0, -11.5374, -30.0)),
    ("17", "12", (-13.0541, -40.0, 0.0109, -30.0)),
    ("18", "12", (0.0, -40.0, 13.065, -30.0)),
    ("19", "12", (11.547, -40.0, 26.119, -30.0)),
    ("20", "12", (23.094, -40.0, 39.1731, -30.0)),
    ("21", "12", (34.641, -40.0, 52.2272, -30.0)),
    ("22", "12", (46.188, -40.0, 65.2812, -30.0)),
    ("23", "12", (57.735, -40.0, 78.3353, -30.0)),
    ("24", "12", (69.282, -40.0, 91.3894, -30.0)),
    ("25", "12", (80.829, -40.0, 104.4435, -30.0)),
    ("26", "12", (92.376, -40.0, 117.4975, -30.0)),
    ("27", "12", (103.923, -40.0, 130.5516, -30.0)),
    ("28", "12", (115.4701, -40.0, 143.6057, -30.0)),
    ("29", "12", (127.0171, -40.0, 156.6598, -30.0)),
    ("30", "12", (138.5641, -40.0, 169.7138, -30.0)),
    ("31", "12", (150.1111, -40.0, 180.0, -30.0)),
    ("32", "12", (161.6581, -38.9417, 180.0, -30.0)),
    ("33", "12", (173.2051, -33.5583, 180.0, -30.0)),
    ("04", "13", (-180.0, -43.7667, -169.6921, -40.0)),
    ("05", "13", (-180.0, -48.1917, -156.638, -40.0)),
    ("06", "13", (-180.0, -50.0, -143.5839, -40.0)),
    ("07", "13", (-171.1296, -50.0, -130.5299, -40.0)),
    ("08", "13", (-155.5724, -50.0, -117.4758, -40.0)),
    ("09", "13", (-140.0151, -50.0, -104.4217, -40.0)),
    ("10", "13", (-124.4579, -50.0, -91.3676, -40.0)),
    ("11", "13", (-108.9007, -50.0, -78.3136, -40.0)),
    ("12", "13", (-93.3434, -50.0, -65.2595, -40.0)),
    ("13", "13", (-77.7862, -50.0, -52.2054, -40.0)),
    ("14", "13", (-62.229, -50.0, -39.1513, -40.0)),
    ("15", "13", (-46.6717, -50.0, -26.0973, -40.0)),
    ("16", "13", (-31.1145, -50.0, -13.0432, -40.0)),
    ("17", "13", (-15.5572, -50.0, 0.013, -40.0)),
    ("18", "13", (0.0, -50.0, 15.5702, -40.0)),
    ("19", "13", (13.0541, -50.0, 31.1274, -40.0)),
    ("20", "13", (26.1081, -50.0, 46.6847, -40.0)),
    ("21", "13", (39.1622, -50.0, 62.2419, -40.0)),
    ("22", "13", (52.2163, -50.0, 77.7992, -40.0)),
    ("23", "13", (65.2704, -50.0, 93.3564, -40.0)),
    ("24", "13", (78.3244, -50.0, 108.9136, -40.0)),
    ("25", "13", (91.3785, -50.0, 124.4709, -40.0)),
    ("26", "13", (104.4326, -50.0, 140.0281, -40.0)),
    ("27", "13", (117.4867, -50.0, 155.5853, -40.0)),
    ("28", "13", (130.5407, -50.0, 171.1426, -40.0)),
    ("29", "13", (143.5948, -50.0, 180.0, -40.0)),
    ("30", "13", (156.6489, -48.1917, 180.0, -40.0)),
    ("31", "13", (169.7029, -43.7583, 180.0, -40.0)),
    ("06", "14", (-180.0, -52.3333, -171.1167, -50.0)),
    ("07", "14", (-180.0, -56.2583, -155.5594, -50.0)),
    ("08", "14", (-180.0, -60.0, -140.0022, -50.0)),
    ("09", "14", (-180.0, -60.0, -124.4449, -50.0)),
    ("10", "14", (-160.0, -60.0, -108.8877, -50.0)),
    ("11", "14", (-140.0, -60.0, -93.3305, -50.0)),
    ("12", "14", (-120.0, -60.0, -77.7732, -50.0)),
    ("13", "14", (-100.0, -60.0, -62.216, -50.0)),
    ("14", "14", (-80.0, -60.0, -46.6588, -50.0)),
    ("15", "14", (-60.0, -60.0, -31.1015, -50.0)),
    ("16", "14", (-40.0, -60.0, -15.5443, -50.0)),
    ("17", "14", (-20.0, -60.0, 0.0167, -50.0)),
    ("18", "14", (0.0, -60.0, 20.0167, -50.0)),
    ("19", "14", (15.5572, -60.0, 40.0167, -50.0)),
    ("20", "14", (31.1145, -60.0, 60.0167, -50.0)),
    ("21", "14", (46.6717, -60.0, 80.0167, -50.0)),
    ("22", "14", (62.229, -60.0, 100.0167, -50.0)),
    ("23", "14", (77.7862, -60.0, 120.0167, -50.0)),
    ("24", "14", (93.3434, -60.0, 140.0167, -50.0)),
    ("25", "14", (108.9007, -60.0, 160.0167, -50.0)),
    ("26", "14", (124.4579, -60.0, 180.0, -50.0)),
    ("27", "14", (140.0151, -60.0, 180.0, -50.0)),
    ("28", "14", (155.5724, -56.25, 180.0, -50.0)),
    ("29", "14", (171.1296, -52.3333, 180.0, -50.0)),
    ("09", "15", (-180.0, -63.6167, -159.9833, -60.0)),
    ("10", "15", (-180.0, -67.1167, -139.9833, -60.0)),
    ("11", "15", (-180.0, -70.0, -119.9833, -60.0)),
    ("12", "15", (-175.4283, -70.0, -99.9833, -60.0)),
    ("13", "15", (-146.1902, -70.0, -79.9833, -60.0)),
    ("14", "15", (-116.9522, -70.0, -59.9833, -60.0)),
    ("15", "15", (-87.7141, -70.0, -39.9833, -60.0)),
    ("16", "15", (-58.4761, -70.0, -19.9833, -60.0)),
    ("17", "15", (-29.238, -70.0, 0.0244, -60.0)),
    ("18", "15", (0.0, -70.0, 29.2624, -60.0)),
    ("19", "15", (20.0, -70.0, 58.5005, -60.0)),
    ("20", "15", (40.0, -70.0, 87.7385, -60.0)),
    ("21", "15", (60.0, -70.0, 116.9765, -60.0)),
    ("22", "15", (80.0, -70.0, 146.2146, -60.0)),
    ("23", "15", (100.0, -70.0, 175.4526, -60.0)),
    ("24", "15", (120.0, -70.0, 180.0, -60.0)),
    ("25", "15", (140.0, -67.1167, 180.0, -60.0)),
    ("26", "15", (160.0, -63.6167, 180.0, -60.0)),
    ("11", "16", (-180.0, -70.5333, -175.4039, -70.0)),
    ("12", "16", (-180.0, -73.875, -146.1659, -70.0)),
    ("13", "16", (-180.0, -77.1667, -116.9278, -70.0)),
    ("14", "16", (-180.0, -80.0, -87.6898, -70.0)),
    ("15", "16", (-172.7631, -80.0, -58.4517, -70.0)),
    ("16", "16", (-115.1754, -80.0, -29.2137, -70.0)),
    ("17", "16", (-57.5877, -80.0, 0.048, -70.0)),
    ("18", "16", (0.0, -80.0, 57.6357, -70.0)),
    ("19", "16", (29.238, -80.0, 115.2234, -70.0)),
    ("20", "16", (58.4761, -80.0, 172.8111, -70.0)),
    ("21", "16", (87.7141, -80.0, 180.0, -70.0)),
    ("22", "16", (116.9522, -77.1583, 180.0, -70.0)),
    ("23", "16", (146.1902, -73.875, 180.0, -70.0)),
    ("24", "16", (175.4283, -70.5333, 180.0, -70.0)),
    ("14", "17", (-180.0, -80.4083, -172.7151, -80.0)),
    ("15", "17", (-180.0, -83.625, -115.1274, -80.0)),
    ("16", "17", (-180.0, -86.8167, -57.5397, -80.0)),
    ("17", "17", (-180.0, -90.0, 57.2957, -80.0)),
    ("18", "17", (-0.004, -90.0, 180.0, -80.0)),
    ("19", "17", (57.5877, -86.8167, 180.0, -80.0)),
    ("20", "17", (115.1754, -83.625, 180.0, -80.0)),
    ("21", "17", (172.7631, -80.4083, 180.0, -80.0)),
]


def tile_bbox(
    horizontal_grid: str, vertical_grid: str
) -> Tuple[float, float, float, float]:
    """Get WGS84 boundbox for any modland grid index."""
    grid = list(
        filter(
            lambda x: x[0] == horizontal_grid and x[1] == vertical_grid, MODLAND_GRID
        )
    )
    if not grid:
        raise InvalidModlandGridID(
            f"Could not find bounds for grid h{horizontal_grid}v{vertical_grid}"
        )
    return grid[0][2]
