"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isCompleteHandler = exports.onDelete = exports.startBuild = exports.onEventhandler = void 0;
const AWS = require("aws-sdk");
const cb = new AWS.CodeBuild();
async function onEventhandler(event) {
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            return startBuild(event);
        case 'Delete':
            return onDelete();
    }
}
exports.onEventhandler = onEventhandler;
async function startBuild(event) {
    const projectName = event.ResourceProperties.projectName;
    const buildOutput = await cb.startBuild({
        projectName: projectName,
    }).promise();
    const buildId = buildOutput.build?.id;
    if (buildId) {
        return {
            BuildId: buildId,
        };
    }
    else {
        throw new Error('BuildId does not exist after CodeBuild:StartBuild call');
    }
}
exports.startBuild = startBuild;
// pass isComplete a value to indicate it does not need to wait on delete events
async function onDelete() {
    return {
        BuildId: 'onDelete',
    };
}
exports.onDelete = onDelete;
async function isCompleteHandler(event) {
    const buildId = event.BuildId; // BuildId is passed in from onEvent CodeBuild:StartBuild call
    if (!buildId) {
        throw new Error('BuildId was not found or undefined');
    }
    // isComplete does not need to wait on delete events
    if (buildId === 'onDelete') {
        return { IsComplete: true };
    }
    const build = await cb.batchGetBuilds({
        ids: [buildId],
    }).promise();
    // we should always have a build since we have a valid buildId
    if (!build.builds || build.builds.length <= 0) {
        throw new Error(`Build does not exist for BuildId: ${buildId}`);
    }
    const buildResponse = build.builds[0];
    const currentPhase = buildResponse.currentPhase;
    const buildStatus = buildResponse.buildStatus;
    if (currentPhase === 'COMPLETED' && buildStatus === 'SUCCEEDED') {
        return {
            IsComplete: true,
            Data: {
                Status: 'CodeBuild completed successfully',
                LogsUrl: `${JSON.stringify(buildResponse.logs?.deepLink)}`,
            },
        };
    }
    else if (currentPhase === 'COMPLETED' && buildStatus === 'FAILED') {
        if (buildResponse.logs?.deepLink) {
            throw new Error(`CodeBuild failed, check the logs here: ${buildResponse.logs.deepLink}`);
        }
        else {
            throw new Error('CodeBuild failed'); // this case should never be reached
        }
    }
    else {
        return { IsComplete: false }; // not finished
    }
}
exports.isCompleteHandler = isCompleteHandler;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY29kZWJ1aWxkLWhhbmRsZXIvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsK0JBQStCO0FBRS9CLE1BQU0sRUFBRSxHQUFHLElBQUksR0FBRyxDQUFDLFNBQVMsRUFBRSxDQUFDO0FBRXhCLEtBQUssVUFBVSxjQUFjLENBQUMsS0FBa0Q7SUFDckYsUUFBUSxLQUFLLENBQUMsV0FBVyxFQUFFO1FBQ3pCLEtBQUssUUFBUSxDQUFDO1FBQ2QsS0FBSyxRQUFRO1lBQ1gsT0FBTyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0IsS0FBSyxRQUFRO1lBQ1gsT0FBTyxRQUFRLEVBQUUsQ0FBQztLQUNyQjtBQUNILENBQUM7QUFSRCx3Q0FRQztBQUVNLEtBQUssVUFBVSxVQUFVLENBQUMsS0FBa0Q7SUFDakYsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQztJQUN6RCxNQUFNLFdBQVcsR0FBRyxNQUFNLEVBQUUsQ0FBQyxVQUFVLENBQUM7UUFDdEMsV0FBVyxFQUFFLFdBQVc7S0FDekIsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBRWIsTUFBTSxPQUFPLEdBQUcsV0FBVyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUM7SUFFdEMsSUFBSSxPQUFPLEVBQUU7UUFDWCxPQUFPO1lBQ0wsT0FBTyxFQUFFLE9BQU87U0FDakIsQ0FBQztLQUNIO1NBQU07UUFDTCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7S0FDM0U7QUFDSCxDQUFDO0FBZkQsZ0NBZUM7QUFFRCxnRkFBZ0Y7QUFDekUsS0FBSyxVQUFVLFFBQVE7SUFDNUIsT0FBTztRQUNMLE9BQU8sRUFBRSxVQUFVO0tBQ3BCLENBQUM7QUFDSixDQUFDO0FBSkQsNEJBSUM7QUFFTSxLQUFLLFVBQVUsaUJBQWlCLENBQUMsS0FBa0Q7SUFFeEYsTUFBTSxPQUFPLEdBQUksS0FBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLDhEQUE4RDtJQUV0RyxJQUFJLENBQUMsT0FBTyxFQUFFO1FBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0tBQ3ZEO0lBRUQsb0RBQW9EO0lBQ3BELElBQUksT0FBTyxLQUFLLFVBQVUsRUFBRTtRQUMxQixPQUFPLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDO0tBQzdCO0lBRUQsTUFBTSxLQUFLLEdBQUcsTUFBTSxFQUFFLENBQUMsY0FBYyxDQUFDO1FBQ3BDLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQztLQUNmLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUViLDhEQUE4RDtJQUM5RCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLE1BQU0sSUFBSSxDQUFDLEVBQUU7UUFDN0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsT0FBTyxFQUFFLENBQUMsQ0FBQztLQUNqRTtJQUVELE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdEMsTUFBTSxZQUFZLEdBQUcsYUFBYSxDQUFDLFlBQVksQ0FBQztJQUNoRCxNQUFNLFdBQVcsR0FBRyxhQUFhLENBQUMsV0FBVyxDQUFDO0lBRTlDLElBQUksWUFBWSxLQUFLLFdBQVcsSUFBSSxXQUFXLEtBQUssV0FBVyxFQUFFO1FBQy9ELE9BQU87WUFDTCxVQUFVLEVBQUUsSUFBSTtZQUNoQixJQUFJLEVBQUU7Z0JBQ0osTUFBTSxFQUFFLGtDQUFrQztnQkFDMUMsT0FBTyxFQUFFLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxFQUFFO2FBQzNEO1NBQ0YsQ0FBQztLQUNIO1NBQU0sSUFBSSxZQUFZLEtBQUssV0FBVyxJQUFJLFdBQVcsS0FBSyxRQUFRLEVBQUU7UUFDbkUsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUNoQyxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxhQUFhLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDMUY7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLG9DQUFvQztTQUMxRTtLQUNGO1NBQU07UUFDTCxPQUFPLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsZUFBZTtLQUM5QztBQUNILENBQUM7QUEzQ0QsOENBMkNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgQVdTTGFtYmRhIGZyb20gJ2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgQVdTIGZyb20gJ2F3cy1zZGsnO1xuXG5jb25zdCBjYiA9IG5ldyBBV1MuQ29kZUJ1aWxkKCk7XG5cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBvbkV2ZW50aGFuZGxlcihldmVudDogQVdTTGFtYmRhLkNsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VFdmVudCkge1xuICBzd2l0Y2ggKGV2ZW50LlJlcXVlc3RUeXBlKSB7XG4gICAgY2FzZSAnQ3JlYXRlJzpcbiAgICBjYXNlICdVcGRhdGUnOlxuICAgICAgcmV0dXJuIHN0YXJ0QnVpbGQoZXZlbnQpO1xuICAgIGNhc2UgJ0RlbGV0ZSc6XG4gICAgICByZXR1cm4gb25EZWxldGUoKTtcbiAgfVxufVxuXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gc3RhcnRCdWlsZChldmVudDogQVdTTGFtYmRhLkNsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VFdmVudCkge1xuICBjb25zdCBwcm9qZWN0TmFtZSA9IGV2ZW50LlJlc291cmNlUHJvcGVydGllcy5wcm9qZWN0TmFtZTtcbiAgY29uc3QgYnVpbGRPdXRwdXQgPSBhd2FpdCBjYi5zdGFydEJ1aWxkKHtcbiAgICBwcm9qZWN0TmFtZTogcHJvamVjdE5hbWUsXG4gIH0pLnByb21pc2UoKTtcblxuICBjb25zdCBidWlsZElkID0gYnVpbGRPdXRwdXQuYnVpbGQ/LmlkO1xuXG4gIGlmIChidWlsZElkKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIEJ1aWxkSWQ6IGJ1aWxkSWQsIC8vIHBhc3MgdG8gaXNDb21wbGV0ZVxuICAgIH07XG4gIH0gZWxzZSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdCdWlsZElkIGRvZXMgbm90IGV4aXN0IGFmdGVyIENvZGVCdWlsZDpTdGFydEJ1aWxkIGNhbGwnKTtcbiAgfVxufVxuXG4vLyBwYXNzIGlzQ29tcGxldGUgYSB2YWx1ZSB0byBpbmRpY2F0ZSBpdCBkb2VzIG5vdCBuZWVkIHRvIHdhaXQgb24gZGVsZXRlIGV2ZW50c1xuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIG9uRGVsZXRlKCkge1xuICByZXR1cm4ge1xuICAgIEJ1aWxkSWQ6ICdvbkRlbGV0ZScsXG4gIH07XG59XG5cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBpc0NvbXBsZXRlSGFuZGxlcihldmVudDogQVdTTGFtYmRhLkNsb3VkRm9ybWF0aW9uQ3VzdG9tUmVzb3VyY2VFdmVudCkge1xuXG4gIGNvbnN0IGJ1aWxkSWQgPSAoZXZlbnQgYXMgYW55KS5CdWlsZElkOyAvLyBCdWlsZElkIGlzIHBhc3NlZCBpbiBmcm9tIG9uRXZlbnQgQ29kZUJ1aWxkOlN0YXJ0QnVpbGQgY2FsbFxuXG4gIGlmICghYnVpbGRJZCkge1xuICAgIHRocm93IG5ldyBFcnJvcignQnVpbGRJZCB3YXMgbm90IGZvdW5kIG9yIHVuZGVmaW5lZCcpO1xuICB9XG5cbiAgLy8gaXNDb21wbGV0ZSBkb2VzIG5vdCBuZWVkIHRvIHdhaXQgb24gZGVsZXRlIGV2ZW50c1xuICBpZiAoYnVpbGRJZCA9PT0gJ29uRGVsZXRlJykge1xuICAgIHJldHVybiB7IElzQ29tcGxldGU6IHRydWUgfTtcbiAgfVxuXG4gIGNvbnN0IGJ1aWxkID0gYXdhaXQgY2IuYmF0Y2hHZXRCdWlsZHMoe1xuICAgIGlkczogW2J1aWxkSWRdLFxuICB9KS5wcm9taXNlKCk7XG5cbiAgLy8gd2Ugc2hvdWxkIGFsd2F5cyBoYXZlIGEgYnVpbGQgc2luY2Ugd2UgaGF2ZSBhIHZhbGlkIGJ1aWxkSWRcbiAgaWYgKCFidWlsZC5idWlsZHMgfHwgYnVpbGQuYnVpbGRzLmxlbmd0aCA8PSAwKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBCdWlsZCBkb2VzIG5vdCBleGlzdCBmb3IgQnVpbGRJZDogJHtidWlsZElkfWApO1xuICB9XG5cbiAgY29uc3QgYnVpbGRSZXNwb25zZSA9IGJ1aWxkLmJ1aWxkc1swXTtcbiAgY29uc3QgY3VycmVudFBoYXNlID0gYnVpbGRSZXNwb25zZS5jdXJyZW50UGhhc2U7XG4gIGNvbnN0IGJ1aWxkU3RhdHVzID0gYnVpbGRSZXNwb25zZS5idWlsZFN0YXR1cztcblxuICBpZiAoY3VycmVudFBoYXNlID09PSAnQ09NUExFVEVEJyAmJiBidWlsZFN0YXR1cyA9PT0gJ1NVQ0NFRURFRCcpIHtcbiAgICByZXR1cm4ge1xuICAgICAgSXNDb21wbGV0ZTogdHJ1ZSxcbiAgICAgIERhdGE6IHtcbiAgICAgICAgU3RhdHVzOiAnQ29kZUJ1aWxkIGNvbXBsZXRlZCBzdWNjZXNzZnVsbHknLFxuICAgICAgICBMb2dzVXJsOiBgJHtKU09OLnN0cmluZ2lmeShidWlsZFJlc3BvbnNlLmxvZ3M/LmRlZXBMaW5rKX1gLFxuICAgICAgfSxcbiAgICB9O1xuICB9IGVsc2UgaWYgKGN1cnJlbnRQaGFzZSA9PT0gJ0NPTVBMRVRFRCcgJiYgYnVpbGRTdGF0dXMgPT09ICdGQUlMRUQnKSB7XG4gICAgaWYgKGJ1aWxkUmVzcG9uc2UubG9ncz8uZGVlcExpbmspIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ29kZUJ1aWxkIGZhaWxlZCwgY2hlY2sgdGhlIGxvZ3MgaGVyZTogJHtidWlsZFJlc3BvbnNlLmxvZ3MuZGVlcExpbmt9YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29kZUJ1aWxkIGZhaWxlZCcpOyAvLyB0aGlzIGNhc2Ugc2hvdWxkIG5ldmVyIGJlIHJlYWNoZWRcbiAgICB9XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogZmFsc2UgfTsgLy8gbm90IGZpbmlzaGVkXG4gIH1cbn1cbiJdfQ==