import abc
import dataclasses
import typing
from collections import OrderedDict

from dataclasses_avroschema import utils
from dataclasses_avroschema.fields import AvroField, FieldType

try:
    import faust
except ImportError:  # pragma: no cover
    faust = None  # type: ignore # pragma: no cover


@dataclasses.dataclass  # type: ignore
class BaseSchemaDefinition(abc.ABC):
    """
    Minimal Schema definition
    """

    type: str
    klass: typing.Any
    parent: typing.Any

    @abc.abstractmethod
    def get_rendered_fields(self) -> typing.List[OrderedDict]:
        ...  # pragma: no cover

    @abc.abstractmethod
    def render(self) -> OrderedDict:
        ...  # pragma: no cover

    def get_schema_name(self) -> str:
        return self.klass.metadata.schema_name or self.klass.__name__

    def generate_documentation(self) -> typing.Optional[str]:
        doc = self.klass.__doc__
        if doc is not None:
            return doc.replace("\n", "")

    @property
    def is_faust_record(self) -> bool:
        if faust:
            return issubclass(self.klass, faust.Record)
        return False


@dataclasses.dataclass
class AvroSchemaDefinition(BaseSchemaDefinition):
    metadata: utils.SchemaMetadata
    fields: typing.List[FieldType] = dataclasses.field(default_factory=list)

    def __post_init__(self) -> None:
        self.fields = self.parse_dataclasses_fields()

    def parse_dataclasses_fields(self) -> typing.List[FieldType]:
        if self.is_faust_record:
            return self.parse_faust_record_fields()
        return self.parse_fields()

    def parse_fields(self) -> typing.List[FieldType]:
        return [
            AvroField(
                dataclass_field.name,
                dataclass_field.type,
                default=dataclass_field.default,
                default_factory=dataclass_field.default_factory,  # type: ignore  # TODO: resolve mypy
                metadata=dataclass_field.metadata,
                model_metadata=self.metadata,
                parent=self.parent,
            )
            for dataclass_field in dataclasses.fields(self.klass)
        ]

    def parse_faust_record_fields(self) -> typing.List[FieldType]:
        schema_fields = []

        for dataclass_field in dataclasses.fields(self.klass):
            faust_field = dataclass_field.default
            default_factory = dataclasses.MISSING

            if faust_field.required:
                default = dataclasses.MISSING
            else:
                default = faust_field.default

                if isinstance(default, dataclasses.Field):
                    default_factory = default.default_factory  # type: ignore  # TODO: resolve mypy
                    default = dataclasses.MISSING

            schema_fields.append(
                AvroField(
                    dataclass_field.name,
                    dataclass_field.type,
                    default=default,
                    default_factory=default_factory,
                    model_metadata=self.metadata,
                    parent=self.parent,
                )
            )

        return schema_fields

    def get_rendered_fields(self) -> typing.List[OrderedDict]:
        return [field.render() for field in self.fields]

    def render(self) -> OrderedDict:
        schema = OrderedDict(
            [
                ("type", self.type),
                ("name", self.get_schema_name()),
                ("fields", self.get_rendered_fields()),
            ]
        )

        if self.metadata.schema_doc:
            doc = self.generate_documentation()
            if doc is not None:
                schema["doc"] = doc

        if self.metadata.namespace is not None:
            schema["namespace"] = self.metadata.namespace

        if self.metadata.aliases is not None:
            schema["aliases"] = self.metadata.aliases

        return schema
