# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hrepr']

package_data = \
{'': ['*'], 'hrepr': ['style/*']}

install_requires = \
['ovld>=0.3.2,<0.4.0']

setup_kwargs = {
    'name': 'hrepr',
    'version': '0.5.1',
    'description': 'Extensible HTML representation for Python objects.',
    'long_description': '# hrepr\n\n`hrepr` outputs HTML/pretty representations for Python objects.\n\n✅ Nice, colourful representations of lists, dicts, dataclasses, booleans...<br/>\n✅ Ridiculously extensible and configurable<br/>\n✅ Handles recursive data structures<br/>\n✅ Compatible with Jupyter notebooks<br/>\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr1.png" width="400px"><img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr2.png" width="400px">\n\nI suggest studying the example file to learn `hrepr`:\n\n* `python examples/exhibit.py > exhibit.html` (and then view the HTML file)\n\nAlso see the Jupyter notebook at `examples/Basics.ipynb`, but keep in mind that GitHub can\'t display it properly because of the injected styles/scripts.\n\n\n## Install\n\n```python\npip install hrepr\n```\n\n\n## Usage\n\n```python\nfrom hrepr import hrepr\nobj = {\'potatoes\': [1, 2, 3], \'bananas\': {\'cantaloups\': 8}}\n\n# Print the HTML representation of obj\nprint(hrepr(obj))\n\n# Wrap the representation in <html><body> tags and embed the default\n# css style files in a standalone page, which is saved to obj.html\nhrepr.page(obj, file="obj.html")\n```\n\nIn a Jupyter Notebook, you can return `hrepr(obj)` from any cell and it will show its representation for you. You can also write `display_html(hrepr(obj))`.\n\n\n## Custom representations\n\nA custom representation for an object can be defined using the following three methods (it is not necessary to define all of them, only those that are relevant to your case):\n\n* `__hrepr__(self, H, hrepr)` returns the normal HTML representation.\n    * Use `H.span["some-class"](some-content, some_attr=some_value)` to generate HTML.\n    * Use `hrepr(self.x)` to generate the representation for some subfield `x`.\n    * `hrepr.config` contains any keyword arguments given in the top level call to `hrepr`. For instance, if you call `hrepr(obj, blah=3)`, then `hrepr.config.blah == 3` in all calls to `__hrepr__` down the line (the default value for all keys is `None`).\n* `__hrepr_short__(self, H, hrepr)` returns a *short* representation, ideally of a constant size.\n    * The output of this method is used when we hit max depth, or for repeated references.\n    * Only include bare minimum information. Short means short.\n* `__hrepr_resources__(cls, H)` is a **classmethod** that returns resources common to all instances of the class (typically a stylesheet or a script).\n    * When generating a page, the resources will go in `<head>`.\n    * You can return a list of resources.\n\nNo dependency on `hrepr` is necessary.\n\nFor example:\n\n```python\nclass Person:\n    def __init__(self, name, age, job):\n        self.name = name\n        self.age = age\n        self.job = job\n\n    @classmethod\n    def __hrepr_resources__(cls, H):\n        # Note: you might need to add "!important" next to some rules if\n        # they conflict with defaults from hrepr\'s own CSS.\n        return H.style("""\n            .person {\n                background: magenta !important;\n                border-color: magenta !important;\n            }\n            .person-short { font-weight: bold; color: green; }\n        """)\n\n    def __hrepr__(self, H, hrepr):\n        # hrepr.make.instance is a helper to show a table with a header that\n        # describes some kind of object\n        return hrepr.make.instance(\n            title=self.name,\n            fields=[["age", self.age], ["job", self.job]],\n            delimiter=" ↦ ",\n            type="person",\n        )\n\n    def __hrepr_short__(self, H, hrepr):\n        return H.span["person-short"](self.name)\n```\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr3.png" width="600px">\n\n\n## References\n\n`hrepr` can handle circular references. Furthermore, if an object is found at several places in a structure, only the first occurrence will be printed in full, and any other will be a numeric reference mapped to the short representation for the object. It looks like this:\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr4.png" width="600px">\n\nThe `shortrefs` and `norefs` configuration keys control the representation of references:\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr5.png" width="600px">\n\n`norefs` is ignored when there are circular references.\n\n\n## HTML generation\n\nGenerate HTML using the `H` parameter to `__hrepr__`, or import it and use it directly:\n\n```python\nfrom hrepr import H\nhtml = H.span["bear"](\n    "Only ", H.b("YOU"), " can prevent forest fires!",\n    style="color: brown;"\n)\nprint(html)\n# <span class="bear" style="color: brown;">Only <b>YOU</b> can prevent forest fires!</span>\n```\n\n`H` can be built incrementally: if you have an element, you can call it to add children, index it to add classes, and so on. For instance:\n\n```python\nfrom hrepr import H\nhtml = H.span()\nhtml = html("Only ")\nhtml = html(style="color: brown;")["bear"]\nhtml = html(H.b("YOU"), " can prevent forest fires!")\nprint(html)\n# <span class="bear" style="color: brown;">Only <b>YOU</b> can prevent forest fires!</span>\n```\n\nThis can be handy if you want to tweak generated HTML a little. For example, `hrepr(obj)["fox"]` will tack on the class `fox` to the representation of the object.\n\n\n### Helpers\n\n* `hrepr.make.instance(title, fields, delimiter=None, type=None)`: formats the fields like a dataclass, with title on top.\n* `hrepr.make.bracketed(body, start, end, type=None)`: formats the body with the specified start/end bracket.\n\n\n### Constructed elements\n\nTo make it a bit easier to include and use JavaScript libraries, you can use the special `__constructor` attribute.\n\nFor example, you can load Plotly and create a plot like this:\n\n```python\ndef Plot(data):\n    return H.div(\n        __constructor={\n            "script": "https://cdn.plot.ly/plotly-latest.min.js",\n            "symbol": "Plotly.newPlot",\n            "options": [{"x": list(range(len(data))), "y": list(data)}],\n        }\n    )\nprint(Plot([math.sin(x / 10) for x in range(100)]))\n```\n\nThe above will:\n\n* Load the specified script.\n* Get the `Plotly.newPlot` function in the global namespace.\n* Call it with the `div` element as the first argument, and the `options` as the second argument.\n\nIt will look like this:\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr6.png" width="600px">\n\n\n### Modules\n\nAnother example, this time using ESM (modules):\n\n```python\nnode = H.div(\n    style="width:500px;height:500px;border:1px solid black;",\n    __constructor={\n        "module": "https://cdnjs.cloudflare.com/ajax/libs/cytoscape/3.23.0/cytoscape.esm.min.js",\n        "arguments": {\n            "container": H.self(),\n            "elements": [\n                {"data": {"id": "A"}},\n                {"data": {"id": "B"}},\n                {"data": {"id": "C"}},\n                {"data": {"source": "A", "target": "B"}},\n                {"data": {"source": "B", "target": "C"}},\n                {"data": {"source": "C", "target": "A"}},\n            ],\n            "style": cystyle,\n            "layout": {"name": "cose"},\n        },\n    },\n)\nprint(node)\n```\n\nThe above will:\n\n* Import the specified module.\n* Call the module\'s default export with `arguments`.\n  * Note the use of `H.self()` to refer to the target `div` in the arguments.\n\nIf you wish to use a non-default export, set the `symbol` key in the `__constructor` attribute to the name of the export you want.\n\n\n## Customize hrepr\n\n### Mixins\n\nIf you want to *really* customize hrepr, you can use mixins. They look like a bit of black magic, but they\'re simple enough:\n\n```python\n# ovld is one of the dependencies of hrepr\nfrom ovld import ovld, extend_super, has_attribute, OvldMC\nfrom hrepr import hrepr\n\nclass MyMixin(metaclass=OvldMC):\n    # Change the representation of integers\n\n    @extend_super\n    def hrepr_resources(self, cls: int):\n        # Note: in hrepr_resources, cls is the int type, not an integer\n        return self.H.style(".my-integer { color: fuchsia; }")\n\n    @extend_super\n    def hrepr(self, n: int):\n        return self.H.span["my-integer"]("The number ", str(n))\n\n    # Specially handle any object with a "quack" method\n\n    def hrepr(self, duck: has_attribute("quack")):\n        return self.H.span("🦆")\n```\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr7.png" width="600px">\n\nThe annotation for a rule can either be a type, `ovld.has_attribute`, or pretty much any function wrapped with the `ovld.meta` decorator, as long as the function operates on classes. See the documentation for [ovld](https://github.com/breuleux/ovld#other-features) for more information.\n\nAnd yes, you can define `hrepr` multiple times inside the class, as long as they have distinct annotations and you inherit from `Hrepr`. You can also define `hrepr_short` or `hrepr_resources` the same way.\n\n### Postprocessors\n\n`hrepr` can be given a postprocessor that is called on the representation of any object. You can use this to do things like highlighting specific objects:\n\n```python\nfrom hrepr import H\n\nstyle = H.style(".highlight { border: 3px solid red !important; }")\n\ndef highlight(x):\n    def postprocess(element, obj, hrepr):\n        if obj == x:\n            # Adds the "highlight" class and attaches a style\n            return element["highlight"].fill(resources=style)\n        else:\n            return element\n\n    return postprocess\n\nhrepr([1, 2, [3, 4, 2]], postprocess=highlight(2))\n```\n\n<img src="https://raw.githubusercontent.com/breuleux/hrepr/master/images/hrepr8.png" width="600px">\n\n\n### hrepr variants\n\nTo put this all together, you can create a *variant* of `hrepr`:\n\n```python\nhrepr2 = hrepr.variant(mixins=MyMixin, postprocess=highlight(2))\nhrepr2([1, 2, 3])  # Will use the mixins and postprocessor\n```\n\n\n### Configure the hrepr function itself\n\nAlternatively, you can configure the main `hrepr`:\n\n```python\nhrepr.configure(mixins=MyMixin, postprocess=highlight(2))\n```\n\nBut keep in mind that unlike the variant, the above will modify `hrepr` for everything else as well.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/breuleux/hrepr',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
