"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 * A Function can be used directly as a Resource, but this class mirrors
 * integration with other AWS services via a specific class instance.
 */
class PublishToTopic {
    constructor(topic, props) {
        this.topic = topic;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SNS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is missing in message (pass Context.taskToken somewhere in message)');
            }
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("sns", "publish", this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    actions: ['sns:Publish'],
                    resources: [this.topic.topicArn]
                })],
            parameters: {
                TopicArn: this.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? "json" : undefined,
                Subject: this.props.subject,
            }
        };
    }
}
exports.PublishToTopic = PublishToTopic;
//# sourceMappingURL=data:application/json;base64,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