"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("lambda", "invoke", this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ["lambda:InvokeFunction"],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier
            }
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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