"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*']
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'], }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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