"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SagemakerTrainTask {
    constructor(props) {
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSizeInGB: 10
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1)
        };
        // check that either algorithm name or image is defined
        if ((!props.algorithmSpecification.algorithmName) && (!props.algorithmSpecification.trainingImage)) {
            throw new Error("Must define either an algorithm name or training image URI in the algorithm specification");
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = (props.algorithmSpecification.trainingInputMode) ?
            (props.algorithmSpecification) :
            ({ ...props.algorithmSpecification, trainingInputMode: sagemaker_task_base_types_1.InputMode.FILE });
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map(config => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return Object.assign({}, config, { dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX } } });
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = (props.vpcConfig.subnets) ?
                (this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds) : this.vpc.selectSubnets().subnetIds;
        }
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error(`Principal not available yet--use the object in a Task first`);
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    bind(task) {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role = this.props.role || new iam.Role(task, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateTrainingJob: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'cloudwatch:PutMetricData',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                                'logs:CreateLogGroup',
                                'logs:DescribeLogStreams',
                                'ecr:GetAuthorizationToken',
                                ...this.props.vpcConfig
                                    ? [
                                        'ec2:CreateNetworkInterface',
                                        'ec2:CreateNetworkInterfacePermission',
                                        'ec2:DeleteNetworkInterface',
                                        'ec2:DeleteNetworkInterfacePermission',
                                        'ec2:DescribeNetworkInterfaces',
                                        'ec2:DescribeVpcs',
                                        'ec2:DescribeDhcpOptions',
                                        'ec2:DescribeSubnets',
                                        'ec2:DescribeSecurityGroups',
                                    ]
                                    : [],
                            ],
                            resources: ['*'],
                        })
                    ]
                }),
            }
        });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(task, 'TrainJobSecurityGroup', {
                vpc: this.vpc
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("sagemaker", "createTrainingJob", this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...(this.renderAlgorithmSpecification(this.algorithmSpecification)),
            ...(this.renderInputDataConfig(this.inputDataConfig)),
            ...(this.renderOutputDataConfig(this.props.outputDataConfig)),
            ...(this.renderResourceConfig(this.resourceConfig)),
            ...(this.renderStoppingCondition(this.stoppingCondition)),
            ...(this.renderHyperparameters(this.props.hyperparameters)),
            ...(this.renderTags(this.props.tags)),
            ...(this.renderVpcConfig(this.props.vpcConfig)),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage) ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {},
                ...(spec.algorithmName) ? { AlgorithmName: spec.algorithmName } : {},
                ...(spec.metricDefinitions) ?
                    { MetricDefinitions: spec.metricDefinitions
                            .map(metric => ({ Name: metric.name, Regex: metric.regex })) } : {}
            }
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map(channel => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType) ?
                            { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType } : {},
                        ...(channel.dataSource.s3DataSource.attributeNames) ?
                            { AtttributeNames: channel.dataSource.s3DataSource.attributeNames } : {},
                    }
                },
                ...(channel.compressionType) ? { CompressionType: channel.compressionType } : {},
                ...(channel.contentType) ? { ContentType: channel.contentType } : {},
                ...(channel.inputMode) ? { InputMode: channel.inputMode } : {},
                ...(channel.recordWrapperType) ? { RecordWrapperType: channel.recordWrapperType } : {},
            }))
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey) ? { KmsKeyId: config.encryptionKey.keyArn } : {},
            }
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSizeInGB,
                ...(config.volumeEncryptionKey) ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {},
            }
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds()
            }
        };
    }
    renderHyperparameters(params) {
        return (params) ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return (config) ? { VpcConfig: {
                SecurityGroupIds: core_1.Lazy.listValue({ produce: () => (this.securityGroups.map(sg => (sg.securityGroupId))) }),
                Subnets: this.subnets,
            } } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.Data.isJsonPathString(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`
                    })
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*']
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { "iam:PassedToService": "sagemaker.amazonaws.com" }
                }
            })
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ["events:PutTargets", "events:PutRule", "events:DescribeRule"],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule'
                    })]
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTrainTask = SagemakerTrainTask;
//# sourceMappingURL=data:application/json;base64,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