"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
const sagemaker_task_base_types_1 = require("./sagemaker-task-base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 *  @experimental
 */
class SagemakerTransformTask {
    constructor(props) {
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call SageMaker.`);
        }
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = (props.transformInput.transformDataSource.s3DataSource.s3DataType) ? (props.transformInput) :
            Object.assign({}, props.transformInput, { transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource,
                        s3DataType: sagemaker_task_base_types_1.S3DataType.S3_PREFIX
                    }
                }
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
    }
    bind(task) {
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(task, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')
                ]
            });
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn("sagemaker", "createTransformJob", this.integrationPattern),
            parameters: this.renderParameters(),
            policyStatements: this.makePolicyStatements(task),
        };
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error(`role not available yet--use the object in a Task first`);
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy) ? { BatchStrategy: this.props.batchStrategy } : {},
            ...(this.renderEnvironment(this.props.environment)),
            ...(this.props.maxConcurrentTransforms) ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {},
            ...(this.props.maxPayloadInMB) ? { MaxPayloadInMB: this.props.maxPayloadInMB } : {},
            ModelName: this.props.modelName,
            ...(this.renderTags(this.props.tags)),
            ...(this.renderTransformInput(this.transformInput)),
            TransformJobName: this.props.transformJobName,
            ...(this.renderTransformOutput(this.props.transformOutput)),
            ...(this.renderTransformResources(this.transformResources)),
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType) ? { CompressionType: input.compressionType } : {},
                ...(input.contentType) ? { ContentType: input.contentType } : {},
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    }
                },
                ...(input.splitType) ? { SplitType: input.splitType } : {},
            }
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey) ? { KmsKeyId: output.encryptionKey.keyArn } : {},
                ...(output.accept) ? { Accept: output.accept } : {},
                ...(output.assembleWith) ? { AssembleWith: output.assembleWith } : {},
            }
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: 'ml.' + resources.instanceType,
                ...(resources.volumeKmsKeyId) ? { VolumeKmsKeyId: resources.volumeKmsKeyId.keyArn } : {},
            }
        };
    }
    renderEnvironment(environment) {
        return (environment) ? { Environment: environment } : {};
    }
    renderTags(tags) {
        return (tags) ? { Tags: Object.keys(tags).map(key => ({ Key: key, Value: tags[key] })) } : {};
    }
    makePolicyStatements(task) {
        const stack = core_1.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*'
                    })]
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { "iam:PassedToService": "sagemaker.amazonaws.com" }
                }
            })
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ["events:PutTargets", "events:PutRule", "events:DescribeRule"],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule'
                    })]
            }));
        }
        return policyStatements;
    }
}
exports.SagemakerTransformTask = SagemakerTransformTask;
//# sourceMappingURL=data:application/json;base64,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