"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let clusterRole;
let serviceRole;
let autoScalingRole;
beforeEach(() => {
    var _a;
    // GIVEN
    stack = new cdk.Stack();
    clusterRole = new iam.Role(stack, 'ClusterRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
    });
    serviceRole = new iam.Role(stack, 'ServiceRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com')
    });
    autoScalingRole = new iam.Role(stack, 'AutoScalingRole', {
        assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com')
    });
    (_a = autoScalingRole.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        principals: [
            new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        ],
        actions: [
            'sts:AssumeRole'
        ]
    }));
});
test('Create Cluster with FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with SYNC integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with clusterConfiguration Name from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: sfn.TaskInput.fromDataAt('$.ClusterName').value,
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            'Name.$': '$.ClusterName',
            'Instances': {
                KeepJobFlowAliveWhenNoSteps: true
            },
            'VisibleToAllUsers': true,
            'JobFlowRole': {
                Ref: 'ClusterRoleD9CA7471',
            },
            'ServiceRole': {
                Ref: 'ServiceRole4288B192'
            },
            'AutoScalingRole': {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with Tags', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            tags: [{
                    key: 'Key',
                    value: 'Value'
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Tags: [{
                    Key: 'Key',
                    Value: 'Value'
                }]
        },
    });
});
test('Create Cluster with Applications', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            applications: [
                { name: 'Hive', version: '0.0' },
                { name: 'Spark', version: '0.0' }
            ],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Applications: [
                { Name: 'Hive', Version: '0.0' },
                { Name: 'Spark', Version: '0.0' }
            ]
        },
    });
});
test('Create Cluster with Bootstrap Actions', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            bootstrapActions: [{
                    name: 'Bootstrap',
                    scriptBootstrapAction: {
                        path: 's3://null',
                        args: ['Arg']
                    }
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            BootstrapActions: [{
                    Name: 'Bootstrap',
                    ScriptBootstrapAction: {
                        Path: 's3://null',
                        Args: ['Arg']
                    }
                }]
        },
    });
});
test('Create Cluster with Configurations', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            configurations: [{
                    classification: 'classification',
                    properties: {
                        Key: 'Value'
                    }
                }],
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            Configurations: [{
                    Classification: 'classification',
                    Properties: {
                        Key: 'Value'
                    }
                }]
        },
    });
});
test('Create Cluster with KerberosAttributes', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {},
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            kerberosAttributes: {
                realm: 'realm',
                adDomainJoinPassword: 'password1',
                adDomainJoinUser: 'user',
                crossRealmTrustPrincipalPassword: 'password2',
                kdcAdminPassword: 'password3'
            },
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            },
            KerberosAttributes: {
                Realm: 'realm',
                ADDomainJoinPassword: 'password1',
                ADDomainJoinUser: 'user',
                CrossRealmTrustPrincipalPassword: 'password2',
                KdcAdminPassword: 'password3'
            }
        },
    });
});
test('Create Cluster without Roles', () => {
    // WHEN
    const createClusterTask = new tasks.EmrCreateCluster({
        instances: {},
        name: 'Cluster',
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC
    });
    const task = new sfn.Task(stack, 'Task', { task: createClusterTask });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'TaskInstanceRoleB72072BF'
            },
            ServiceRole: {
                Ref: 'TaskServiceRoleBF55F61E'
            },
            AutoScalingRole: {
                Ref: 'TaskAutoScalingRoleD06F8423'
            }
        },
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
    // The stack renders the ec2.amazonaws.com Service principal id with a
    // Join to the URLSuffix
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: {
                            'Fn::Join': [
                                '',
                                [
                                    'ec2.',
                                    {
                                        Ref: 'AWS::URLSuffix'
                                    }
                                ]
                            ]
                        }
                    },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Principal: { Service: 'elasticmapreduce.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                },
                {
                    Principal: { Service: 'application-autoscaling.amazonaws.com' },
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow'
                }
            ],
        }
    });
});
test('Create Cluster with Instances configuration', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                additionalMasterSecurityGroups: ['MasterGroup'],
                additionalSlaveSecurityGroups: ['SlaveGroup'],
                ec2KeyName: 'Ec2KeyName',
                ec2SubnetId: 'Ec2SubnetId',
                ec2SubnetIds: ['Ec2SubnetId'],
                emrManagedMasterSecurityGroup: 'MasterGroup',
                emrManagedSlaveSecurityGroup: 'SlaveGroup',
                hadoopVersion: 'HadoopVersion',
                instanceCount: 1,
                masterInstanceType: 'MasterInstanceType',
                placement: {
                    availabilityZone: 'AvailabilityZone',
                    availabilityZones: ['AvailabilityZone']
                },
                serviceAccessSecurityGroup: 'ServiceAccessGroup',
                slaveInstanceType: 'SlaveInstanceType',
                terminationProtected: true
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                AdditionalMasterSecurityGroups: ['MasterGroup'],
                AdditionalSlaveSecurityGroups: ['SlaveGroup'],
                Ec2KeyName: 'Ec2KeyName',
                Ec2SubnetId: 'Ec2SubnetId',
                Ec2SubnetIds: ['Ec2SubnetId'],
                EmrManagedMasterSecurityGroup: 'MasterGroup',
                EmrManagedSlaveSecurityGroup: 'SlaveGroup',
                HadoopVersion: 'HadoopVersion',
                InstanceCount: 1,
                KeepJobFlowAliveWhenNoSteps: true,
                MasterInstanceType: 'MasterInstanceType',
                Placement: {
                    AvailabilityZone: 'AvailabilityZone',
                    AvailabilityZones: ['AvailabilityZone']
                },
                ServiceAccessSecurityGroup: 'ServiceAccessGroup',
                SlaveInstanceType: 'SlaveInstanceType',
                TerminationProtected: true
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Create Cluster with InstanceFleet', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                instanceFleets: [{
                        instanceFleetType: tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                        instanceTypeConfigs: [{
                                bidPrice: '1',
                                bidPriceAsPercentageOfOnDemandPrice: 1,
                                configurations: [{
                                        classification: 'Classification',
                                        properties: {
                                            Key: 'Value'
                                        }
                                    }],
                                ebsConfiguration: {
                                    ebsBlockDeviceConfigs: [{
                                            volumeSpecification: {
                                                iops: 1,
                                                sizeInGB: 1,
                                                volumeType: tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                            },
                                            volumesPerInstance: 1
                                        }],
                                    ebsOptimized: true
                                },
                                instanceType: 'm5.xlarge',
                                weightedCapacity: 1
                            }],
                        launchSpecifications: {
                            spotSpecification: {
                                blockDurationMinutes: 1,
                                timeoutAction: tasks.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                                timeoutDurationMinutes: 1
                            }
                        },
                        name: 'Master',
                        targetOnDemandCapacity: 1,
                        targetSpotCapacity: 1
                    }]
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceFleets: [{
                        InstanceFleetType: 'MASTER',
                        InstanceTypeConfigs: [{
                                BidPrice: '1',
                                BidPriceAsPercentageOfOnDemandPrice: 1,
                                Configurations: [{
                                        Classification: 'Classification',
                                        Properties: {
                                            Key: 'Value'
                                        }
                                    }],
                                EbsConfiguration: {
                                    EbsBlockDeviceConfigs: [{
                                            VolumeSpecification: {
                                                Iops: 1,
                                                SizeInGB: 1,
                                                VolumeType: 'standard'
                                            },
                                            VolumesPerInstance: 1
                                        }],
                                    EbsOptimized: true
                                },
                                InstanceType: 'm5.xlarge',
                                WeightedCapacity: 1
                            }],
                        LaunchSpecifications: {
                            SpotSpecification: {
                                BlockDurationMinutes: 1,
                                TimeoutAction: 'TERMINATE_CLUSTER',
                                TimeoutDurationMinutes: 1
                            }
                        },
                        Name: 'Master',
                        TargetOnDemandCapacity: 1,
                        TargetSpotCapacity: 1
                    }]
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            }
        },
    });
});
test('Create Cluster with InstanceGroup', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', { task: new tasks.EmrCreateCluster({
            instances: {
                instanceGroups: [{
                        autoScalingPolicy: {
                            constraints: {
                                maxCapacity: 2,
                                minCapacity: 1,
                            },
                            rules: [{
                                    action: {
                                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                        simpleScalingPolicyConfiguration: {
                                            adjustmentType: tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                            coolDown: 1,
                                            scalingAdjustment: 1
                                        }
                                    },
                                    description: 'Description',
                                    name: 'Name',
                                    trigger: {
                                        cloudWatchAlarmDefinition: {
                                            comparisonOperator: tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                            dimensions: [{
                                                    key: 'Key',
                                                    value: 'Value'
                                                }],
                                            evalutionPeriods: 1,
                                            metricName: 'Name',
                                            namespace: 'Namespace',
                                            period: cdk.Duration.seconds(300),
                                            statistic: tasks.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                            threshold: 1,
                                            unit: tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                        }
                                    }
                                }, {
                                    action: {
                                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                                        simpleScalingPolicyConfiguration: {
                                            adjustmentType: tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                            coolDown: 1,
                                            scalingAdjustment: 1
                                        }
                                    },
                                    description: 'Description',
                                    name: 'Name',
                                    trigger: {
                                        cloudWatchAlarmDefinition: {
                                            comparisonOperator: tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN,
                                            dimensions: [{
                                                    key: 'Key',
                                                    value: 'Value'
                                                }],
                                            evalutionPeriods: 1,
                                            metricName: 'Name',
                                            namespace: 'Namespace',
                                            period: cdk.Duration.seconds(sfn.Data.numberAt('$.CloudWatchPeriod')),
                                            statistic: tasks.EmrCreateCluster.CloudWatchAlarmStatistic.AVERAGE,
                                            threshold: 1,
                                            unit: tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                        }
                                    }
                                }]
                        },
                        bidPrice: '1',
                        configurations: [{
                                classification: 'Classification',
                                properties: {
                                    Key: 'Value'
                                }
                            }],
                        ebsConfiguration: {
                            ebsBlockDeviceConfigs: [{
                                    volumeSpecification: {
                                        iops: 1,
                                        sizeInGB: 1,
                                        volumeType: tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.STANDARD,
                                    },
                                    volumesPerInstance: 1
                                }],
                            ebsOptimized: true
                        },
                        instanceCount: 1,
                        instanceRole: tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                        instanceType: 'm5.xlarge',
                        market: tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                        name: 'Name'
                    }]
            },
            clusterRole,
            name: 'Cluster',
            serviceRole,
            autoScalingRole,
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:createCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            Name: 'Cluster',
            Instances: {
                KeepJobFlowAliveWhenNoSteps: true,
                InstanceGroups: [{
                        AutoScalingPolicy: {
                            Constraints: {
                                MaxCapacity: 2,
                                MinCapacity: 1
                            },
                            Rules: [{
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1
                                        }
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            ComparisonOperator: 'GREATER_THAN',
                                            Dimensions: [{
                                                    Key: 'Key',
                                                    Value: 'Value'
                                                }],
                                            EvaluationPeriods: 1,
                                            MetricName: 'Name',
                                            Namespace: 'Namespace',
                                            Period: 300,
                                            Statistic: 'AVERAGE',
                                            Threshold: 1,
                                            Unit: 'NONE'
                                        }
                                    }
                                }, {
                                    Action: {
                                        Market: 'ON_DEMAND',
                                        SimpleScalingPolicyConfiguration: {
                                            AdjustmentType: 'CHANGE_IN_CAPACITY',
                                            CoolDown: 1,
                                            ScalingAdjustment: 1
                                        }
                                    },
                                    Description: 'Description',
                                    Name: 'Name',
                                    Trigger: {
                                        CloudWatchAlarmDefinition: {
                                            'ComparisonOperator': 'GREATER_THAN',
                                            'Dimensions': [{
                                                    Key: 'Key',
                                                    Value: 'Value'
                                                }],
                                            'EvaluationPeriods': 1,
                                            'MetricName': 'Name',
                                            'Namespace': 'Namespace',
                                            'Period.$': '$.CloudWatchPeriod',
                                            'Statistic': 'AVERAGE',
                                            'Threshold': 1,
                                            'Unit': 'NONE'
                                        }
                                    }
                                }]
                        },
                        BidPrice: '1',
                        Configurations: [{
                                Classification: 'Classification',
                                Properties: {
                                    Key: 'Value'
                                }
                            }],
                        EbsConfiguration: {
                            EbsBlockDeviceConfigs: [{
                                    VolumeSpecification: {
                                        Iops: 1,
                                        SizeInGB: 1,
                                        VolumeType: 'standard'
                                    },
                                    VolumesPerInstance: 1
                                }],
                            EbsOptimized: true
                        },
                        InstanceCount: 1,
                        InstanceRole: 'MASTER',
                        InstanceType: 'm5.xlarge',
                        Market: 'ON_DEMAND',
                        Name: 'Name'
                    }]
            },
            VisibleToAllUsers: true,
            JobFlowRole: {
                Ref: 'ClusterRoleD9CA7471',
            },
            ServiceRole: {
                Ref: 'ServiceRole4288B192'
            },
            AutoScalingRole: {
                Ref: 'AutoScalingRole015ADA0A'
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.EmrCreateCluster({
                instances: {},
                clusterRole,
                name: 'Cluster',
                serviceRole,
                autoScalingRole,
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call CreateCluster./i);
});
//# sourceMappingURL=data:application/json;base64,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