"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Terminate cluster with static ClusterId', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrTerminateCluster({
            clusterId: 'ClusterId',
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId'
        },
    });
});
test('Terminate cluster with ClusterId from payload', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrTerminateCluster({
            clusterId: sfn.TaskInput.fromDataAt('$.ClusterId').value,
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId'
        },
    });
});
test('Terminate cluster with static ClusterId and SYNC integrationPattern', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.EmrTerminateCluster({
            clusterId: 'ClusterId',
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:terminateCluster',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId'
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.EmrTerminateCluster({
                clusterId: 'ClusterId',
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call TerminateCluster./i);
});
//# sourceMappingURL=data:application/json;base64,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