"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../lib");
const jobName = "GlueJob";
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
});
test('Invoke glue job with just job ARN', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName)
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::glue:startJobRun",
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: jobName
        },
    });
});
test('Invoke glue job with full properties', () => {
    const jobArguments = {
        key: "value"
    };
    const timeoutMinutes = 1440;
    const timeout = core_1.Duration.minutes(timeoutMinutes);
    const securityConfiguration = "securityConfiguration";
    const notifyDelayAfterMinutes = 10;
    const notifyDelayAfter = core_1.Duration.minutes(notifyDelayAfterMinutes);
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            arguments: jobArguments,
            timeout,
            securityConfiguration,
            notifyDelayAfter
        })
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::glue:startJobRun.sync",
                ],
            ],
        },
        End: true,
        Parameters: {
            JobName: jobName,
            Arguments: jobArguments,
            Timeout: timeoutMinutes,
            SecurityConfiguration: securityConfiguration,
            NotificationProperty: {
                NotifyDelayAfter: notifyDelayAfterMinutes
            }
        },
    });
});
test('permitted role actions limited to start job run if service integration pattern is FIRE_AND_FORGET', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
        })
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: "glue:StartJobRun"
                }]
        }
    });
});
test('permitted role actions include start, get, and stop job run if service integration pattern is SYNC', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunGlueJobTask(jobName, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        })
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [{
                    Action: [
                        "glue:StartJobRun",
                        "glue:GetJobRun",
                        "glue:GetJobRuns",
                        "glue:BatchStopJobRun"
                    ]
                }]
        }
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunGlueJobTask(jobName, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            })
        });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call Glue./i);
});
//# sourceMappingURL=data:application/json;base64,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