"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let role;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')
        ],
    });
});
test('create basic transform job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: "MyTransformJob",
            modelName: "MyModelName",
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                    }
                }
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
            },
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTransformJob",
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    }
                }
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                transformJobName: "MyTransformJob",
                modelName: "MyModelName",
                transformInput: {
                    transformDataSource: {
                        s3DataSource: {
                            s3Uri: 's3://inputbucket/prefix',
                        }
                    }
                },
                transformOutput: {
                    s3OutputPath: 's3://outputbucket/prefix',
                },
            }) });
    }).toThrow(/Invalid Service Integration Pattern: WAIT_FOR_TASK_TOKEN is not supported to call SageMaker./i);
});
test('create complex transform job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: "MyTransformJob",
            modelName: "MyModelName",
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            role,
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                    }
                }
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
                encryptionKey: kmsKey,
            },
            transformResources: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
                volumeKmsKeyId: kmsKey,
            },
            tags: {
                Project: 'MyProject',
            },
            batchStrategy: tasks.BatchStrategy.MULTI_RECORD,
            environment: {
                SOMEVAR: 'myvalue'
            },
            maxConcurrentTransforms: 3,
            maxPayloadInMB: 100,
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTransformJob.sync",
                ],
            ],
        },
        End: true,
        Parameters: {
            TransformJobName: 'MyTransformJob',
            ModelName: 'MyModelName',
            TransformInput: {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    }
                }
            },
            TransformOutput: {
                S3OutputPath: 's3://outputbucket/prefix',
                KmsKeyId: { "Fn::GetAtt": ["Key961B73FD", "Arn"] },
            },
            TransformResources: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeKmsKeyId: { "Fn::GetAtt": ["Key961B73FD", "Arn"] },
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' }
            ],
            MaxConcurrentTransforms: 3,
            MaxPayloadInMB: 100,
            Environment: {
                SOMEVAR: 'myvalue'
            },
            BatchStrategy: 'MultiRecord'
        },
    });
});
test('pass param to transform job', () => {
    // WHEN
    const task = new sfn.Task(stack, 'TransformTask', { task: new tasks.SagemakerTransformTask({
            transformJobName: sfn.Data.stringAt('$.TransformJobName'),
            modelName: sfn.Data.stringAt('$.ModelName'),
            role,
            transformInput: {
                transformDataSource: {
                    s3DataSource: {
                        s3Uri: 's3://inputbucket/prefix',
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                    }
                }
            },
            transformOutput: {
                s3OutputPath: 's3://outputbucket/prefix',
            },
            transformResources: {
                instanceCount: 1,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            }
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sagemaker:createTransformJob",
                ],
            ],
        },
        End: true,
        Parameters: {
            'TransformJobName.$': '$.TransformJobName',
            'ModelName.$': '$.ModelName',
            'TransformInput': {
                DataSource: {
                    S3DataSource: {
                        S3Uri: 's3://inputbucket/prefix',
                        S3DataType: 'S3Prefix',
                    }
                }
            },
            'TransformOutput': {
                S3OutputPath: 's3://outputbucket/prefix',
            },
            'TransformResources': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
            }
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2FnZW1ha2VyLXRyYW5zZm9ybS1qb2IudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNhZ2VtYWtlci10cmFuc2Zvcm0tam9iLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsa0RBQWtEO0FBQ2xELHFDQUFxQztBQUNyQyxnQ0FBZ0M7QUFFaEMsSUFBSSxLQUFnQixDQUFDO0FBQ3JCLElBQUksSUFBYyxDQUFDO0FBRW5CLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDWixRQUFRO0lBQ1IsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQ3hCLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUMvQixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7UUFDOUQsZUFBZSxFQUFFO1lBQ2IsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQywyQkFBMkIsQ0FBQztTQUMxRTtLQUNKLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUNwQyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDdkYsZ0JBQWdCLEVBQUUsZ0JBQWdCO1lBQ2xDLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLGNBQWMsRUFBRTtnQkFDWixtQkFBbUIsRUFBRTtvQkFDakIsWUFBWSxFQUFFO3dCQUNWLEtBQUssRUFBRSx5QkFBeUI7cUJBQ25DO2lCQUNKO2FBQ0o7WUFDRCxlQUFlLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLDBCQUEwQjthQUMzQztTQUNKLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELHdDQUF3QztpQkFDekM7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsU0FBUyxFQUFFLGFBQWE7WUFDeEIsY0FBYyxFQUFFO2dCQUNaLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLFVBQVU7cUJBQ3pCO2lCQUNKO2FBQ0o7WUFDRCxlQUFlLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLDBCQUEwQjthQUMzQztZQUNELGtCQUFrQixFQUFFO2dCQUNoQixhQUFhLEVBQUUsQ0FBQztnQkFDaEIsWUFBWSxFQUFFLGNBQWM7YUFDL0I7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtFQUErRSxFQUFFLEdBQUcsRUFBRTtJQUN2RixNQUFNLENBQUMsR0FBRyxFQUFFO1FBQ1IsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7Z0JBQzFFLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxtQkFBbUI7Z0JBQ3JFLGdCQUFnQixFQUFFLGdCQUFnQjtnQkFDbEMsU0FBUyxFQUFFLGFBQWE7Z0JBQ3hCLGNBQWMsRUFBRTtvQkFDWixtQkFBbUIsRUFBRTt3QkFDakIsWUFBWSxFQUFFOzRCQUNWLEtBQUssRUFBRSx5QkFBeUI7eUJBQ25DO3FCQUNKO2lCQUNKO2dCQUNELGVBQWUsRUFBRTtvQkFDYixZQUFZLEVBQUUsMEJBQTBCO2lCQUMzQzthQUNKLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDVixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsK0ZBQStGLENBQUMsQ0FBQztBQUM5RyxDQUFDLENBQUMsQ0FBQztBQUVMLElBQUksQ0FBQyw4QkFBOEIsRUFBRSxHQUFHLEVBQUU7SUFDdEMsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDekMsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDdkYsZ0JBQWdCLEVBQUUsZ0JBQWdCO1lBQ2xDLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxJQUFJO1lBQ3RELElBQUk7WUFDSixjQUFjLEVBQUU7Z0JBQ1osbUJBQW1CLEVBQUU7b0JBQ2pCLFlBQVksRUFBRTt3QkFDVixLQUFLLEVBQUUseUJBQXlCO3dCQUNoQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxTQUFTO3FCQUN6QztpQkFDSjthQUNKO1lBQ0QsZUFBZSxFQUFFO2dCQUNiLFlBQVksRUFBRSwwQkFBMEI7Z0JBQ3hDLGFBQWEsRUFBRSxNQUFNO2FBQ3hCO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2hCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7Z0JBQ2pGLGNBQWMsRUFBRSxNQUFNO2FBQ3pCO1lBQ0QsSUFBSSxFQUFFO2dCQUNGLE9BQU8sRUFBRSxXQUFXO2FBQ3ZCO1lBQ0QsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsWUFBWTtZQUMvQyxXQUFXLEVBQUU7Z0JBQ1QsT0FBTyxFQUFFLFNBQVM7YUFDckI7WUFDRCx1QkFBdUIsRUFBRSxDQUFDO1lBQzFCLGNBQWMsRUFBRSxHQUFHO1NBQ3RCLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFTixPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDZDQUE2QztpQkFDOUM7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsU0FBUyxFQUFFLGFBQWE7WUFDeEIsY0FBYyxFQUFFO2dCQUNaLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLFVBQVU7cUJBQ3pCO2lCQUNKO2FBQ0o7WUFDRCxlQUFlLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLDBCQUEwQjtnQkFDeEMsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsYUFBYSxFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQ3ZEO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2hCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsZUFBZTtnQkFDN0IsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsYUFBYSxFQUFFLEtBQUssQ0FBRSxFQUFFO2FBQzdEO1lBQ0QsSUFBSSxFQUFFO2dCQUNGLEVBQUUsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFFO2FBQ3pDO1lBQ0QsdUJBQXVCLEVBQUUsQ0FBQztZQUMxQixjQUFjLEVBQUUsR0FBRztZQUNuQixXQUFXLEVBQUU7Z0JBQ1QsT0FBTyxFQUFFLFNBQVM7YUFDckI7WUFDRCxhQUFhLEVBQUUsYUFBYTtTQUM3QjtLQUNGLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZCQUE2QixFQUFFLEdBQUcsRUFBRTtJQUNyQyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsc0JBQXNCLENBQUM7WUFDdkYsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUM7WUFDekQsU0FBUyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztZQUMzQyxJQUFJO1lBQ0osY0FBYyxFQUFFO2dCQUNaLG1CQUFtQixFQUFFO29CQUNqQixZQUFZLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUztxQkFDekM7aUJBQ0o7YUFDSjtZQUNELGVBQWUsRUFBRTtnQkFDYixZQUFZLEVBQUUsMEJBQTBCO2FBQzNDO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2hCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7YUFDcEY7U0FDSixDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRU4sT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQzlDLElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ04sVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCx3Q0FBd0M7aUJBQ3pDO2FBQ0Y7U0FDSjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1Ysb0JBQW9CLEVBQUUsb0JBQW9CO1lBQzFDLGFBQWEsRUFBRSxhQUFhO1lBQzVCLGdCQUFnQixFQUFFO2dCQUNkLFVBQVUsRUFBRTtvQkFDUixZQUFZLEVBQUU7d0JBQ1YsS0FBSyxFQUFFLHlCQUF5Qjt3QkFDaEMsVUFBVSxFQUFFLFVBQVU7cUJBQ3pCO2lCQUNKO2FBQ0o7WUFDRCxpQkFBaUIsRUFBRTtnQkFDZixZQUFZLEVBQUUsMEJBQTBCO2FBQzNDO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ2xCLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsZUFBZTthQUNoQztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ1QsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyB0YXNrcyBmcm9tICcuLi9saWInO1xuXG5sZXQgc3RhY2s6IGNkay5TdGFjaztcbmxldCByb2xlOiBpYW0uUm9sZTtcblxuYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3NhZ2VtYWtlci5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJylcbiAgICAgICAgXSxcbiAgICB9KTtcbn0pO1xuXG50ZXN0KCdjcmVhdGUgYmFzaWMgdHJhbnNmb3JtIGpvYicsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFzayA9IG5ldyBzZm4uVGFzayhzdGFjaywgJ1RyYW5zZm9ybVRhc2snLCB7IHRhc2s6IG5ldyB0YXNrcy5TYWdlbWFrZXJUcmFuc2Zvcm1UYXNrKHtcbiAgICAgICAgdHJhbnNmb3JtSm9iTmFtZTogXCJNeVRyYW5zZm9ybUpvYlwiLFxuICAgICAgICBtb2RlbE5hbWU6IFwiTXlNb2RlbE5hbWVcIixcbiAgICAgICAgdHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgICAgIHRyYW5zZm9ybURhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgczNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgICAgICAgIHMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgIH0sXG4gICAgfSkgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgICBUeXBlOiAnVGFzaycsXG4gICAgICBSZXNvdXJjZToge1xuICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICBcIlwiLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpQYXJ0aXRpb25cIixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIjpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhbnNmb3JtSm9iXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIFRyYW5zZm9ybUpvYk5hbWU6ICdNeVRyYW5zZm9ybUpvYicsXG4gICAgICAgIE1vZGVsTmFtZTogJ015TW9kZWxOYW1lJyxcbiAgICAgICAgVHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgUzNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICAgICAgICAgIFMzRGF0YVR5cGU6ICdTM1ByZWZpeCcsXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBUcmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgICAgICAgIFMzT3V0cHV0UGF0aDogJ3MzOi8vb3V0cHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgIH0sXG4gICAgICAgIFRyYW5zZm9ybVJlc291cmNlczoge1xuICAgICAgICAgICAgSW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgICAgIEluc3RhbmNlVHlwZTogJ21sLm00LnhsYXJnZScsXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgfSk7XG59KTtcblxudGVzdCgnVGFzayB0aHJvd3MgaWYgV0FJVF9GT1JfVEFTS19UT0tFTiBpcyBzdXBwbGllZCBhcyBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4nLCAoKSA9PiB7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgICAgbmV3IHNmbi5UYXNrKHN0YWNrLCAnVHJhbnNmb3JtVGFzaycsIHsgdGFzazogbmV3IHRhc2tzLlNhZ2VtYWtlclRyYW5zZm9ybVRhc2soe1xuICAgICAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgICAgICAgICAgdHJhbnNmb3JtSm9iTmFtZTogXCJNeVRyYW5zZm9ybUpvYlwiLFxuICAgICAgICAgICAgbW9kZWxOYW1lOiBcIk15TW9kZWxOYW1lXCIsXG4gICAgICAgICAgICB0cmFuc2Zvcm1JbnB1dDoge1xuICAgICAgICAgICAgICAgIHRyYW5zZm9ybURhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgczNEYXRhU291cmNlOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICBzM1VyaTogJ3MzOi8vaW5wdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB0cmFuc2Zvcm1PdXRwdXQ6IHtcbiAgICAgICAgICAgICAgICBzM091dHB1dFBhdGg6ICdzMzovL291dHB1dGJ1Y2tldC9wcmVmaXgnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSkgfSk7XG4gICAgfSkudG9UaHJvdygvSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46IFdBSVRfRk9SX1RBU0tfVE9LRU4gaXMgbm90IHN1cHBvcnRlZCB0byBjYWxsIFNhZ2VNYWtlci4vaSk7XG4gIH0pO1xuXG50ZXN0KCdjcmVhdGUgY29tcGxleCB0cmFuc2Zvcm0gam9iJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBrbXNLZXkgPSBuZXcga21zLktleShzdGFjaywgJ0tleScpO1xuICAgIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUcmFuc2Zvcm1UYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuU2FnZW1ha2VyVHJhbnNmb3JtVGFzayh7XG4gICAgICAgIHRyYW5zZm9ybUpvYk5hbWU6IFwiTXlUcmFuc2Zvcm1Kb2JcIixcbiAgICAgICAgbW9kZWxOYW1lOiBcIk15TW9kZWxOYW1lXCIsXG4gICAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLlNlcnZpY2VJbnRlZ3JhdGlvblBhdHRlcm4uU1lOQyxcbiAgICAgICAgcm9sZSxcbiAgICAgICAgdHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgICAgIHRyYW5zZm9ybURhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgczNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICAgICAgICAgIHMzRGF0YVR5cGU6IHRhc2tzLlMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgdHJhbnNmb3JtT3V0cHV0OiB7XG4gICAgICAgICAgICBzM091dHB1dFBhdGg6ICdzMzovL291dHB1dGJ1Y2tldC9wcmVmaXgnLFxuICAgICAgICAgICAgZW5jcnlwdGlvbktleToga21zS2V5LFxuICAgICAgICB9LFxuICAgICAgICB0cmFuc2Zvcm1SZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuUDMsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFMiksXG4gICAgICAgICAgICB2b2x1bWVLbXNLZXlJZDoga21zS2V5LFxuICAgICAgICB9LFxuICAgICAgICB0YWdzOiB7XG4gICAgICAgICAgICBQcm9qZWN0OiAnTXlQcm9qZWN0JyxcbiAgICAgICAgfSxcbiAgICAgICAgYmF0Y2hTdHJhdGVneTogdGFza3MuQmF0Y2hTdHJhdGVneS5NVUxUSV9SRUNPUkQsXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgICBTT01FVkFSOiAnbXl2YWx1ZSdcbiAgICAgICAgfSxcbiAgICAgICAgbWF4Q29uY3VycmVudFRyYW5zZm9ybXM6IDMsXG4gICAgICAgIG1heFBheWxvYWRJbk1COiAxMDAsXG4gICAgfSkgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgICBUeXBlOiAnVGFzaycsXG4gICAgICBSZXNvdXJjZToge1xuICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICBcIlwiLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpQYXJ0aXRpb25cIixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIjpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhbnNmb3JtSm9iLnN5bmNcIixcbiAgICAgICAgICBdLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICAgIEVuZDogdHJ1ZSxcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgVHJhbnNmb3JtSm9iTmFtZTogJ015VHJhbnNmb3JtSm9iJyxcbiAgICAgICAgTW9kZWxOYW1lOiAnTXlNb2RlbE5hbWUnLFxuICAgICAgICBUcmFuc2Zvcm1JbnB1dDoge1xuICAgICAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICAgIFMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICAgICAgICBTM1VyaTogJ3MzOi8vaW5wdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICAgICAgICAgICAgUzNEYXRhVHlwZTogJ1MzUHJlZml4JyxcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFRyYW5zZm9ybU91dHB1dDoge1xuICAgICAgICAgICAgUzNPdXRwdXRQYXRoOiAnczM6Ly9vdXRwdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICAgIEttc0tleUlkOiB7IFwiRm46OkdldEF0dFwiOiBbIFwiS2V5OTYxQjczRkRcIiwgXCJBcm5cIiBdIH0sXG4gICAgICAgIH0sXG4gICAgICAgIFRyYW5zZm9ybVJlc291cmNlczoge1xuICAgICAgICAgICAgSW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgICAgIEluc3RhbmNlVHlwZTogJ21sLnAzLjJ4bGFyZ2UnLFxuICAgICAgICAgICAgVm9sdW1lS21zS2V5SWQ6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJLZXk5NjFCNzNGRFwiLCBcIkFyblwiIF0gfSxcbiAgICAgICAgfSxcbiAgICAgICAgVGFnczogW1xuICAgICAgICAgICAgeyBLZXk6ICdQcm9qZWN0JywgVmFsdWU6ICdNeVByb2plY3QnIH1cbiAgICAgICAgXSxcbiAgICAgICAgTWF4Q29uY3VycmVudFRyYW5zZm9ybXM6IDMsXG4gICAgICAgIE1heFBheWxvYWRJbk1COiAxMDAsXG4gICAgICAgIEVudmlyb25tZW50OiB7XG4gICAgICAgICAgICBTT01FVkFSOiAnbXl2YWx1ZSdcbiAgICAgICAgfSxcbiAgICAgICAgQmF0Y2hTdHJhdGVneTogJ011bHRpUmVjb3JkJ1xuICAgICAgfSxcbiAgICB9KTtcbn0pO1xuXG50ZXN0KCdwYXNzIHBhcmFtIHRvIHRyYW5zZm9ybSBqb2InLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUcmFuc2Zvcm1UYXNrJywgeyB0YXNrOiBuZXcgdGFza3MuU2FnZW1ha2VyVHJhbnNmb3JtVGFzayh7XG4gICAgICAgIHRyYW5zZm9ybUpvYk5hbWU6IHNmbi5EYXRhLnN0cmluZ0F0KCckLlRyYW5zZm9ybUpvYk5hbWUnKSxcbiAgICAgICAgbW9kZWxOYW1lOiBzZm4uRGF0YS5zdHJpbmdBdCgnJC5Nb2RlbE5hbWUnKSxcbiAgICAgICAgcm9sZSxcbiAgICAgICAgdHJhbnNmb3JtSW5wdXQ6IHtcbiAgICAgICAgICAgIHRyYW5zZm9ybURhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgICAgICAgczNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICAgICAgICAgIHMzRGF0YVR5cGU6IHRhc2tzLlMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgdHJhbnNmb3JtT3V0cHV0OiB7XG4gICAgICAgICAgICBzM091dHB1dFBhdGg6ICdzMzovL291dHB1dGJ1Y2tldC9wcmVmaXgnLFxuICAgICAgICB9LFxuICAgICAgICB0cmFuc2Zvcm1SZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuUDMsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFMiksXG4gICAgICAgIH1cbiAgICB9KSB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICAgICAgVHlwZTogJ1Rhc2snLFxuICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICBcImFybjpcIixcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpQYXJ0aXRpb25cIixcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIFwiOnN0YXRlczo6OnNhZ2VtYWtlcjpjcmVhdGVUcmFuc2Zvcm1Kb2JcIixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIEVuZDogdHJ1ZSxcbiAgICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICAgICdUcmFuc2Zvcm1Kb2JOYW1lLiQnOiAnJC5UcmFuc2Zvcm1Kb2JOYW1lJyxcbiAgICAgICAgICAnTW9kZWxOYW1lLiQnOiAnJC5Nb2RlbE5hbWUnLFxuICAgICAgICAgICdUcmFuc2Zvcm1JbnB1dCc6IHtcbiAgICAgICAgICAgICAgRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICAgICAgUzNEYXRhU291cmNlOiB7XG4gICAgICAgICAgICAgICAgICAgICAgUzNVcmk6ICdzMzovL2lucHV0YnVja2V0L3ByZWZpeCcsXG4gICAgICAgICAgICAgICAgICAgICAgUzNEYXRhVHlwZTogJ1MzUHJlZml4JyxcbiAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgJ1RyYW5zZm9ybU91dHB1dCc6IHtcbiAgICAgICAgICAgICAgUzNPdXRwdXRQYXRoOiAnczM6Ly9vdXRwdXRidWNrZXQvcHJlZml4JyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICdUcmFuc2Zvcm1SZXNvdXJjZXMnOiB7XG4gICAgICAgICAgICAgIEluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgICAgICAgIEluc3RhbmNlVHlwZTogJ21sLnAzLjJ4bGFyZ2UnLFxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgIH0pO1xufSk7XG4iXX0=