"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sqs = require("@aws-cdk/aws-sqs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let queue;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    queue = new sqs.Queue(stack, 'Queue');
});
test('Send message to queue', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', { task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromText('Send this message'),
            messageDeduplicationId: sfn.Data.stringAt('$.deduping'),
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sqs:sendMessage",
                ],
            ],
        },
        End: true,
        Parameters: {
            'QueueUrl': { Ref: 'Queue4A7E3555' },
            'MessageBody': 'Send this message',
            'MessageDeduplicationId.$': '$.deduping'
        },
    });
});
test('Send message to SQS queue with task token', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', { task: new tasks.SendToQueue(queue, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            messageBody: sfn.TaskInput.fromObject({
                Input: 'Send this message',
                Token: sfn.Context.taskToken
            })
        }) });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sqs:sendMessage.waitForTaskToken",
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                'Input': 'Send this message',
                'Token.$': '$$.Task.Token'
            }
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in messageBody', () => {
    expect(() => {
        // WHEN
        new sfn.Task(stack, 'Send', { task: new tasks.SendToQueue(queue, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
                messageBody: sfn.TaskInput.fromText('Send this message')
            }) });
        // THEN
    }).toThrow(/Task Token is missing in messageBody/i);
});
test('Message body can come from state', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromDataAt('$.theMessage')
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sqs:sendMessage",
                ],
            ],
        },
        End: true,
        Parameters: {
            'QueueUrl': { Ref: 'Queue4A7E3555' },
            'MessageBody.$': '$.theMessage',
        },
    });
});
test('Message body can be an object', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromObject({
                literal: 'literal',
                SomeInput: sfn.Data.stringAt('$.theMessage')
            })
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sqs:sendMessage",
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                'literal': 'literal',
                'SomeInput.$': '$.theMessage',
            }
        },
    });
});
test('Message body object can contain references', () => {
    // WHEN
    const task = new sfn.Task(stack, 'Send', {
        task: new tasks.SendToQueue(queue, {
            messageBody: sfn.TaskInput.fromObject({
                queueArn: queue.queueArn
            })
        })
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition",
                    },
                    ":states:::sqs:sendMessage",
                ],
            ],
        },
        End: true,
        Parameters: {
            QueueUrl: { Ref: 'Queue4A7E3555' },
            MessageBody: {
                queueArn: { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] }
            }
        },
    });
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Send', { task: new tasks.SendToQueue(queue, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
                messageBody: sfn.TaskInput.fromText('Send this message')
            }) });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call SQS./i);
});
//# sourceMappingURL=data:application/json;base64,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